import React from 'react';
import { PieChart, Pie, Cell, BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from 'recharts';

interface VerificationDocument {
  id: number;
  doc_type: string;
  verification_status: string;
  features?: {
    feature_4_authenticity_scoring?: {
      authenticity_score: number;
    };
    feature_5_dispute_prediction?: {
      dispute_probability: number;
    };
  };
}

interface VerificationChartsProps {
  documents: VerificationDocument[];
}

export const VerificationCharts: React.FC<VerificationChartsProps> = ({ documents }) => {
  // Status distribution data
  const statusData = [
    {
      name: 'Verified',
      value: documents.filter(d => d.verification_status === 'completed').length,
      color: '#10b981'
    },
    {
      name: 'Pending',
      value: documents.filter(d => d.verification_status === 'pending').length,
      color: '#f59e0b'
    },
    {
      name: 'Rejected',
      value: documents.filter(d => d.verification_status === 'failed').length,
      color: '#ef4444'
    }
  ].filter(item => item.value > 0);

  // Document type distribution
  const typeData = documents.reduce((acc, doc) => {
    const type = doc.doc_type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
    const existing = acc.find(item => item.name === type);
    if (existing) {
      existing.value += 1;
    } else {
      acc.push({ name: type, value: 1 });
    }
    return acc;
  }, [] as { name: string; value: number }[]);

  // Authenticity scores
  const authenticityData = documents
    .filter(d => d.features?.feature_4_authenticity_scoring?.authenticity_score)
    .map(d => ({
      name: d.doc_type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase()),
      score: d.features?.feature_4_authenticity_scoring?.authenticity_score || 0
    }));

  if (documents.length === 0) {
    return null;
  }

  return (
    <div className="bg-white rounded-xl border border-gray-200 shadow-sm p-6">
      <h3 className="text-xl font-bold text-gray-900 mb-6">Verification Analytics</h3>
      
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Status Distribution */}
        {statusData.length > 0 && (
          <div>
            <h4 className="text-lg font-semibold text-gray-800 mb-4">Verification Status</h4>
            <ResponsiveContainer width="100%" height={250}>
              <PieChart>
                <Pie
                  data={statusData}
                  cx="50%"
                  cy="50%"
                  outerRadius={80}
                  dataKey="value"
                  label={({ name, value }) => `${name}: ${value}`}
                >
                  {statusData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={entry.color} />
                  ))}
                </Pie>
                <Tooltip />
              </PieChart>
            </ResponsiveContainer>
          </div>
        )}

        {/* Document Types */}
        {typeData.length > 0 && (
          <div>
            <h4 className="text-lg font-semibold text-gray-800 mb-4">Document Types</h4>
            <ResponsiveContainer width="100%" height={250}>
              <BarChart data={typeData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis 
                  dataKey="name" 
                  angle={-45}
                  textAnchor="end"
                  height={80}
                  fontSize={12}
                />
                <YAxis />
                <Tooltip />
                <Bar dataKey="value" fill="#3b82f6" />
              </BarChart>
            </ResponsiveContainer>
          </div>
        )}

        {/* Authenticity Scores */}
        {authenticityData.length > 0 && (
          <div className="lg:col-span-2">
            <h4 className="text-lg font-semibold text-gray-800 mb-4">Authenticity Scores</h4>
            <ResponsiveContainer width="100%" height={250}>
              <BarChart data={authenticityData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis 
                  dataKey="name" 
                  angle={-45}
                  textAnchor="end"
                  height={80}
                  fontSize={12}
                />
                <YAxis domain={[0, 100]} />
                <Tooltip formatter={(value) => [`${value}/100`, 'Authenticity Score']} />
                <Bar 
                  dataKey="score" 
                  fill="#10b981"
                  name="Authenticity Score"
                />
              </BarChart>
            </ResponsiveContainer>
          </div>
        )}
      </div>
    </div>
  );
};