import React, { useState, useEffect } from 'react';
import { addToWishlist, removeFromWishlist, isInWishlist } from '../utils/wishlistApi';

interface WishlistButtonProps {
  propertyId: string;
  variant?: 'card' | 'detail';
  className?: string;
}

export const WishlistButton: React.FC<WishlistButtonProps> = ({ 
  propertyId, 
  variant = 'card',
  className = '' 
}) => {
  const [isSaved, setIsSaved] = useState(false);
  const [isLoading, setIsLoading] = useState(false);

  useEffect(() => {
    checkWishlistStatus();
  }, [propertyId]);

  const checkWishlistStatus = async () => {
    try {
      const saved = await isInWishlist(propertyId);
      setIsSaved(saved);
    } catch (error) {
      console.error('Error checking wishlist status:', error);
    }
  };

  const handleToggleWishlist = async (e: React.MouseEvent) => {
    console.log('🔴 WISHLIST BUTTON CLICKED - HANDLER TRIGGERED');
    e.preventDefault();
    e.stopPropagation();
    
    console.log('Wishlist button clicked:', { propertyId, isSaved });
    
    const wasNotSaved = !isSaved;
    
    setIsLoading(true);
    try {
      if (isSaved) {
        console.log('Removing from wishlist...');
        await removeFromWishlist(propertyId);
        setIsSaved(false);
        console.log('Removed from wishlist successfully');
      } else {
        console.log('Adding to wishlist...');
        await addToWishlist(propertyId);
        console.log('Added to wishlist successfully');
        setIsSaved(true);
        
        // Track immediately after successful add
        console.log('🔍 Starting wishlist tracking for property:', propertyId);
        let propertyTitle = `Property #${propertyId}`;
        try {
          const response = await fetch(`http://localhost:8090/api/properties/${propertyId}`);
          if (response.ok) {
            const property = await response.json();
            propertyTitle = property.title || propertyTitle;
            console.log('✅ Fetched property title:', propertyTitle);
          }
        } catch (err) {
          console.warn('Could not fetch property title:', err);
        }
        
        try {
          const { behaviorTracker } = await import('../utils/behaviorTracker');
          console.log('📊 Calling trackWishlistAdd with:', { propertyId: parseInt(propertyId), title: propertyTitle });
          await behaviorTracker.trackWishlistAdd(parseInt(propertyId), { title: propertyTitle });
          console.log('✅ Wishlist tracking completed successfully');
        } catch (trackError) {
          console.error('❌ Wishlist tracking failed with error:', trackError);
        }
      }
    } catch (error) {
      console.error('Error toggling wishlist:', error);
      alert('Error saving property. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  if (variant === 'card') {
    return (
      <button
        onClick={handleToggleWishlist}
        disabled={isLoading}
        className={`absolute top-3 right-3 p-2 rounded-full bg-white/80 backdrop-blur-sm hover:bg-white transition-all duration-200 shadow-md hover:shadow-lg ${className}`}
      >
        <svg 
          className={`w-5 h-5 transition-colors ${
            isSaved ? 'text-red-500' : 'text-gray-600 hover:text-red-500'
          }`}
          fill={isSaved ? 'currentColor' : 'none'}
          stroke={isSaved ? 'none' : 'currentColor'}
          viewBox="0 0 24 24"
        >
          <path 
            strokeLinecap="round" 
            strokeLinejoin="round" 
            strokeWidth={isSaved ? 0 : 2} 
            d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" 
          />
        </svg>
      </button>
    );
  }

  return (
    <button
      onClick={handleToggleWishlist}
      disabled={isLoading}
      className={`flex items-center space-x-2 px-4 py-2 border rounded-lg transition-all duration-200 ${
        isSaved 
          ? 'bg-red-50 border-red-200 text-red-700 hover:bg-red-100' 
          : 'bg-white border-gray-300 text-gray-700 hover:bg-gray-50'
      } ${className}`}
    >
      <svg 
        className={`w-5 h-5 ${isSaved ? 'text-red-500' : 'text-gray-500'}`}
        fill={isSaved ? 'currentColor' : 'none'}
        stroke={isSaved ? 'none' : 'currentColor'}
        viewBox="0 0 24 24"
      >
        <path 
          strokeLinecap="round" 
          strokeLinejoin="round" 
          strokeWidth={isSaved ? 0 : 2} 
          d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" 
        />
      </svg>
      <span className="font-medium">
        {isSaved ? 'Saved' : 'Save Property'}
      </span>
    </button>
  );
};