import React, { useState } from 'react';
import { useWishlistProperty } from '../hooks/usePropertyState';
import { toast } from 'sonner';

interface WishlistHeartButtonProps {
  propertyId: number;
}

export const WishlistHeartButton: React.FC<WishlistHeartButtonProps> = ({ propertyId }) => {
  const [loading, setLoading] = useState(false);
  const { isWishlisted, toggleWishlist } = useWishlistProperty(propertyId);

  const handleWishlistToggle = async (e: React.MouseEvent) => {
    e.stopPropagation();
    setLoading(true);
    
    try {
      const newState = await toggleWishlist();
      if (newState) {
        toast.success('Added to Wishlist', {
          description: 'Property added to your favorites collection',
          duration: 3000,
        });
        
        // Track wishlist add event
        try {
          let propertyTitle = `Property #${propertyId}`;
          const response = await fetch(`http://localhost:8090/api/properties/${propertyId}`);
          if (response.ok) {
            const property = await response.json();
            propertyTitle = property.title || propertyTitle;
          }
          
          const { behaviorTracker } = await import('../utils/behaviorTracker');
          await behaviorTracker.trackWishlistAdd(propertyId, { title: propertyTitle });
        } catch (trackError) {
          console.error('Wishlist tracking failed:', trackError);
        }
      } else {
        toast.success('Removed from Wishlist', {
          description: 'Property removed from your favorites',
          duration: 3000,
        });
      }
    } catch (error) {
      console.error('Error toggling wishlist:', error);
      if (error instanceof Error && error.message.includes('login')) {
        toast.error('Authentication Required', {
          description: 'Please log in to add properties to your wishlist',
          duration: 4000,
        });
      } else {
        toast.error('Action Failed', {
          description: 'Unable to update wishlist. Please try again.',
          duration: 4000,
        });
      }
    } finally {
      setLoading(false);
    }
  };

  return (
     <button
      onClick={handleWishlistToggle}
      disabled={loading}
      className="p-2 rounded-full shadow-lg hover:scale-110 transition-all duration-300 disabled:opacity-50 border border-white/20 hover:shadow-xl"
      style={{ backgroundColor: 'rgb(238, 239, 232)' }}
      title={isWishlisted ? "Remove from Wishlist" : "Add to Wishlist"}
    >
      {isWishlisted ? (
        <svg
          xmlns="http://www.w3.org/2000/svg"
          className="h-5 w-5"
          fill="#ef4444"
          viewBox="0 0 24 24"
        >
          <path d="M12 21.35l-1.45-1.32C5.4 15.36 2 12.28 2 8.5 2 5.42 4.42 3 7.5 3c1.74 0 3.41.81 4.5 2.09C13.09 3.81 14.76 3 16.5 3 19.58 3 22 5.42 22 8.5c0 3.78-3.4 6.86-8.55 11.54L12 21.35z" />
        </svg>
      ) : (
        <svg
          xmlns="http://www.w3.org/2000/svg"
          className="h-5 w-5"
          fill="none"
          viewBox="0 0 24 24"
          stroke="#4b5563"
          strokeWidth={2}
        >
          <path
            strokeLinecap="round"
            strokeLinejoin="round"
            d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"
          />
        </svg>
      )}
    </button>
  );
};