import React, { useState } from 'react';
import { 
  Eye, 
  MessageCircle, 
  Search, 
  Heart, 
  MapPin,
  Trash2,
  ExternalLink,
  Bookmark,
  Clock
} from 'lucide-react';

interface UserStats {
  properties_viewed: number;
  inquiries_sent: number;
  searches_performed: number;
  properties_wishlisted: number;
}

interface SavedProperty {
  id: string;
  property_id: number;
  saved_at: string;
  property?: {
    title: string;
    location: string;
    price: number;
    image_url?: string;
  };
}

const YourActivity: React.FC = () => {
  const [activeTab, setActiveTab] = useState('viewed');
  const [loading] = useState(false);
  const [error] = useState('');
  const [pendingCount] = useState(2);

  const stats: UserStats = {
    properties_viewed: 0,
    inquiries_sent: 0,
    searches_performed: 0,
    properties_wishlisted: 0
  };

  const removeSavedProperty = async (propertyId: number) => {
    console.log('Remove property:', propertyId);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Page Header */}
      <div className="bg-white border-b border-gray-200">
        <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <h1 className="text-3xl font-bold text-gray-900">Your Activity</h1>
          <p className="text-gray-600 mt-2">Track your engagement and manage your saved properties</p>
        </div>
      </div>

      <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {error && (
          <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
            {error}
          </div>
        )}



        {/* Tabs */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden mt-4">
          <div className="border-b border-gray-200">
            <nav className="flex">
              {[
                { id: 'viewed', label: 'Properties Viewed', icon: Eye },
                { id: 'inquiries', label: 'Inquiries Sent', icon: MessageCircle },
                { id: 'searches', label: 'Searches Performed', icon: Search },
                { id: 'wishlisted', label: 'Properties Wishlisted', icon: Heart },
                { id: 'saved', label: 'Saved Properties', icon: Bookmark }
              ].map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`flex items-center gap-2 px-6 py-4 text-sm font-medium border-b-2 transition-colors ${
                    activeTab === tab.id
                      ? 'border-blue-600 text-blue-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700'
                  }`}
                >
                  <tab.icon className="w-4 h-4" />
                  {tab.label}
                  {tab.count && tab.count > 0 && (
                    <span className="ml-1 bg-red-500 text-white text-xs rounded-full px-2 py-0.5 min-w-[20px] h-5 flex items-center justify-center">
                      {tab.count}
                    </span>
                  )}
                </button>
              ))}
            </nav>
          </div>

          <div className="p-6">
            {activeTab === 'viewed' && (
              <div className="text-center py-16">
                <Eye className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No Properties Viewed Yet</h3>
                <p className="text-gray-500 mb-6">Start exploring properties to see your viewing history here.</p>
                <a href="/properties" className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
                  Browse Properties
                </a>
              </div>
            )}

            {activeTab === 'inquiries' && (
              <div className="text-center py-16">
                <MessageCircle className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No Inquiries Sent</h3>
                <p className="text-gray-500 mb-6">When you contact property owners, your inquiries will appear here.</p>
                <a href="/properties" className="inline-flex items-center px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors">
                  Find Properties
                </a>
              </div>
            )}

            {activeTab === 'searches' && (
              <div className="text-center py-16">
                <Search className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No Search History</h3>
                <p className="text-gray-500 mb-6">Your property searches will be saved here for easy access.</p>
                <a href="/properties" className="inline-flex items-center px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors">
                  Start Searching
                </a>
              </div>
            )}

            {activeTab === 'wishlisted' && (
              <div className="text-center py-16">
                <Heart className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No Wishlisted Properties</h3>
                <p className="text-gray-500 mb-6">Heart your favorite properties to create your dream home wishlist.</p>
                <a href="/properties" className="inline-flex items-center px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors">
                  Explore Properties
                </a>
              </div>
            )}

            {activeTab === 'saved' && (
              <div className="text-center py-16">
                <Bookmark className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No Saved Properties</h3>
                <p className="text-gray-500 mb-6">Bookmark properties for quick access and comparison.</p>
                <a href="/properties" className="inline-flex items-center px-4 py-2 bg-orange-600 text-white rounded-lg hover:bg-orange-700 transition-colors">
                  Browse Properties
                </a>
              </div>
            )}


          </div>
        </div>
      </div>
    </div>
  );
};

export default YourActivity;