import React, { useState, useEffect } from 'react';
import { Shield, Unlock, RefreshCw, Clock, User } from 'lucide-react';
import { toast } from 'sonner';

interface LockedAccount {
  user_id: string;
  user_type: string;
  failed_attempts: number;
  locked_at: string;
  unlock_at: string;
}

export const AccountLockoutAdmin: React.FC = () => {
  const [lockedAccounts, setLockedAccounts] = useState<LockedAccount[]>([]);
  const [loading, setLoading] = useState(false);
  const [unlockingId, setUnlockingId] = useState<string | null>(null);

  const fetchLockedAccounts = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('/api/account/lockout/locked-accounts', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        const data = await response.json();
        setLockedAccounts(data.data || []);
      } else {
        throw new Error('Failed to fetch locked accounts');
      }
    } catch (error) {
      console.error('Error fetching locked accounts:', error);
      toast.error('Failed to load locked accounts');
    } finally {
      setLoading(false);
    }
  };

  const unlockAccount = async (userId: string, userType: string) => {
    setUnlockingId(userId);
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch(`/api/account/lockout/unlock/${userId}?user_type=${userType}`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        toast.success('Account unlocked successfully');
        fetchLockedAccounts(); // Refresh the list
      } else {
        throw new Error('Failed to unlock account');
      }
    } catch (error) {
      console.error('Error unlocking account:', error);
      toast.error('Failed to unlock account');
    } finally {
      setUnlockingId(null);
    }
  };

  const triggerAutoUnlock = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('/api/account/lockout/auto-unlock', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        const data = await response.json();
        toast.success(data.message);
        fetchLockedAccounts(); // Refresh the list
      } else {
        throw new Error('Failed to trigger auto-unlock');
      }
    } catch (error) {
      console.error('Error triggering auto-unlock:', error);
      toast.error('Failed to trigger auto-unlock');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchLockedAccounts();
    // Auto-refresh every 30 seconds
    const interval = setInterval(fetchLockedAccounts, 30000);
    return () => clearInterval(interval);
  }, []);

  const formatTimeRemaining = (unlockAt: string) => {
    const now = new Date();
    const unlock = new Date(unlockAt);
    const diff = unlock.getTime() - now.getTime();
    
    if (diff <= 0) return 'Expired';
    
    const minutes = Math.floor(diff / (1000 * 60));
    const hours = Math.floor(minutes / 60);
    const remainingMinutes = minutes % 60;
    
    if (hours > 0) {
      return `${hours}h ${remainingMinutes}m`;
    }
    return `${remainingMinutes}m`;
  };

  return (
    <div className="bg-white rounded-lg shadow p-6">
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center gap-2">
          <Shield className="h-5 w-5 text-red-500" />
          <h2 className="text-lg font-semibold text-gray-900">Account Lockouts</h2>
        </div>
        <div className="flex gap-2">
          <button
            onClick={triggerAutoUnlock}
            disabled={loading}
            className="flex items-center gap-2 px-3 py-2 text-sm bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50"
          >
            <RefreshCw className={`h-4 w-4 ${loading ? 'animate-spin' : ''}`} />
            Auto-Unlock Expired
          </button>
          <button
            onClick={fetchLockedAccounts}
            disabled={loading}
            className="flex items-center gap-2 px-3 py-2 text-sm bg-gray-600 text-white rounded-md hover:bg-gray-700 disabled:opacity-50"
          >
            <RefreshCw className={`h-4 w-4 ${loading ? 'animate-spin' : ''}`} />
            Refresh
          </button>
        </div>
      </div>

      {loading && lockedAccounts.length === 0 ? (
        <div className="text-center py-8">
          <RefreshCw className="h-8 w-8 animate-spin mx-auto text-gray-400 mb-2" />
          <p className="text-gray-500">Loading locked accounts...</p>
        </div>
      ) : lockedAccounts.length === 0 ? (
        <div className="text-center py-8">
          <Shield className="h-8 w-8 mx-auto text-green-400 mb-2" />
          <p className="text-gray-500">No accounts are currently locked</p>
        </div>
      ) : (
        <div className="space-y-4">
          {lockedAccounts.map((account) => (
            <div
              key={`${account.user_id}-${account.user_type}`}
              className="border border-red-200 rounded-lg p-4 bg-red-50"
            >
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <User className="h-5 w-5 text-red-500" />
                  <div>
                    <p className="font-medium text-gray-900">
                      {account.user_type.charAt(0).toUpperCase() + account.user_type.slice(1)} Account
                    </p>
                    <p className="text-sm text-gray-600">ID: {account.user_id}</p>
                  </div>
                </div>
                <div className="text-right">
                  <div className="flex items-center gap-2 text-sm text-red-600 mb-1">
                    <Clock className="h-4 w-4" />
                    <span>Unlocks in: {formatTimeRemaining(account.unlock_at)}</span>
                  </div>
                  <p className="text-xs text-gray-500">
                    Failed attempts: {account.failed_attempts}
                  </p>
                </div>
              </div>
              <div className="mt-3 flex items-center justify-between">
                <div className="text-xs text-gray-500">
                  <p>Locked: {new Date(account.locked_at).toLocaleString()}</p>
                  <p>Unlocks: {new Date(account.unlock_at).toLocaleString()}</p>
                </div>
                <button
                  onClick={() => unlockAccount(account.user_id, account.user_type)}
                  disabled={unlockingId === account.user_id}
                  className="flex items-center gap-2 px-3 py-1 text-sm bg-red-600 text-white rounded hover:bg-red-700 disabled:opacity-50"
                >
                  <Unlock className={`h-4 w-4 ${unlockingId === account.user_id ? 'animate-pulse' : ''}`} />
                  {unlockingId === account.user_id ? 'Unlocking...' : 'Unlock Now'}
                </button>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
};