import React, { useState } from 'react';
import { useCoupons } from '../../hooks/usePromotions';
import { Coupon } from '../../types/promotions';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090';

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    'Content-Type': 'application/json',
    'Authorization': token ? `Bearer ${token}` : ''
  };
};

export const AdminCoupons: React.FC = () => {
  const { coupons, loading, refetch } = useCoupons();
  const [showModal, setShowModal] = useState(false);
  const [editingCoupon, setEditingCoupon] = useState<Coupon | null>(null);
  const [formData, setFormData] = useState<Partial<Coupon>>({
    code: '',
    description: '',
    discount_type: 'percentage',
    discount_value: 0,
    applicable_to: 'all',
    is_active: true
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      const url = editingCoupon 
        ? `${API_URL}/api/admin/coupons/${editingCoupon.id}/`
        : `${API_URL}/api/admin/coupons/`;
      
      await fetch(url, {
        method: editingCoupon ? 'PUT' : 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify(formData)
      });
      
      setShowModal(false);
      setEditingCoupon(null);
      setFormData({ code: '', description: '', discount_type: 'percentage', discount_value: 0, applicable_to: 'all', is_active: true });
      refetch();
    } catch (error) {
      console.error('Error saving coupon:', error);
    }
  };

  const toggleStatus = async (id: number) => {
    try {
      await fetch(`${API_URL}/api/admin/coupons/${id}/status/`, { 
        method: 'PATCH',
        headers: getAuthHeaders()
      });
      refetch();
    } catch (error) {
      console.error('Error toggling status:', error);
    }
  };

  const openEditModal = (coupon: Coupon) => {
    setEditingCoupon(coupon);
    setFormData(coupon);
    setShowModal(true);
  };

  if (loading) return <div className="p-6">Loading...</div>;

  return (
    <div className="p-6">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold">Coupon Management</h1>
        <button
          onClick={() => { setShowModal(true); setEditingCoupon(null); }}
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
        >
          Create Coupon
        </button>
      </div>

      <div className="bg-white rounded-lg shadow overflow-hidden">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Code</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Value</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Usage</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Valid Until</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {coupons.map((coupon) => (
              <tr key={coupon.id}>
                <td className="px-6 py-4 whitespace-nowrap font-mono font-bold">{coupon.code}</td>
                <td className="px-6 py-4 whitespace-nowrap">{coupon.discount_type}</td>
                <td className="px-6 py-4 whitespace-nowrap">
                  {coupon.discount_type === 'percentage' ? `${coupon.discount_value}%` : `₹${coupon.discount_value}`}
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  {coupon.used_count} / {coupon.usage_limit || '∞'}
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  {coupon.valid_until ? new Date(coupon.valid_until).toLocaleDateString() : 'No expiry'}
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <span className={`px-2 py-1 rounded text-xs ${coupon.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                    {coupon.is_active ? 'Active' : 'Inactive'}
                  </span>
                </td>
                <td className="px-6 py-4 whitespace-nowrap space-x-2">
                  <button onClick={() => openEditModal(coupon)} className="text-blue-600 hover:text-blue-800">Edit</button>
                  <button onClick={() => toggleStatus(coupon.id!)} className="text-gray-600 hover:text-gray-800">Toggle</button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {showModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4 overflow-y-auto">
          <div className="bg-white rounded-lg w-full max-w-lg my-8 flex flex-col max-h-[calc(100vh-4rem)]">
            <div className="p-4 border-b flex-shrink-0">
              <h2 className="text-xl font-bold">{editingCoupon ? 'Edit Coupon' : 'Create Coupon'}</h2>
            </div>
            <form onSubmit={handleSubmit} className="flex-1 flex flex-col overflow-hidden min-h-0">
            <div className="flex-1 overflow-y-auto p-4 space-y-3">
              <div>
                <label className="block text-sm font-medium mb-1">Coupon Code *</label>
                <input
                  type="text"
                  value={formData.code}
                  onChange={(e) => setFormData({ ...formData, code: e.target.value.toUpperCase() })}
                  className="w-full border rounded px-3 py-2"
                  placeholder="e.g., SUMMER50"
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Description</label>
                <textarea
                  value={formData.description}
                  onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                  className="w-full border rounded px-3 py-2"
                  rows={2}
                  placeholder="Brief description of the coupon"
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Discount Type *</label>
                  <select
                    value={formData.discount_type}
                    onChange={(e) => setFormData({ ...formData, discount_type: e.target.value as any })}
                    className="w-full border rounded px-3 py-2"
                  >
                    <option value="percentage">Percentage (%)</option>
                    <option value="fixed">Fixed Amount (₹)</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Discount Value *</label>
                  <input
                    type="number"
                    step="0.01"
                    value={formData.discount_value}
                    onChange={(e) => setFormData({ ...formData, discount_value: parseFloat(e.target.value) })}
                    className="w-full border rounded px-3 py-2"
                    placeholder={formData.discount_type === 'percentage' ? '50' : '500'}
                    required
                  />
                </div>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Max Discount (₹)</label>
                  <input
                    type="number"
                    value={formData.max_discount || ''}
                    onChange={(e) => setFormData({ ...formData, max_discount: e.target.value ? parseFloat(e.target.value) : undefined })}
                    className="w-full border rounded px-3 py-2"
                    placeholder="Optional"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Usage Limit</label>
                  <input
                    type="number"
                    value={formData.usage_limit || ''}
                    onChange={(e) => setFormData({ ...formData, usage_limit: e.target.value ? parseInt(e.target.value) : undefined })}
                    className="w-full border rounded px-3 py-2"
                    placeholder="Unlimited"
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Valid Until</label>
                <input
                  type="datetime-local"
                  value={formData.valid_until ? formData.valid_until.slice(0, 16) : ''}
                  onChange={(e) => setFormData({ ...formData, valid_until: e.target.value })}
                  className="w-full border rounded px-3 py-2"
                />
              </div>
              <div className="flex items-center">
                <input
                  type="checkbox"
                  checked={formData.is_active}
                  onChange={(e) => setFormData({ ...formData, is_active: e.target.checked })}
                  className="mr-2"
                  id="coupon-active"
                />
                <label htmlFor="coupon-active" className="text-sm font-medium">Active</label>
              </div>
            </div>
            <div className="p-4 border-t bg-gray-50 flex-shrink-0">
              <div className="flex justify-end space-x-2">
                <button
                  type="button"
                  onClick={() => { setShowModal(false); setEditingCoupon(null); }}
                  className="px-4 py-2 border rounded hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button type="submit" className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700">
                  {editingCoupon ? 'Update' : 'Create'}
                </button>
              </div>
            </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};
