import React, { useState, useEffect, useMemo, useCallback } from "react";
import { useNavigate, useLocation, Routes, Route } from "react-router-dom";
import Sidebar from "./Sidebar";
import { Header } from "./Header";
import { Header as MainHeader } from "../Header";
import { Footer } from "./Footer";
import { DashboardOverview } from "./DashboardOverview";
import { PropertyManagement } from "./PropertyManagement";
import { UserManagement } from "./UserManagement";
import { FinancialsPayments } from "./FinancialsPayments";
import { PaymentAnalytics } from "./PaymentAnalytics";
import { RefundManagement } from "./RefundManagement";
import { CampaignManagement } from "./CampaignManagement";
import { BusinessRevenueDashboard } from "./BusinessRevenueDashboard";

import { SecurityAudit } from "./SecurityAudit";
import { ConfigurationManagement } from "./ConfigurationManagement";
import { PropertyTrends } from "./PropertyTrends";
import { SupportTickets } from "./SupportTickets";
import GrievancePage from '../../pages/GrievancePage';
import { VisitSchedules } from "./VisitSchedules";
import { InquiryManagement } from "./InquiryManagement";
import { AuctionManagement } from "./AuctionManagement";
import { RiskComplianceAdmin } from "./RiskComplianceAdmin";
import { HyperlocalEconomic } from "./HyperlocalEconomic";
import { NRIAssistance } from "./NRIAssistance";
import { ComplianceManagement } from "./ComplianceManagement";
import { FeatureFlags } from "./FeatureFlags";
import { UsageAnalytics } from "./UsageAnalytics";
import { BillingDashboard } from "./BillingDashboard";
import { ContentQualityGovernance } from "./ContentQualityGovernance";
import { PricePrediction } from "../../pages/price-prediction/PricePrediction";
import { AdminEscrowList } from "../../pages/AdminEscrowList";
import MarketAnalysis from "./MarketAnalysis";
import { AdminSystemHealth } from "./AdminSystemHealth";
import DataRetentionDashboard from '../data-retention/DataRetentionDashboard';
import SystemHealthDashboard from '../../system-health/components/SystemHealthDashboard';
import AdminListingPackages from './AdminListingPackages';
import AdminWallet from '../wallet/AdminWallet';
import { AdminCoupons } from './AdminCoupons';
import { AdminDiscountRules } from './AdminDiscountRules';
import { AdminScholarships } from './AdminScholarships';
import AdminEntitlementManager from '../entitlements/AdminEntitlementManager';
import { DataManagement } from "./DataManagement";
import PlatformSettings from "../admin/PlatformSettings";
import AdminAPIKeys from "../../pages/admin/developer/APIKeys";
import AdminWebhooks from "../../pages/admin/developer/Webhooks";
import Marketplace from '../../pages/admin/developer/Marketplace';
import Connectors from '../../pages/admin/developer/Connectors';
import Vault from '../../pages/admin/developer/Vault';
import ConnectorCallback from '../../pages/admin/developer/ConnectorCallback';
import LedgerPage from './Finance/LedgerPage';
import RefundsPage from './Finance/RefundsPage';
import TaxSettingsPage from './Finance/TaxSettingsPage';
import UsageLimitsPage from './Finance/UsageLimitsPage';
import { DocumentVerification } from "./DocumentVerification";

export function AdminDashboard() {
  const [adminName, setAdminName] = useState("");
  const [isSidebarCollapsed, setIsSidebarCollapsed] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const navigate = useNavigate();
  const location = useLocation();

  // Prevent flash of content before auth check
  const [isInitialized, setIsInitialized] = useState(false);

  // Memoize the sidebar toggle handler
  const handleSidebarToggle = useCallback((collapsed: boolean) => {
    setIsSidebarCollapsed(collapsed);
  }, []);

  // Handle browser back/forward navigation
  useEffect(() => {
    const handlePopState = () => {
      // Ensure we stay in admin context
      if (window.location.pathname.startsWith('/admin')) {
        setIsLoading(false);
        setIsInitialized(true);
      }
    };

    window.addEventListener('popstate', handlePopState);
    return () => window.removeEventListener('popstate', handlePopState);
  }, []);

  // Initialize once on mount
  useEffect(() => {
    const initializeDashboard = async () => {
      try {
        // Allow access to any logged-in user
        const accessToken = localStorage.getItem('access_token');
        const adminName = localStorage.getItem("adminName");
        const user = JSON.parse(localStorage.getItem('user') || '{}');

        if (!accessToken) {
          navigate("/login");
          return;
        }

        // Get admin name from token or user data
        let displayName = "Admin";
        try {
          const payload = JSON.parse(atob(accessToken.split('.')[1]));
          const userEmail = localStorage.getItem('user_email') || user.email;
          if (userEmail) {
            displayName = userEmail.split('@')[0] || "Admin";
          }
        } catch (e) {
          displayName = adminName || user.first_name || "Admin";
        }
        setAdminName(displayName);

        // Handle navigation from notifications
        if (location.state?.activeSection) {
          // Clear the navigation state to prevent issues on refresh
          window.history.replaceState({}, document.title);
        }
      } catch (error) {
        console.error('Dashboard initialization error:', error);
        navigate("/login");
      } finally {
        // Small delay to prevent flash
        setIsLoading(false);
        setIsInitialized(true);
      }
    };

    initializeDashboard();
  }, [navigate, location.state]);

  const handleLogout = () => {
    localStorage.removeItem('access_token');
    localStorage.removeItem('refresh_token');
    localStorage.removeItem('user');
    localStorage.removeItem("adminLoggedIn");
    localStorage.removeItem("adminName");
    navigate("/login");
  };

  const renderActiveSection = () => {
    return (
      <Routes>
        <Route path="/" element={<DashboardOverview />} />
        <Route path="/properties" element={<PropertyManagement />} />
        <Route path="/users" element={<UserManagement />} />
        <Route path="/visits" element={<VisitSchedules />} />

        <Route path="/financials" element={<LedgerPage />} />
        <Route path="/documents" element={<DocumentVerification />} />
        <Route path="/support" element={<SupportTickets />} />
        <Route path="/grievances" element={<GrievancePage />} />

        <Route path="/auctions" element={<AuctionManagement />} />
        <Route path="/inquiries" element={<InquiryManagement />} />
        <Route path="/risk-compliance" element={<RiskComplianceAdmin />} />
        <Route path="/compliance" element={<ComplianceManagement />} />
        <Route path="/feature-flags" element={<FeatureFlags />} />
        <Route path="/usage-analytics" element={<UsageAnalytics />} />
        <Route path="/hyperlocal" element={<HyperlocalEconomic />} />
        <Route path="/price-prediction" element={<PricePrediction />} />
        <Route path="/escrows" element={<AdminEscrowList />} />
        <Route path="/payment-analytics" element={<PaymentAnalytics />} />
        <Route path="/refunds" element={<RefundsPage />} />
        <Route path="/tax-settings" element={<TaxSettingsPage />} />
        <Route path="/usage-limits" element={<UsageLimitsPage />} />
        <Route path="/campaigns" element={<CampaignManagement />} />

        <Route path="/nri-assistance" element={<NRIAssistance />} />
        <Route path="/trends" element={<PropertyTrends />} />
        <Route path="/developer/api-keys" element={<AdminAPIKeys />} />
        <Route path="/developer/webhooks" element={<AdminWebhooks />} />
      </Routes>
    );
  };
  // Memoize the routes to prevent re-creation
  const routes = useMemo(() => (
    <Routes>
      <Route path="/" element={<DashboardOverview />} />
      <Route path="/properties" element={<PropertyManagement />} />
      <Route path="/users" element={<UserManagement />} />
      <Route path="/visits" element={<VisitSchedules />} />
      <Route path="/listing-packages" element={<AdminListingPackages />} />
      <Route path="/coupons" element={<AdminCoupons />} />
      <Route path="/discount-rules" element={<AdminDiscountRules />} />
      <Route path="/scholarships" element={<AdminScholarships />} />
      <Route path="/wallet" element={<AdminWallet />} />
      <Route path="/entitlements" element={<AdminEntitlementManager />} />

      <Route path="/financials" element={<LedgerPage />} />
      <Route path="/documents" element={<DocumentVerification />} />
      <Route path="/support" element={<SupportTickets />} />
      <Route path="/grievances" element={<GrievancePage />} />
      <Route path="/settings" element={<SecurityAudit />} />
      <Route path="/financials" element={<FinancialsPayments />} />
      <Route path="/security" element={<SecurityAudit />} />
      <Route path="/configuration" element={<ConfigurationManagement />} />

      <Route path="/auctions" element={<AuctionManagement />} />
      <Route path="/inquiries" element={<InquiryManagement />} />
      <Route path="/risk-compliance" element={<RiskComplianceAdmin />} />
      <Route path="/compliance" element={<ComplianceManagement />} />
      <Route path="/feature-flags" element={<FeatureFlags />} />
      <Route path="/usage-analytics" element={<UsageAnalytics />} />
      <Route path="/billing" element={<BillingDashboard />} />
      <Route path="/content-quality" element={<ContentQualityGovernance />} />
      <Route path="/hyperlocal" element={<HyperlocalEconomic />} />
      <Route path="/price-prediction" element={<PricePrediction />} />
      <Route path="/escrows" element={<AdminEscrowList />} />
      <Route path="/payment-analytics" element={<PaymentAnalytics />} />
      <Route path="/refunds" element={<RefundsPage />} />
      <Route path="/tax-settings" element={<TaxSettingsPage />} />
      <Route path="/usage-limits" element={<UsageLimitsPage />} />
      <Route path="/campaigns" element={<CampaignManagement />} />
      <Route path="/campaign-management" element={<CampaignManagement />} />
      <Route path="/business-revenue" element={<BusinessRevenueDashboard />} />
      <Route path="/system-health" element={<SystemHealthDashboard />} />
      <Route path="/admin-system-health" element={<AdminSystemHealth />} />

      <Route path="/nri-assistance" element={<NRIAssistance />} />
      <Route path="/trends" element={<PropertyTrends />} />
      <Route path="/market-analysis" element={<MarketAnalysis />} />
      <Route path="/data-retention" element={<DataRetentionDashboard />} />
      <Route path="/data-management" element={<DataManagement />} />
      <Route path="/platform-settings" element={<PlatformSettings />} />
      <Route path="developer/api-keys" element={<AdminAPIKeys />} />
      <Route path="developer/webhooks" element={<AdminWebhooks />} />
      <Route path="developer/marketplace" element={<Marketplace />} />
      <Route path="developer/connectors" element={<Connectors />} />
      <Route path="developer/connectors/callback" element={<ConnectorCallback />} />
      <Route path="developer/vault" element={<Vault />} />
    </Routes>
  ), []);

  if (!isInitialized) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading dashboard...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <MainHeader />
      <Sidebar
        isSidebarCollapsed={isSidebarCollapsed}
        setIsSidebarCollapsed={handleSidebarToggle}
      />

      <div
        className="min-h-screen bg-gray-50 transition-all duration-300"
        style={{
          marginLeft: isSidebarCollapsed ? '80px' : '256px',
          marginTop: '64px'
        }}
      >
        <main className="p-8">
          {routes}
        </main>
        <Footer />
      </div>
    </div>
  );
}

