import React, { useState } from 'react';
import { useDiscountRules } from '../../hooks/usePromotions';
import { DiscountRule } from '../../types/promotions';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090';

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    'Content-Type': 'application/json',
    'Authorization': token ? `Bearer ${token}` : ''
  };
};

export const AdminDiscountRules: React.FC = () => {
  const { rules, loading, refetch } = useDiscountRules();
  const [showModal, setShowModal] = useState(false);
  const [editingRule, setEditingRule] = useState<DiscountRule | null>(null);
  const [formData, setFormData] = useState<Partial<DiscountRule>>({
    name: '',
    description: '',
    rule_type: 'first_time',
    discount_type: 'percentage',
    discount_value: 0,
    conditions: {},
    priority: 0,
    is_active: true
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      const url = editingRule 
        ? `${API_URL}/api/admin/discount-rules/${editingRule.id}/`
        : `${API_URL}/api/admin/discount-rules/`;
      
      await fetch(url, {
        method: editingRule ? 'PUT' : 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify(formData)
      });
      
      setShowModal(false);
      setEditingRule(null);
      setFormData({ name: '', description: '', rule_type: 'first_time', discount_type: 'percentage', discount_value: 0, conditions: {}, priority: 0, is_active: true });
      refetch();
    } catch (error) {
      console.error('Error saving rule:', error);
    }
  };

  const toggleStatus = async (id: number) => {
    try {
      await fetch(`${API_URL}/api/admin/discount-rules/${id}/status/`, { 
        method: 'PATCH',
        headers: getAuthHeaders()
      });
      refetch();
    } catch (error) {
      console.error('Error toggling status:', error);
    }
  };

  const openEditModal = (rule: DiscountRule) => {
    setEditingRule(rule);
    setFormData(rule);
    setShowModal(true);
  };

  if (loading) return <div className="p-6">Loading...</div>;

  return (
    <div className="p-6">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold">Automatic Discount Rules</h1>
        <button
          onClick={() => { setShowModal(true); setEditingRule(null); }}
          className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
        >
          Create Rule
        </button>
      </div>

      <div className="grid gap-4">
        {rules.map((rule) => (
          <div key={rule.id} className="bg-white rounded-lg shadow p-6">
            <div className="flex justify-between items-start">
              <div className="flex-1">
                <div className="flex items-center gap-3 mb-2">
                  <h3 className="text-lg font-semibold">{rule.name}</h3>
                  <span className={`px-2 py-1 rounded text-xs ${rule.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                    {rule.is_active ? 'Active' : 'Inactive'}
                  </span>
                  <span className="px-2 py-1 rounded text-xs bg-blue-100 text-blue-800">
                    Priority: {rule.priority}
                  </span>
                </div>
                <p className="text-gray-600 text-sm mb-3">{rule.description}</p>
                <div className="flex gap-4 text-sm">
                  <div>
                    <span className="font-medium">Type:</span> {rule.rule_type}
                  </div>
                  <div>
                    <span className="font-medium">Discount:</span>{' '}
                    {rule.discount_type === 'percentage' ? `${rule.discount_value}%` : `₹${rule.discount_value}`}
                  </div>
                  <div>
                    <span className="font-medium">Conditions:</span> {JSON.stringify(rule.conditions)}
                  </div>
                </div>
              </div>
              <div className="flex gap-2">
                <button onClick={() => openEditModal(rule)} className="text-blue-600 hover:text-blue-800 text-sm">
                  Edit
                </button>
                <button onClick={() => toggleStatus(rule.id!)} className="text-gray-600 hover:text-gray-800 text-sm">
                  Toggle
                </button>
              </div>
            </div>
          </div>
        ))}
      </div>

      {showModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4 overflow-y-auto">
          <div className="bg-white rounded-lg w-full max-w-lg my-8 flex flex-col max-h-[calc(100vh-4rem)]">
            <div className="p-4 border-b flex-shrink-0">
              <h2 className="text-xl font-bold">{editingRule ? 'Edit Rule' : 'Create Rule'}</h2>
            </div>
            <form onSubmit={handleSubmit} className="flex-1 flex flex-col overflow-hidden min-h-0">
            <div className="flex-1 overflow-y-auto p-4 space-y-3">
              <div>
                <label className="block text-sm font-medium mb-1">Rule Name *</label>
                <input
                  type="text"
                  value={formData.name}
                  onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                  className="w-full border rounded px-3 py-2"
                  placeholder="e.g., First Time User Discount"
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Description</label>
                <textarea
                  value={formData.description}
                  onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                  className="w-full border rounded px-3 py-2"
                  rows={2}
                  placeholder="Explain when this discount applies"
                />
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Rule Type *</label>
                  <select
                    value={formData.rule_type}
                    onChange={(e) => setFormData({ ...formData, rule_type: e.target.value as any })}
                    className="w-full border rounded px-3 py-2"
                  >
                    <option value="first_time">First Time User</option>
                    <option value="bulk">Bulk Purchase</option>
                    <option value="seasonal">Seasonal</option>
                    <option value="upgrade">Upgrade</option>
                    <option value="bundle">Bundle</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Priority *</label>
                  <input
                    type="number"
                    value={formData.priority}
                    onChange={(e) => setFormData({ ...formData, priority: parseInt(e.target.value) })}
                    className="w-full border rounded px-3 py-2"
                    placeholder="Higher = Applied first"
                  />
                </div>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Discount Type *</label>
                  <select
                    value={formData.discount_type}
                    onChange={(e) => setFormData({ ...formData, discount_type: e.target.value as any })}
                    className="w-full border rounded px-3 py-2"
                  >
                    <option value="percentage">Percentage (%)</option>
                    <option value="fixed">Fixed Amount (₹)</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Discount Value *</label>
                  <input
                    type="number"
                    step="0.01"
                    value={formData.discount_value}
                    onChange={(e) => setFormData({ ...formData, discount_value: parseFloat(e.target.value) })}
                    className="w-full border rounded px-3 py-2"
                    placeholder={formData.discount_type === 'percentage' ? '15' : '1000'}
                    required
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Conditions (JSON)</label>
                <textarea
                  value={JSON.stringify(formData.conditions, null, 2)}
                  onChange={(e) => {
                    try {
                      setFormData({ ...formData, conditions: JSON.parse(e.target.value) });
                    } catch {}
                  }}
                  className="w-full border rounded px-3 py-2 font-mono text-sm"
                  rows={4}
                  placeholder='{"billing_cycle": "Yearly"}'
                />
              </div>
              <div className="flex items-center">
                <input
                  type="checkbox"
                  checked={formData.is_active}
                  onChange={(e) => setFormData({ ...formData, is_active: e.target.checked })}
                  className="mr-2"
                  id="rule-active"
                />
                <label htmlFor="rule-active" className="text-sm font-medium">Active</label>
              </div>
            </div>
            <div className="p-4 border-t bg-gray-50 flex-shrink-0">
              <div className="flex justify-end space-x-2">
                <button
                  type="button"
                  onClick={() => { setShowModal(false); setEditingRule(null); }}
                  className="px-4 py-2 border rounded hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button type="submit" className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700">
                  {editingRule ? 'Update' : 'Create'}
                </button>
              </div>
            </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};
