import React, { useState } from 'react';
import { usePricingPlans, usePackages } from '../../hooks/useListingPackages';
import { PricingPlan, ListingPackage } from '../../types/listingPackages';
import { Card, CardContent } from '../ui/card';
import { Button } from '../ui/button';
import { Badge } from '../ui/badge';
import { Package, Plus, Edit, Power, Star, ChevronLeft, ChevronRight, X } from 'lucide-react';

const API_BASE_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090';

const AdminListingPackages: React.FC = () => {
  const { plans, loading, refetch } = usePricingPlans();
  const { packages } = usePackages();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingPlan, setEditingPlan] = useState<PricingPlan | null>(null);
  const [formData, setFormData] = useState({
    name: '',
    package_id: '',
    billing_cycle: '',
    price: 0,
    currency: 'INR',
    discount_percent: 0,
    duration_days: 30,
    limits_json: {},
    is_default: false,
    is_recommended: false,
    is_active: true
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    const url = editingPlan 
      ? `${API_BASE_URL}/api/admin/pricing-plans/${editingPlan.id}/` 
      : `${API_BASE_URL}/api/admin/pricing-plans/`;
    const method = editingPlan ? 'PUT' : 'POST';
    
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch(url, {
        method,
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify(formData)
      });
      
      if (response.ok) {
        refetch();
        setIsModalOpen(false);
        resetForm();
      }
    } catch (error) {
      console.error('Error saving plan:', error);
    }
  };

  const togglePlanStatus = async (id: number) => {
    try {
      const token = localStorage.getItem('access_token');
      await fetch(`${API_BASE_URL}/api/admin/pricing-plans/${id}/status/`, { 
        method: 'PATCH',
        headers: { 'Authorization': `Bearer ${token}` }
      });
      refetch();
    } catch (error) {
      console.error('Error toggling status:', error);
    }
  };

  const setDefaultPlan = async (id: number) => {
    try {
      const token = localStorage.getItem('access_token');
      await fetch(`${API_BASE_URL}/api/admin/pricing-plans/${id}/set-default/`, { 
        method: 'PATCH',
        headers: { 'Authorization': `Bearer ${token}` }
      });
      refetch();
    } catch (error) {
      console.error('Error setting default:', error);
    }
  };

  const resetForm = () => {
    setFormData({
      name: '',
      package_id: '',
      billing_cycle: '',
      price: 0,
      currency: 'INR',
      discount_percent: 0,
      duration_days: 30,
      limits_json: {},
      is_default: false,
      is_recommended: false,
      is_active: true
    });
    setEditingPlan(null);
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900">Listing Packages Management</h1>
          <p className="text-sm text-gray-600 mt-1">Manage pricing plans and subscription packages</p>
        </div>
        <Button 
          onClick={() => setIsModalOpen(true)}
          size="sm"
          className="bg-blue-600 hover:bg-blue-700 text-xs"
        >
          <Plus className="w-3 h-3 mr-1" />
          Add New Plan
        </Button>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Total Plans</p>
                <p className="text-2xl font-semibold text-gray-900 mt-1">{plans.length}</p>
              </div>
              <div className="p-2 bg-[#2B256D]/10 rounded-lg">
                <Package className="w-5 h-5 text-[#2B256D]" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Active Plans</p>
                <p className="text-2xl font-semibold text-gray-900 mt-1">
                  {plans.filter(p => p.is_active).length}
                </p>
              </div>
              <div className="p-2 bg-green-100 rounded-lg">
                <Package className="w-5 h-5 text-green-600" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Packages</p>
                <p className="text-2xl font-semibold text-gray-900 mt-1">{packages.length}</p>
              </div>
              <div className="p-2 bg-blue-100 rounded-lg">
                <Package className="w-5 h-5 text-blue-600" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Recommended</p>
                <p className="text-2xl font-semibold text-gray-900 mt-1">
                  {plans.filter(p => p.is_recommended).length}
                </p>
              </div>
              <div className="p-2 bg-yellow-100 rounded-lg">
                <Star className="w-5 h-5 text-yellow-600" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Table */}
      <Card className="border border-gray-200 shadow-sm">
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50 border-b">
                <tr>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Plan Details</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Package</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Billing</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Price</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Status</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {plans.map((plan) => (
                  <tr key={plan.id} className="hover:bg-gray-50">
                    <td className="px-4 py-3">
                      <div className="flex items-center space-x-3">
                        <div className="w-7 h-7 bg-[#2B256D] rounded-full flex items-center justify-center text-white text-xs font-medium">
                          <Package className="w-3 h-3" />
                        </div>
                        <div>
                          <div className="flex items-center space-x-2">
                            <span className="text-sm font-medium text-gray-900">{plan.name}</span>
                            {plan.is_recommended && (
                              <Badge className="bg-yellow-100 text-yellow-800 text-xs">Recommended</Badge>
                            )}
                            {plan.is_default && (
                              <Badge className="bg-blue-100 text-blue-800 text-xs">Default</Badge>
                            )}
                          </div>
                          <div className="text-xs text-gray-500">ID: {plan.id}</div>
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="text-sm text-gray-900">{plan.package.name}</div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="space-y-1">
                        <div className="text-sm text-gray-900">{plan.billing_cycle}</div>
                        <div className="text-xs text-gray-500">{plan.duration_days} days</div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="text-sm font-medium text-gray-900">
                        {plan.currency} {plan.price.toLocaleString('en-IN')}
                      </div>
                      {plan.discount_percent > 0 && (
                        <div className="text-xs text-green-600">{plan.discount_percent}% off</div>
                      )}
                    </td>
                    <td className="px-4 py-3">
                      <Badge className={`${plan.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'} text-xs`}>
                        {plan.is_active ? 'Active' : 'Inactive'}
                      </Badge>
                    </td>
                    <td className="px-4 py-3">
                      <div className="flex items-center space-x-1">
                        <Button 
                          variant="ghost" 
                          size="sm" 
                          className="text-blue-600 hover:bg-blue-50 p-1"
                          onClick={() => {
                            setEditingPlan(plan);
                            setFormData({
                              name: plan.name,
                              package_id: plan.package.id.toString(),
                              billing_cycle: plan.billing_cycle,
                              price: plan.price,
                              currency: plan.currency,
                              discount_percent: plan.discount_percent,
                              duration_days: plan.duration_days,
                              limits_json: plan.limits_json,
                              is_default: plan.is_default,
                              is_recommended: plan.is_recommended,
                              is_active: plan.is_active
                            });
                            setIsModalOpen(true);
                          }}
                          title="Edit Plan"
                        >
                          <Edit className="w-3 h-3" />
                        </Button>
                        <Button 
                          variant="ghost" 
                          size="sm" 
                          className="text-orange-600 hover:bg-orange-50 p-1"
                          onClick={() => togglePlanStatus(plan.id)}
                          title="Toggle Status"
                        >
                          <Power className="w-3 h-3" />
                        </Button>
                        <Button 
                          variant="ghost" 
                          size="sm" 
                          className="text-green-600 hover:bg-green-50 p-1"
                          onClick={() => setDefaultPlan(plan.id)}
                          title="Set as Default"
                        >
                          <Star className="w-3 h-3" />
                        </Button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>

      {/* Modal */}
      {isModalOpen && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg w-full max-w-2xl">
            <div className="border-b px-6 py-4 flex items-center justify-between">
              <h2 className="text-xl font-semibold">{editingPlan ? 'Edit Plan' : 'Add New Plan'}</h2>
              <Button variant="ghost" size="sm" onClick={() => { setIsModalOpen(false); resetForm(); }}>
                <X className="w-5 h-5" />
              </Button>
            </div>
            
            <form onSubmit={handleSubmit} className="p-6 space-y-4">
              <div>
                <label className="block text-sm font-medium mb-1">Plan Name</label>
                <input
                  type="text"
                  value={formData.name}
                  onChange={(e) => setFormData({...formData, name: e.target.value})}
                  className="w-full border border-gray-300 rounded-md px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>

              <div>
                <label className="block text-sm font-medium mb-1">Package</label>
                <select
                  value={formData.package_id}
                  onChange={(e) => setFormData({...formData, package_id: e.target.value})}
                  className="w-full border border-gray-300 rounded-md px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                >
                  <option value="">Select Package</option>
                  {packages.map(pkg => (
                    <option key={pkg.id} value={pkg.id}>{pkg.name}</option>
                  ))}
                </select>
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Billing Cycle</label>
                  <select
                    value={formData.billing_cycle}
                    onChange={(e) => setFormData({...formData, billing_cycle: e.target.value})}
                    className="w-full border border-gray-300 rounded-md px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    required
                  >
                    <option value="">Select Cycle</option>
                    <option value="Monthly">Monthly</option>
                    <option value="Quarterly">Quarterly</option>
                    <option value="Yearly">Yearly</option>
                    <option value="One-time">One-time</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium mb-1">Price</label>
                  <input
                    type="number"
                    value={formData.price}
                    onChange={(e) => setFormData({...formData, price: parseFloat(e.target.value)})}
                    className="w-full border border-gray-300 rounded-md px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    required
                  />
                </div>
              </div>

              <div className="grid grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Currency</label>
                  <select
                    value={formData.currency}
                    onChange={(e) => setFormData({...formData, currency: e.target.value})}
                    className="w-full border border-gray-300 rounded-md px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  >
                    <option value="INR">INR</option>
                    <option value="USD">USD</option>
                    <option value="EUR">EUR</option>
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium mb-1">Discount %</label>
                  <input
                    type="number"
                    value={formData.discount_percent}
                    onChange={(e) => setFormData({...formData, discount_percent: parseFloat(e.target.value)})}
                    className="w-full border border-gray-300 rounded-md px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium mb-1">Duration (days)</label>
                  <input
                    type="number"
                    value={formData.duration_days}
                    onChange={(e) => setFormData({...formData, duration_days: parseInt(e.target.value)})}
                    className="w-full border border-gray-300 rounded-md px-3 py-2 text-sm focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                </div>
              </div>

              <div className="flex items-center space-x-4">
                <label className="flex items-center">
                  <input
                    type="checkbox"
                    checked={formData.is_recommended}
                    onChange={(e) => setFormData({...formData, is_recommended: e.target.checked})}
                    className="mr-2"
                  />
                  <span className="text-sm">Recommended</span>
                </label>
                <label className="flex items-center">
                  <input
                    type="checkbox"
                    checked={formData.is_default}
                    onChange={(e) => setFormData({...formData, is_default: e.target.checked})}
                    className="mr-2"
                  />
                  <span className="text-sm">Default</span>
                </label>
              </div>

              <div className="border-t pt-4 flex justify-end space-x-3">
                <Button 
                  type="button" 
                  variant="outline"
                  size="sm"
                  onClick={() => { setIsModalOpen(false); resetForm(); }}
                  className="text-xs"
                >
                  Cancel
                </Button>
                <Button 
                  type="submit"
                  size="sm"
                  className="bg-blue-600 hover:bg-blue-700 text-xs"
                >
                  {editingPlan ? 'Update' : 'Create'} Plan
                </Button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default AdminListingPackages;
