import React, { useState } from 'react';
import { useScholarshipPrograms, useScholarshipApplications } from '../../hooks/usePromotions';
import { ScholarshipProgram } from '../../types/promotions';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090';

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    'Content-Type': 'application/json',
    'Authorization': token ? `Bearer ${token}` : ''
  };
};

export const AdminScholarships: React.FC = () => {
  const { programs, loading: programsLoading, refetch: refetchPrograms } = useScholarshipPrograms();
  const { applications, loading: appsLoading, refetch: refetchApps } = useScholarshipApplications();
  const [activeTab, setActiveTab] = useState<'programs' | 'applications'>('programs');
  const [showModal, setShowModal] = useState(false);
  const [editingProgram, setEditingProgram] = useState<ScholarshipProgram | null>(null);
  const [formData, setFormData] = useState<Partial<ScholarshipProgram>>({
    name: '',
    code: '',
    description: '',
    eligibility_criteria: {},
    benefit_type: 'percentage',
    benefit_value: 0,
    duration_months: 12,
    is_active: true
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      const url = editingProgram 
        ? `${API_URL}/api/admin/scholarships/${editingProgram.id}/`
        : `${API_URL}/api/admin/scholarships/`;
      
      await fetch(url, {
        method: editingProgram ? 'PUT' : 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify(formData)
      });
      
      setShowModal(false);
      setEditingProgram(null);
      setFormData({ name: '', code: '', description: '', eligibility_criteria: {}, benefit_type: 'percentage', benefit_value: 0, duration_months: 12, is_active: true });
      refetchPrograms();
    } catch (error) {
      console.error('Error saving program:', error);
    }
  };

  const reviewApplication = async (appId: number, status: 'approved' | 'rejected', notes: string) => {
    try {
      await fetch(`${API_URL}/api/admin/scholarship-applications/${appId}/review/`, {
        method: 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify({ application_id: appId, status, review_notes: notes, reviewed_by: 1 })
      });
      refetchApps();
    } catch (error) {
      console.error('Error reviewing application:', error);
    }
  };

  const openEditModal = (program: ScholarshipProgram) => {
    setEditingProgram(program);
    setFormData(program);
    setShowModal(true);
  };

  if (programsLoading || appsLoading) return <div className="p-6">Loading...</div>;

  return (
    <div className="p-6">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-2xl font-bold">Scholarship Management</h1>
        {activeTab === 'programs' && (
          <button
            onClick={() => { setShowModal(true); setEditingProgram(null); }}
            className="bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700"
          >
            Create Program
          </button>
        )}
      </div>

      <div className="flex gap-2 mb-6">
        <button
          onClick={() => setActiveTab('programs')}
          className={`px-4 py-2 rounded ${activeTab === 'programs' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Programs ({programs.length})
        </button>
        <button
          onClick={() => setActiveTab('applications')}
          className={`px-4 py-2 rounded ${activeTab === 'applications' ? 'bg-blue-600 text-white' : 'bg-gray-200'}`}
        >
          Applications ({applications.length})
        </button>
      </div>

      {activeTab === 'programs' && (
        <div className="grid gap-4">
          {programs.map((program) => (
            <div key={program.id} className="bg-white rounded-lg shadow p-6">
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="text-lg font-semibold">{program.name}</h3>
                    <span className="px-2 py-1 rounded text-xs bg-purple-100 text-purple-800 font-mono">
                      {program.code}
                    </span>
                    <span className={`px-2 py-1 rounded text-xs ${program.is_active ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                      {program.is_active ? 'Active' : 'Inactive'}
                    </span>
                  </div>
                  <p className="text-gray-600 text-sm mb-3">{program.description}</p>
                  <div className="flex gap-4 text-sm">
                    <div>
                      <span className="font-medium">Benefit:</span>{' '}
                      {program.benefit_type === 'free' ? 'Free' : program.benefit_type === 'percentage' ? `${program.benefit_value}% off` : `₹${program.benefit_value} off`}
                    </div>
                    <div>
                      <span className="font-medium">Duration:</span> {program.duration_months} months
                    </div>
                    <div>
                      <span className="font-medium">Beneficiaries:</span> {program.current_beneficiaries} / {program.max_beneficiaries || '∞'}
                    </div>
                  </div>
                </div>
                <button onClick={() => openEditModal(program)} className="text-blue-600 hover:text-blue-800 text-sm">
                  Edit
                </button>
              </div>
            </div>
          ))}
        </div>
      )}

      {activeTab === 'applications' && (
        <div className="bg-white rounded-lg shadow overflow-hidden">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Program</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">User ID</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Applied</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {applications.map((app) => (
                <tr key={app.id}>
                  <td className="px-6 py-4 whitespace-nowrap">{app.program_name}</td>
                  <td className="px-6 py-4 whitespace-nowrap">{app.user_id}</td>
                  <td className="px-6 py-4 whitespace-nowrap">{app.user_type}</td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`px-2 py-1 rounded text-xs ${
                      app.status === 'approved' ? 'bg-green-100 text-green-800' :
                      app.status === 'rejected' ? 'bg-red-100 text-red-800' :
                      'bg-yellow-100 text-yellow-800'
                    }`}>
                      {app.status}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    {app.created_at ? new Date(app.created_at).toLocaleDateString() : '-'}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap space-x-2">
                    {app.status === 'pending' && (
                      <>
                        <button
                          onClick={() => reviewApplication(app.id!, 'approved', 'Approved')}
                          className="text-green-600 hover:text-green-800 text-sm"
                        >
                          Approve
                        </button>
                        <button
                          onClick={() => reviewApplication(app.id!, 'rejected', 'Rejected')}
                          className="text-red-600 hover:text-red-800 text-sm"
                        >
                          Reject
                        </button>
                      </>
                    )}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

      {showModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4 overflow-y-auto">
          <div className="bg-white rounded-lg w-full max-w-lg my-8 flex flex-col max-h-[calc(100vh-4rem)]">
            <div className="p-4 border-b flex-shrink-0">
              <h2 className="text-xl font-bold">{editingProgram ? 'Edit Program' : 'Create Program'}</h2>
            </div>
            <form onSubmit={handleSubmit} className="flex-1 flex flex-col overflow-hidden min-h-0">
            <div className="flex-1 overflow-y-auto p-4 space-y-3">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Program Name *</label>
                  <input
                    type="text"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    className="w-full border rounded px-3 py-2"
                    placeholder="e.g., First Time Seller Program"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Program Code *</label>
                  <input
                    type="text"
                    value={formData.code}
                    onChange={(e) => setFormData({ ...formData, code: e.target.value.toUpperCase() })}
                    className="w-full border rounded px-3 py-2"
                    placeholder="e.g., FTS2024"
                    required
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Description</label>
                <textarea
                  value={formData.description}
                  onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                  className="w-full border rounded px-3 py-2"
                  rows={2}
                  placeholder="Describe the scholarship program"
                />
              </div>
              <div className="grid grid-cols-3 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Benefit Type *</label>
                  <select
                    value={formData.benefit_type}
                    onChange={(e) => setFormData({ ...formData, benefit_type: e.target.value as any })}
                    className="w-full border rounded px-3 py-2"
                  >
                    <option value="free">Free (100%)</option>
                    <option value="percentage">Percentage Discount</option>
                    <option value="fixed">Fixed Amount Off</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Benefit Value</label>
                  <input
                    type="number"
                    step="0.01"
                    value={formData.benefit_value}
                    onChange={(e) => setFormData({ ...formData, benefit_value: parseFloat(e.target.value) })}
                    className="w-full border rounded px-3 py-2"
                    placeholder={formData.benefit_type === 'percentage' ? '50' : '1000'}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Duration (months) *</label>
                  <input
                    type="number"
                    value={formData.duration_months}
                    onChange={(e) => setFormData({ ...formData, duration_months: parseInt(e.target.value) })}
                    className="w-full border rounded px-3 py-2"
                    placeholder="12"
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Max Beneficiaries</label>
                <input
                  type="number"
                  value={formData.max_beneficiaries || ''}
                  onChange={(e) => setFormData({ ...formData, max_beneficiaries: e.target.value ? parseInt(e.target.value) : undefined })}
                  className="w-full border rounded px-3 py-2"
                  placeholder="Leave empty for unlimited"
                />
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">Eligibility Criteria (JSON) *</label>
                <textarea
                  value={JSON.stringify(formData.eligibility_criteria, null, 2)}
                  onChange={(e) => {
                    try {
                      setFormData({ ...formData, eligibility_criteria: JSON.parse(e.target.value) });
                    } catch {}
                  }}
                  className="w-full border rounded px-3 py-2 font-mono text-sm"
                  rows={4}
                  placeholder='{"user_type": "owner", "first_time": true}'
                />
              </div>
              <div className="flex items-center">
                <input
                  type="checkbox"
                  checked={formData.is_active}
                  onChange={(e) => setFormData({ ...formData, is_active: e.target.checked })}
                  className="mr-2"
                  id="program-active"
                />
                <label htmlFor="program-active" className="text-sm font-medium">Active</label>
              </div>
            </div>
            <div className="p-4 border-t bg-gray-50 flex-shrink-0">
              <div className="flex justify-end space-x-2">
                <button
                  type="button"
                  onClick={() => { setShowModal(false); setEditingProgram(null); }}
                  className="px-4 py-2 border rounded hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button type="submit" className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700">
                  {editingProgram ? 'Update' : 'Create'}
                </button>
              </div>
            </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};
