import React, { useState, useEffect } from 'react';
import { auctionManagementAPI, AuctionSession } from '../../api/auction/management';

export function AuctionManagement() {
  const [sessions, setSessions] = useState<AuctionSession[]>([]);
  const [loading, setLoading] = useState(true);
  const [statusFilter, setStatusFilter] = useState<string>('');
  const [extensionMinutes, setExtensionMinutes] = useState(5);

  useEffect(() => {
    fetchSessions();
  }, [statusFilter]);

  const fetchSessions = async () => {
    try {
      setLoading(true);
      const data = await auctionManagementAPI.getAllSessions(1, 20, statusFilter);
      setSessions(data.sessions);
    } catch (error) {
      console.error('Error fetching auction sessions:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleStartAuction = async (sessionId: string) => {
    try {
      await auctionManagementAPI.startAuction(sessionId);
      fetchSessions();
    } catch (error) {
      console.error('Error starting auction:', error);
    }
  };

  const handlePauseAuction = async (sessionId: string) => {
    try {
      await auctionManagementAPI.pauseAuction(sessionId);
      fetchSessions();
    } catch (error) {
      console.error('Error pausing auction:', error);
    }
  };

  const handleCancelAuction = async (sessionId: string) => {
    try {
      await auctionManagementAPI.cancelAuction(sessionId);
      fetchSessions();
    } catch (error) {
      console.error('Error cancelling auction:', error);
    }
  };

  const handleExtendAuction = async (sessionId: string) => {
    try {
      await auctionManagementAPI.extendAuction(sessionId, extensionMinutes);
      fetchSessions();
    } catch (error) {
      console.error('Error extending auction:', error);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'scheduled': return 'bg-blue-100 text-blue-800';
      case 'paused': return 'bg-yellow-100 text-yellow-800';
      case 'completed': return 'bg-gray-100 text-gray-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (loading) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900">Auction Management</h1>
          <p className="text-sm text-gray-600 mt-1">Manage and monitor auction sessions</p>
        </div>
        <select
          value={statusFilter}
          onChange={(e) => setStatusFilter(e.target.value)}
          className="px-4 py-2 border border-gray-300 rounded-lg text-sm"
        >
          <option value="">All Statuses</option>
          <option value="scheduled">Scheduled</option>
          <option value="active">Active</option>
          <option value="paused">Paused</option>
          <option value="completed">Completed</option>
          <option value="cancelled">Cancelled</option>
        </select>
      </div>

      {/* Table */}
      <div className="bg-white rounded-xl border border-gray-200 shadow-sm">
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50 border-b">
              <tr>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Property ID</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Status</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Current Bid</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Bidders</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">End Time</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Actions</th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {sessions.map((session) => (
                <tr key={session.id} className="hover:bg-gray-50">
                  <td className="px-4 py-3">
                    <div className="text-sm font-medium text-gray-900">
                      {session.property_id}
                    </div>
                    <div className="text-xs text-gray-500">ID: {session.id.slice(0, 8)}</div>
                  </td>
                  <td className="px-4 py-3">
                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(session.session_status)}`}>
                      {session.session_status}
                    </span>
                  </td>
                  <td className="px-4 py-3">
                    <div className="text-sm font-medium text-gray-900">
                      ₹{session.current_highest_bid.toLocaleString()}
                    </div>
                  </td>
                  <td className="px-4 py-3">
                    <div className="text-sm text-gray-900">
                      {session.registered_bidders}
                    </div>
                  </td>
                  <td className="px-4 py-3">
                    <div className="text-sm text-gray-900">
                      {new Date(session.end_time).toLocaleString()}
                    </div>
                  </td>
                  <td className="px-4 py-3">
                    <div className="flex items-center space-x-2">
                      {session.session_status === 'scheduled' && (
                        <button
                          onClick={() => handleStartAuction(session.id)}
                          className="text-white bg-green-500 hover:bg-green-600 px-2 py-1 text-xs rounded transition-colors"
                        >
                          Start
                        </button>
                      )}
                      {session.session_status === 'active' && (
                        <>
                          <button
                            onClick={() => handlePauseAuction(session.id)}
                            className="text-white bg-yellow-500 hover:bg-yellow-600 px-2 py-1 text-xs rounded transition-colors"
                          >
                            Pause
                          </button>
                          <button
                            onClick={() => handleExtendAuction(session.id)}
                            className="text-white bg-blue-500 hover:bg-blue-600 px-2 py-1 text-xs rounded transition-colors"
                          >
                            Extend
                          </button>
                        </>
                      )}
                      {(session.session_status === 'scheduled' || session.session_status === 'active') && (
                        <button
                          onClick={() => handleCancelAuction(session.id)}
                          className="text-white bg-red-500 hover:bg-red-600 px-2 py-1 text-xs rounded transition-colors"
                        >
                          Cancel
                        </button>
                      )}
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );
}