import React, { useState, useEffect } from 'react';
import { auditService, AuditLogEntry, AuditLogFilter, UserActivitySummary } from '../../services/auditService';

interface AuditDashboardProps {
  className?: string;
}

export const AuditDashboard: React.FC<AuditDashboardProps> = ({ className = '' }) => {
  const [auditLogs, setAuditLogs] = useState<AuditLogEntry[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [filters, setFilters] = useState<AuditLogFilter>({
    limit: 25,
    offset: 0
  });
  const [totalRecords, setTotalRecords] = useState(0);
  const [selectedUserId, setSelectedUserId] = useState<number | null>(null);
  const [userActivity, setUserActivity] = useState<UserActivitySummary | null>(null);
  const [securityStatus, setSecurityStatus] = useState<any>(null);
  const [suspiciousActivities, setSuspiciousActivities] = useState<any[]>([]);

  // Load audit logs
  const loadAuditLogs = async () => {
    setLoading(true);
    setError(null);
    try {
      const logs = await auditService.getAuditLogs(filters);
      setAuditLogs(logs);
      // Estimate total records (in real app, this should come from API)
      setTotalRecords(logs.length === 25 ? (filters.offset || 0) + 50 : (filters.offset || 0) + logs.length);
    } catch (err) {
      setError('Failed to load audit logs');
      console.error('Error loading audit logs:', err);
    } finally {
      setLoading(false);
    }
  };

  // Load security status
  const loadSecurityStatus = async () => {
    try {
      const status = await auditService.getSecurityStatus();
      setSecurityStatus(status);
    } catch (err) {
      console.error('Error loading security status:', err);
    }
  };

  // Load suspicious activities
  const loadSuspiciousActivities = async () => {
    try {
      const activities = await auditService.getSuspiciousActivities();
      setSuspiciousActivities(activities.alerts || []);
    } catch (err) {
      console.error('Error loading suspicious activities:', err);
    }
  };

  // Load user activity summary
  const loadUserActivity = async (userId: number) => {
    try {
      const activity = await auditService.getUserActivitySummary(userId, 30);
      setUserActivity(activity);
    } catch (err) {
      console.error('Error loading user activity:', err);
    }
  };

  // Seed audit data
  const handleSeedData = async () => {
    try {
      await auditService.seedAuditData();
      await loadAuditLogs();
      alert('Audit data seeded successfully!');
    } catch (err) {
      console.error('Error seeding data:', err);
      alert('Failed to seed audit data');
    }
  };

  // Export to CSV
  const handleExportCSV = async () => {
    try {
      // Create CSV content from current audit logs
      const csvHeaders = ['Log ID', 'User/Admin', 'Action', 'Entity Affected', 'Status', 'IP Address', 'Timestamp'];
      const csvRows = auditLogs.map(log => [
        log.id,
        log.user_id || 'Anonymous',
        log.operation,
        log.resource_type,
        log.success ? 'Success' : 'Failed',
        log.ip_address || 'Unknown',
        formatDate(log.created_at)
      ]);
      
      const csvContent = [csvHeaders, ...csvRows]
        .map(row => row.map(field => `"${field}"`).join(','))
        .join('\n');
      
      // Download CSV file
      const blob = new Blob([csvContent], { type: 'text/csv' });
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `audit-logs-${new Date().toISOString().split('T')[0]}.csv`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      window.URL.revokeObjectURL(url);
    } catch (err) {
      console.error('Error exporting to CSV:', err);
      alert('Failed to export to CSV');
    }
  };

  // Clear all audit logs
  const handleClearAll = async () => {
    if (window.confirm('Are you sure you want to delete all audit logs? This action cannot be undone.')) {
      // Optimistic update - clear UI immediately
      setAuditLogs([]);
      setTotalRecords(0);
      
      try {
        await auditService.clearAllAuditLogs();
        alert('All audit logs cleared successfully!');
      } catch (err) {
        console.error('Error clearing audit logs:', err);
        alert('Failed to clear audit logs');
        // Reload data on error to restore state
        await loadAuditLogs();
      }
    }
  };

  useEffect(() => {
    loadAuditLogs();
    loadSecurityStatus();
    loadSuspiciousActivities();
  }, [filters]);

  useEffect(() => {
    if (selectedUserId) {
      loadUserActivity(selectedUserId);
    }
  }, [selectedUserId]);

  const handleFilterChange = (key: keyof AuditLogFilter, value: any) => {
    setFilters(prev => ({
      ...prev,
      [key]: value,
      offset: 0 // Reset pagination when filters change
    }));
  };

  const handlePageChange = (direction: 'next' | 'prev' | number) => {
    if (typeof direction === 'number') {
      setFilters(prev => ({
        ...prev,
        offset: direction * (prev.limit || 25)
      }));
    } else {
      setFilters(prev => ({
        ...prev,
        offset: direction === 'next' 
          ? (prev.offset || 0) + (prev.limit || 25)
          : Math.max(0, (prev.offset || 0) - (prev.limit || 25))
      }));
    }
  };

  const getCurrentPage = () => Math.floor((filters.offset || 0) / (filters.limit || 25));
  const getTotalPages = () => Math.ceil(totalRecords / (filters.limit || 25));
  
  const getVisiblePages = () => {
    const currentPage = getCurrentPage();
    const totalPages = getTotalPages();
    const pages = [];
    
    // Show previous page if exists
    if (currentPage > 0) {
      pages.push(currentPage - 1);
    }
    
    // Show current page
    pages.push(currentPage);
    
    // Show next page if exists
    if (currentPage < totalPages - 1) {
      pages.push(currentPage + 1);
    }
    
    return pages;
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString();
  };

  const getStatusColor = (success: boolean) => {
    return success ? 'text-green-600' : 'text-red-600';
  };

  const getOperationColor = (operation: string) => {
    const colors: Record<string, string> = {
      'USER_LOGIN': 'bg-blue-100 text-blue-800',
      'USER_LOGOUT': 'bg-gray-100 text-gray-800',
      'DOCUMENT_UPLOAD': 'bg-green-100 text-green-800',
      'DOCUMENT_DELETE': 'bg-red-100 text-red-800',
      'PROPERTY_CREATE': 'bg-purple-100 text-purple-800',
      'ADMIN_ACCESS': 'bg-yellow-100 text-yellow-800',
      'PAYMENT_INITIATE': 'bg-indigo-100 text-indigo-800'
    };
    return colors[operation] || 'bg-gray-100 text-gray-800';
  };

  return (
    <div className={`audit-dashboard ${className}`}>
      <div className="space-y-8 p-6">


        {/* Filters */}
        <div className="bg-white rounded-lg shadow-lg p-6">
          <h2 className="text-2xl font-bold mb-6 text-black">Audit Logs</h2>
          
          <div className="mb-6 grid grid-cols-1 md:grid-cols-4 gap-4">
            <div>
              <label className="block text-sm font-medium text-black mb-1">
                Operation
              </label>
              <select
                value={filters.operation || ''}
                onChange={(e) => handleFilterChange('operation', e.target.value || undefined)}
                className="w-full border border-gray-300 rounded-md px-3 py-2 text-black bg-white"
              >
                <option value="" className="text-black">All Operations</option>
                <option value="USER_LOGIN" className="text-black">User Login</option>
                <option value="USER_LOGOUT" className="text-black">User Logout</option>
                <option value="DOCUMENT_UPLOAD" className="text-black">Document Upload</option>
                <option value="DOCUMENT_DELETE" className="text-black">Document Delete</option>
                <option value="PROPERTY_CREATE" className="text-black">Property Create</option>
                <option value="ADMIN_ACCESS" className="text-black">Admin Access</option>
                <option value="PAYMENT_INITIATE" className="text-black">Payment Initiate</option>
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-black mb-1">
                Resource Type
              </label>
              <select
                value={filters.resource_type || ''}
                onChange={(e) => handleFilterChange('resource_type', e.target.value || undefined)}
                className="w-full border border-gray-300 rounded-md px-3 py-2 text-black bg-white"
              >
                <option value="" className="text-black">All Resources</option>
                <option value="AUTH" className="text-black">Authentication</option>
                <option value="DOCUMENT" className="text-black">Document</option>
                <option value="PROPERTY" className="text-black">Property</option>
                <option value="ADMIN" className="text-black">Admin</option>
                <option value="PAYMENT" className="text-black">Payment</option>
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-black mb-1">
                Success Status
              </label>
              <select
                value={filters.success === undefined ? '' : filters.success.toString()}
                onChange={(e) => handleFilterChange('success', e.target.value === '' ? undefined : e.target.value === 'true')}
                className="w-full border border-gray-300 rounded-md px-3 py-2 text-black bg-white"
              >
                <option value="" className="text-black">All</option>
                <option value="true" className="text-black">Success</option>
                <option value="false" className="text-black">Failed</option>
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-black mb-1">
                User ID
              </label>
              <input
                type="number"
                value={filters.user_id || ''}
                onChange={(e) => handleFilterChange('user_id', e.target.value ? parseInt(e.target.value) : undefined)}
                className="w-full border border-gray-300 rounded-md px-3 py-2 text-black bg-white"
                placeholder="Filter by user ID"
              />
            </div>
          </div>

          {/* Action Buttons */}
          <div className="mb-6 flex gap-4">
            <button
              onClick={loadAuditLogs}
              disabled={loading}
              className="bg-white border-2 border-blue-600 text-black px-4 py-2 rounded-md hover:bg-blue-50 disabled:opacity-50 font-medium"
            >
              {loading ? 'Loading...' : 'Refresh'}
            </button>

            <button
              onClick={handleExportCSV}
              className="bg-white border-2 border-green-600 text-black px-4 py-2 rounded-md hover:bg-green-50 font-medium"
            >
              Export CSV
            </button>

            <button
              onClick={handleClearAll}
              className="bg-white border-2 border-red-600 text-black px-4 py-2 rounded-md hover:bg-red-50 font-medium"
            >
              Clear All
            </button>
          </div>

          {error && (
            <div className="mb-4 p-4 bg-red-100 border border-red-400 text-red-700 rounded">
              {error}
            </div>
          )}

          {/* Audit Logs Table */}
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Log ID
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    User/Admin
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Action
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Entity Affected
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    IP Address
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Timestamp
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {auditLogs.map((log) => (
                  <tr key={log.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      #{log.id}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {log.user_id ? (
                        <button
                          onClick={() => setSelectedUserId(log.user_id!)}
                          className="text-blue-600 hover:text-blue-800 underline font-normal"
                        >
                          {typeof log.user_id === 'string' && log.user_id.length > 8 
                            ? `${log.user_id.substring(0, 8)}...` 
                            : log.user_id}
                        </button>
                      ) : (
                        'Anonymous'
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`inline-flex px-2 py-1 text-xs rounded-full ${getOperationColor(log.operation)}`}>
                        {log.operation}
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {log.resource_type}
                      {log.resource_id && (
                        <div className="text-xs text-gray-500">ID: {log.resource_id}</div>
                      )}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`text-sm ${getStatusColor(log.success)}`}>
                        {log.success ? 'Success' : 'Failed'}
                      </span>
                      <div className="text-xs text-gray-500">{log.response_status}</div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {log.ip_address || 'Unknown'}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {formatDate(log.created_at)}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>

          {/* Pagination */}
          <div className="mt-6 flex justify-between items-center">
            <div className="text-sm text-gray-700">
              Showing {(filters.offset || 0) + 1} to {Math.min((filters.offset || 0) + auditLogs.length, totalRecords)} of {totalRecords} entries
            </div>
            <div className="flex items-center gap-2">
              <button
                onClick={() => handlePageChange('prev')}
                disabled={!filters.offset || filters.offset === 0}
                className="px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 disabled:opacity-50"
              >
                Previous
              </button>
              
              <div className="flex gap-1">
                {getVisiblePages().map((pageNum) => {
                  const isCurrentPage = pageNum === getCurrentPage();
                  return (
                    <button
                      key={pageNum}
                      onClick={() => handlePageChange(pageNum)}
                      className={`w-10 h-10 rounded-full text-sm font-medium transition-colors ${
                        isCurrentPage
                          ? 'bg-blue-600 text-white'
                          : 'bg-white border border-gray-300 text-gray-700 hover:bg-gray-50'
                      }`}
                    >
                      {pageNum + 1}
                    </button>
                  );
                })}
              </div>
              
              <button
                onClick={() => handlePageChange('next')}
                disabled={auditLogs.length < (filters.limit || 25)}
                className="px-3 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 disabled:opacity-50"
              >
                Next
              </button>
            </div>
          </div>
        </div>

        {/* User Activity Summary Modal */}
        {userActivity && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 max-w-2xl w-full mx-4 max-h-96 overflow-y-auto">
              <div className="flex justify-between items-center mb-4">
                <h3 className="text-lg font-semibold">
                  User Activity Summary - User ID: {userActivity.user_id}
                </h3>
                <button
                  onClick={() => setUserActivity(null)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  ✕
                </button>
              </div>
              
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                <div className="text-center">
                  <div className="text-2xl font-bold text-blue-600">{userActivity.total_operations}</div>
                  <div className="text-sm text-gray-600">Total Operations</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-green-600">{userActivity.total_successful}</div>
                  <div className="text-sm text-gray-600">Successful</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-red-600">{userActivity.total_failed}</div>
                  <div className="text-sm text-gray-600">Failed</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-purple-600">{userActivity.success_rate.toFixed(1)}%</div>
                  <div className="text-sm text-gray-600">Success Rate</div>
                </div>
              </div>

              <div className="space-y-2">
                <h4 className="font-medium">Operations Breakdown:</h4>
                {userActivity.operations.map((op, index) => (
                  <div key={index} className="flex justify-between items-center p-2 bg-gray-50 rounded">
                    <span className="font-medium">{op.operation}</span>
                    <div className="text-sm text-gray-600">
                      {op.count} total, {op.successful} success, {op.avg_execution_time.toFixed(0)}ms avg
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};