import React, { useState, useRef, useEffect } from 'react';
import { Upload, Building, FileText, CheckCircle, AlertCircle, Download, Eye, EyeOff, Trash2 } from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Progress } from '../ui/progress';
import { createProperty } from '../../api/properties';
import { parseCSV } from '../../utils/csvParser';
import { parseExcel as parseExcelFile } from '../../utils/excelParser';

interface BulkPropertyData {
  fullname: string;
  company?: string;
  property_manager_contact: string;
  email: string;
  title: string;
  description?: string;
  listing_type: string;
  property_type: string;
  built_up_area: number;
  carpet_area?: number;
  floor?: string;
  total_floors?: string;
  bedrooms: number;
  bathrooms: number;
  balconies?: number;
  facing?: string;
  furnishing?: string;
  property_age?: string;
  amenities?: string[];
  sale_price?: number;
  monthly_rent?: number;
  status?: string;
  address: string;
  city: string;
  locality?: string;
  state: string;
  pincode: string;
  landmark?: string;
  latitude?: number;
  longitude?: number;
  nearby_places?: string;
  image_urls?: string[];
  virtual_tour_url?: string;
  listed_by_id: string;
}

interface ParsedRow extends BulkPropertyData {
  isValid: boolean;
  errors: string[];
}

const BulkInventoryUpload: React.FC = () => {
  const [file, setFile] = useState<File | null>(null);
  const [parsedData, setParsedData] = useState<ParsedRow[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [showPropertyList, setShowPropertyList] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isDragOver, setIsDragOver] = useState(false);
  const [currentUserId, setCurrentUserId] = useState<string>('');
  const fileInputRef = useRef<HTMLInputElement>(null);

  useEffect(() => {
    // Get user ID from local storage
    try {
      const userStr = localStorage.getItem('user');
      if (userStr) {
        const user = JSON.parse(userStr);
        if (user && user.id) {
          setCurrentUserId(String(user.id));
        }
      }
      // If no valid user ID, use a default UUID format
      if (!currentUserId) {
        setCurrentUserId('550e8400-e29b-41d4-a716-446655440000'); // Default UUID
      }
    } catch (e) {
      console.error('Failed to get user ID from local storage', e);
      setCurrentUserId('550e8400-e29b-41d4-a716-446655440000'); // Default UUID
    }
  }, []);

  // Helper function to get value from row with multiple possible keys
  const getValue = (row: any, keys: string[]) => {
    for (const key of keys) {
      if (row[key] !== undefined && row[key] !== null && row[key] !== '') {
        return String(row[key]).trim();
      }
    }
    return '';
  };

  const processRowData = (row: any, index: number): ParsedRow => {
    console.log(`Processing row ${index}:`, row);
    
    // Normalize basic strings
    const listingType = (getValue(row, ['Listing Type', 'listing_type', 'LISTING_TYPE', 'Type', 'ListingType', 'Sale/Rent']) || 'sell').toLowerCase();
    const propertyType = (getValue(row, ['Property Type', 'property_type', 'PROPERTY_TYPE', 'PropertyType', 'Category']) || 'apartment')
      .toLowerCase()
      .replace('pg/co-living', 'pg-coliving')
      .replace('/', '-');

    const property: any = {
      // Basic Info - Keys match database columns where possible
      fullname: getValue(row, ['Full Name', 'Owner Name', 'owner_name', 'OWNER_NAME', 'Name', 'OwnerName', 'Owner']),
      company: getValue(row, ['Company Name', 'Company', 'company', 'COMPANY', 'Agency', 'Broker']),
      property_manager_contact: getValue(row, ['Mobile Number', 'Contact', 'contact', 'CONTACT', 'Phone', 'Mobile', 'Contact Number']),
      email: getValue(row, ['Email Address', 'Email', 'email', 'EMAIL', 'E-mail']),
      
      // Property Details
      title: getValue(row, ['Property Title', 'Title', 'title', 'TITLE', 'Name', 'Property Name']),
      description: getValue(row, ['Property Description', 'Description', 'description', 'DESCRIPTION', 'Details']),
      listing_type: listingType,
      property_type: propertyType,
      
      // Specifications
      built_up_area: parseFloat(getValue(row, ['Built-up Area (sq ft)', 'Built Up Area', 'built_up_area', 'BUILT_UP_AREA', 'Area', 'Size', 'Built Area'])) || 0,
      carpet_area: parseFloat(getValue(row, ['Carpet Area (sq ft)', 'Carpet Area', 'carpet_area', 'CARPET_AREA', 'CarpetArea'])) || 0,
      floor_number: getValue(row, ['Floor Number', 'Floor', 'floor', 'FLOOR']),
      total_floors: getValue(row, ['Total Floors', 'total_floors', 'TOTAL_FLOORS']),
      bedrooms: parseInt(getValue(row, ['Bedrooms', 'bedrooms', 'BEDROOMS', 'BHK', 'Beds', 'Bedroom'])) || 0,
      bathrooms: parseInt(getValue(row, ['Bathrooms', 'bathrooms', 'BATHROOMS', 'Baths', 'Bathroom'])) || 0,
      balconies: parseInt(getValue(row, ['Balconies', 'balconies', 'BALCONIES', 'Balcony'])) || 0,
      facing: getValue(row, ['Facing', 'facing', 'FACING', 'Direction']).toLowerCase(),
      furnishing: (getValue(row, ['Furnishing', 'furnishing', 'FURNISHING', 'Furnished', 'Furniture']) || 'unfurnished').toLowerCase().replace(' ', '-'),
      property_age: getValue(row, ['Property Age', 'Age', 'age', 'AGE', 'Building Age']),
      amenities: (getValue(row, ['Amenities', 'amenities', 'AMENITIES', 'Features']) || '').split(',').map((a: string) => a.trim()).filter((a: string) => a),
      
      // Pricing Placeholders
      sale_price: 0,
      monthly_rent: 0,
      status: getValue(row, ['Possession Status', 'Status', 'status', 'STATUS', 'Availability']),
      
      // Location
      address: getValue(row, ['Property Address', 'Address', 'address', 'ADDRESS', 'Location', 'Full Address']),
      city: getValue(row, ['City', 'city', 'CITY']),
      locality: getValue(row, ['Locality', 'locality', 'LOCALITY', 'Area', 'Neighborhood']),
      state: getValue(row, ['State', 'state', 'STATE']) || 'Maharashtra',
      pincode: getValue(row, ['Pincode', 'pincode', 'PINCODE', 'Pin Code', 'ZIP', 'Postal Code']),
      landmark: getValue(row, ['Nearby Landmark', 'Landmark', 'landmark', 'LANDMARK', 'Near']),
      latitude: parseFloat(getValue(row, ['Latitude', 'latitude', 'LATITUDE', 'Lat'])) || 0,
      longitude: parseFloat(getValue(row, ['Longitude', 'longitude', 'LONGITUDE', 'Lng', 'Long'])) || 0,
      nearby_places: getValue(row, ['Nearby Places', 'Nearby', 'nearby', 'NEARBY', 'Connectivity']),
      
      // Media
      image_urls: (getValue(row, ['Property Images', 'Images', 'images', 'IMAGES', 'Photos', 'Image URL']) || '').split(',').map((u: string) => u.trim()).filter((u: string) => u),
      virtual_tour_url: getValue(row, ['Property Videos', 'Videos', 'videos', 'VIDEOS', 'Video URL']),
      
      // User ID - Must be valid UUID format
      listed_by_id: currentUserId || '550e8400-e29b-41d4-a716-446655440000'
    };
    
    // Parse price
    const priceText = getValue(row, ['Sale/Rent Price (₹)', 'Sale Price', 'sale_price', 'SALE_PRICE', 'Price', 'SalePrice', 'Cost']);
    if (priceText) {
      // Remove currency symbols
      let cleanPrice = priceText.replace(/[₹,₹]/g, '').trim();
      
      if (cleanPrice.includes('/month') || cleanPrice.includes('/mo')) {
        cleanPrice = cleanPrice.replace(/\/month|\/mo/g, '').trim();
        property.monthly_rent = parseFloat(cleanPrice) || 0;
        if (!property.listing_type) property.listing_type = 'rent';
      } else if (cleanPrice.includes('Lakhs') || cleanPrice.includes('L')) {
        cleanPrice = cleanPrice.replace(/Lakhs|L/g, '').trim();
        property.sale_price = (parseFloat(cleanPrice) || 0) * 100000;
      } else if (cleanPrice.includes('Cr') || cleanPrice.includes('Crores')) {
        cleanPrice = cleanPrice.replace(/Cr|Crores/g, '').trim();
        property.sale_price = (parseFloat(cleanPrice) || 0) * 10000000;
      } else {
        const numPrice = parseFloat(cleanPrice) || 0;
        if (property.listing_type === 'rent') {
          property.monthly_rent = numPrice;
        } else {
          property.sale_price = numPrice;
        }
      }
    }
    
    // Status Normalization
    const statusMap: { [key: string]: string } = {
      'under construction': 'under_review',
      'ready to move': 'active',
      'available immediately': 'active',
      'ready': 'active',
      'available': 'active',
      'sold': 'sold',
      'rented': 'rented'
    };
    if (property.status) {
      property.status = statusMap[property.status.toLowerCase()] || 'active';
    } else {
      property.status = 'active';
    }
    
    // Set defaults for required fields only if missing
    if (!property.title) property.title = `Property in ${property.city || 'Mumbai'}`;
    if (!property.fullname) property.fullname = 'Property Owner';
    if (!property.property_manager_contact) property.property_manager_contact = '9999999999';
    if (!property.email) property.email = 'owner@example.com';
    if (!property.address) property.address = `Property Address in ${property.city || 'Mumbai'}`;
    if (!property.city) property.city = 'Mumbai';
    if (!property.state) property.state = 'Maharashtra';
    
    // Validation for required fields
    const errors: string[] = [];
    if (!property.title) errors.push('Missing Property Title');
    if (!property.address) errors.push('Missing Address');
    if (!property.fullname) errors.push('Missing Owner Name');
    if (!property.property_manager_contact) errors.push('Missing Contact Number');
    if (!property.email) errors.push('Missing Email');
    
    // Validate pricing based on listing type
    if (property.listing_type === 'rent' && (!property.monthly_rent || property.monthly_rent <= 0)) {
      errors.push('Missing Rent Price');
    }
    if ((property.listing_type === 'sell' || property.listing_type === 'bidding') && (!property.sale_price || property.sale_price <= 0)) {
      errors.push('Missing Sale Price');
    }

    return {
      ...property,
      isValid: errors.length === 0,
      errors
    };
  };

  const validateFile = (file: File): boolean => {
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    const maxSize = 10 * 1024 * 1024; // 10MB
    
    if (!['csv', 'xlsx'].includes(fileExtension || '')) {
      alert('Please select a CSV or Excel (.xlsx) file');
      return false;
    }
    
    if (file.size > maxSize) {
      alert('File size must be less than 10MB');
      return false;
    }
    
    return true;
  };

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = event.target.files?.[0];
    if (selectedFile && validateFile(selectedFile)) {
      setFile(selectedFile);
      parseFile(selectedFile);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(false);
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(false);
    
    const droppedFile = e.dataTransfer.files[0];
    if (droppedFile && validateFile(droppedFile)) {
      setFile(droppedFile);
      parseFile(droppedFile);
    }
  };

  const clearFile = () => {
    setFile(null);
    setParsedData([]);
    setShowPropertyList(false);
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const parseFile = async (file: File) => {
    setIsLoading(true);
    setUploadProgress(10);
    
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    
    try {
      if (fileExtension === 'csv') {
        await parseCSVFile(file);
      } else if (fileExtension === 'xlsx') {
        await parseExcel(file);
      }
      setUploadProgress(100);
    } catch (error) {
      console.error('File parsing error:', error);
      alert('Error parsing file. Please check the file format.');
    } finally {
      setIsLoading(false);
      setTimeout(() => setUploadProgress(0), 1000);
    }
  };

  const parseCSVFile = (file: File): Promise<void> => {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      
      reader.onload = (e) => {
        try {
          setUploadProgress(30);
          const text = e.target?.result as string;
          const result = parseCSV(text);
          setUploadProgress(60);
      
          const data: ParsedRow[] = result.data.map((row, index) => processRowData(row, index));
          
          setParsedData(data);
          setUploadProgress(90);
          resolve();
        } catch (error) {
          console.error('CSV parsing error:', error);
          reject(error);
        }
      };
      
      reader.onerror = () => reject(new Error('Failed to read file'));
      reader.readAsText(file);
    });
  };

  const parseExcel = async (file: File) => {
    try {
      setUploadProgress(30);
      const result = await parseExcelFile(file);
      setUploadProgress(60);
      
      const data: ParsedRow[] = result.data.map((row, index) => processRowData(row, index));
      
      setParsedData(data);
      setUploadProgress(90);
    } catch (error) {
      console.error('Excel parsing error:', error);
      throw error;
    }
  };

  const handleSubmit = async () => {
    if (!file) {
      alert('Please select a file first');
      return;
    }

    const validProperties = parsedData.filter(row => row.isValid);
    if (validProperties.length === 0) {
      alert('No valid properties to upload. Please check your data.');
      return;
    }

    const invalidProperties = parsedData.filter(row => !row.isValid);
    if (invalidProperties.length > 0) {
      const proceed = confirm(`Found ${invalidProperties.length} invalid properties. Do you want to upload only the ${validProperties.length} valid properties?`);
      if (!proceed) return;
    }

    setIsLoading(true);
    
    let created_count = 0;
    let failed_count = 0;
    const errors: string[] = [];
    const created_properties: any[] = [];
    
    try {
      console.log('Starting individual property creation for', validProperties.length, 'properties');
      
      for (let i = 0; i < validProperties.length; i++) {
        const property = validProperties[i];
        
        try {
          console.log(`Creating property ${i + 1}/${validProperties.length}: ${property.title}`);
          
          // Transform property data to match PropertyCreate model
          const propertyData = {
            title: property.title,
            description: property.description || '',
            listing_type: property.listing_type,
            property_type: property.property_type,
            status: 'pending',
            address: property.address,
            city: property.city,
            locality: property.locality || '',
            state: property.state || 'Maharashtra',
            pincode: property.pincode || '',
            landmark: property.landmark || '',
            latitude: property.latitude || null,
            longitude: property.longitude || null,
            built_up_area: property.built_up_area || null,
            carpet_area: property.carpet_area || null,
            bedrooms: property.bedrooms || null,
            bathrooms: property.bathrooms || null,
            balconies: property.balconies || null,
            floor_number: property.floor_number || null,
            total_floors: property.total_floors || null,
            facing: property.facing || null,
            furnishing: property.furnishing || 'unfurnished',
            property_age: property.property_age || null,
            sale_price: property.sale_price || null,
            monthly_rent: property.monthly_rent || null,
            security_deposit: null,
            price_negotiable: false,
            loan_available: false,
            possession_status: null,
            amenities: property.amenities || [],
            custom_amenities: [],
            property_highlights: [],
            nearby_places: property.nearby_places ? property.nearby_places.split(',').map(p => p.trim()) : [],
            primary_image: null,
            virtual_tour_url: property.virtual_tour_url || null,
            image_urls: property.image_urls || [],
            views: 0,
            inquiries: 0,
            property_manager_contact: property.property_manager_contact,
            email: property.email,
            fullname: property.fullname,
            contact_type: 'owner',
            listed_by_id: property.listed_by_id || '550e8400-e29b-41d4-a716-446655440000'
          };
          
          console.log('Sending property data with UUID:', propertyData.listed_by_id);
          
          const createdProperty = await createProperty(propertyData);
          console.log('Property created successfully:', createdProperty.id);
          
          created_properties.push(createdProperty);
          created_count++;
          
        } catch (error: any) {
          console.error(`Failed to create property ${i + 1}:`, error);
          console.error('Full error details:', error.response || error);
          const errorMsg = error.response?.data?.message || error.message || 'Unknown error';
          errors.push(`Property ${i + 1} (${property.title}): ${errorMsg}`);
          failed_count++;
        }
      }
      
      console.log(`Upload completed: ${created_count} created, ${failed_count} failed`);
      
      if (created_count > 0) {
        alert(`✅ Successfully uploaded ${created_count} properties!`);
        clearFile();
        setTimeout(() => window.location.reload(), 2000);
      } else {
        alert(`❌ No properties were created. Check console for detailed errors.`);
        console.error('All properties failed. Detailed errors:', errors);
        if (errors.length > 0) {
          console.error('First error details:', errors[0]);
        }
      }
      
    } catch (error: any) {
      console.error('Upload process failed:', error);
      alert(`❌ Upload failed: ${error.message}`);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl shadow-sm p-6 border border-blue-100">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <div className="bg-[#0056D2] p-3 rounded-lg">
              <Upload className="w-8 h-8 text-white" />
            </div>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">Bulk Inventory Upload</h1>
              <p className="text-gray-600 mt-1">Upload CSV or Excel files to automatically add properties to database</p>
              <p className="text-sm text-blue-600 mt-1">🚀 Properties will be saved directly to database like individual property submissions</p>
            </div>
          </div>
          <div className="text-right">
            <p className="text-sm text-gray-500">Supported formats</p>
            <div className="flex space-x-2 mt-1">
              <span className="bg-green-100 text-green-800 px-2 py-1 rounded text-xs font-medium">CSV</span>
              <span className="bg-blue-100 text-blue-800 px-2 py-1 rounded text-xs font-medium">XLSX</span>
            </div>
          </div>
        </div>
      </div>

      {/* Sample Format Info */}
      <div className="bg-gradient-to-r from-yellow-50 to-orange-50 border border-yellow-200 rounded-lg p-4">
        <div className="flex items-start space-x-3">
          <div className="bg-yellow-100 p-2 rounded-lg">
            <FileText className="w-5 h-5 text-yellow-600" />
          </div>
          <div className="flex-1">
            <h3 className="text-sm font-medium text-yellow-800 mb-2">📋 Supported Excel/CSV Columns (case-insensitive):</h3>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-3 text-xs text-yellow-700">
              <div className="bg-white bg-opacity-60 p-2 rounded">
                <strong>👤 Owner Info:</strong> Owner Name, Company, Contact, Email
              </div>
              <div className="bg-white bg-opacity-60 p-2 rounded">
                <strong>🏠 Property:</strong> Title, Description, Listing Type, Property Type, Built Up Area, Carpet Area, Floor, Bedrooms, Bathrooms, Balconies, Facing, Furnishing, Age, Amenities
              </div>
              <div className="bg-white bg-opacity-60 p-2 rounded">
                <strong>📍 Location & Media:</strong> Address, City, Locality, State, Pincode, Landmark, Latitude, Longitude, Nearby, Images, Videos
              </div>
            </div>
            <div className="mt-3 p-2 bg-red-50 border border-red-200 rounded">
              <p className="text-xs text-red-700">
                <strong>⚠️ Required Fields:</strong> Title, Address, Price/Rent, Owner Name, Contact
              </p>
            </div>
            <div className="mt-2 p-2 bg-green-50 border border-green-200 rounded">
              <p className="text-xs text-green-700">
                <strong>✅ Auto-Upload:</strong> Click "Upload Properties" to automatically save all valid properties to database
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* File Upload */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Upload File</h2>
              <p className="text-sm text-gray-500 mt-1">Maximum file size: 10MB</p>
            </div>
            {file && (
              <div className="flex space-x-2">
                <Button
                  onClick={() => setShowPropertyList(!showPropertyList)}
                  variant="outline"
                  className="flex items-center space-x-2 hover:bg-blue-50"
                >
                  {showPropertyList ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                  <span>{showPropertyList ? 'Hide' : 'Preview'} ({parsedData.length})</span>
                </Button>
                <Button
                  onClick={clearFile}
                  variant="outline"
                  className="flex items-center space-x-2 text-red-600 hover:bg-red-50 hover:border-red-200"
                >
                  <Trash2 className="w-4 h-4" />
                  <span>Clear</span>
                </Button>
              </div>
            )}
          </div>
        </div>
        
        <div className="p-6">
          {!file ? (
            <div 
              className={`border-2 border-dashed rounded-xl p-12 text-center transition-all duration-200 ${
                isDragOver 
                  ? 'border-blue-400 bg-blue-50' 
                  : 'border-gray-300 hover:border-gray-400 hover:bg-gray-50'
              }`}
              onDragOver={handleDragOver}
              onDragLeave={handleDragLeave}
              onDrop={handleDrop}
            >
              <div className="space-y-4">
                <div className={`mx-auto w-16 h-16 rounded-full flex items-center justify-center ${
                  isDragOver ? 'bg-blue-100' : 'bg-gray-100'
                }`}>
                  <Upload className={`w-8 h-8 ${
                    isDragOver ? 'text-blue-600' : 'text-gray-400'
                  }`} />
                </div>
                <div>
                  <p className="text-xl font-medium text-gray-900 mb-2">
                    {isDragOver ? 'Drop your file here' : 'Drag & drop your file here'}
                  </p>
                  <p className="text-gray-500 mb-4">or</p>
                  <Button 
                    onClick={() => fileInputRef.current?.click()}
                    className="bg-[#0056D2] hover:bg-[#0041A3] text-white px-6 py-2"
                  >
                    Choose File
                  </Button>
                </div>
                <div className="text-sm text-gray-500 space-y-1">
                  <p>Supports CSV and Excel (.xlsx) files</p>
                </div>
              </div>
              <Input
                ref={fileInputRef}
                type="file"
                accept=".csv,.xlsx"
                onChange={handleFileChange}
                className="hidden"
              />
            </div>
          ) : (
            <div className="space-y-4">
              {/* File Info */}
              <div className="bg-gray-50 rounded-lg p-4 border">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-3">
                    <div className="bg-blue-100 p-2 rounded">
                      <FileText className="w-5 h-5 text-blue-600" />
                    </div>
                    <div>
                      <p className="font-medium text-gray-900">{file.name}</p>
                      <p className="text-sm text-gray-500">
                        {(file.size / 1024 / 1024).toFixed(2)} MB • {file.type || 'Unknown type'}
                      </p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-2">
                    {parsedData.length > 0 && (
                      <div className="flex items-center space-x-2">
                        <CheckCircle className="w-5 h-5 text-green-500" />
                        <span className="text-sm font-medium text-green-700">
                          {parsedData.filter(p => p.isValid).length} valid properties
                        </span>
                      </div>
                    )}
                  </div>
                </div>
              </div>
              
              {/* Progress Bar */}
              {(isLoading || uploadProgress > 0) && (
                <div className="space-y-2">
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-600">
                      {uploadProgress === 0 && isLoading ? 'Uploading to database...' : 'Processing file...'}
                    </span>
                    <span className="text-gray-600">{uploadProgress}%</span>
                  </div>
                  <Progress value={uploadProgress} className="h-2" />
                  {uploadProgress === 0 && isLoading && (
                    <p className="text-xs text-blue-600 animate-pulse">
                      🔄 Saving properties to database, please wait...
                    </p>
                  )}
                </div>
              )}
            </div>
          )}
        </div>
        
        {/* Submit Button */}
        {file && !isLoading && (
          <div className="p-6 border-t border-gray-200 bg-gray-50 rounded-b-xl flex justify-between items-center">
            <div className="text-sm text-gray-600">
              <span className="font-medium">{parsedData.filter(p => p.isValid).length}</span> valid properties ready to upload
            </div>
            <Button
              onClick={handleSubmit}
              className="bg-[#0056D2] hover:bg-[#0041A3] text-white px-8 py-3 font-semibold flex items-center space-x-2"
            >
              <Upload className="w-4 h-4" />
              <span>Upload Properties to Database</span>
            </Button>
          </div>
        )}
      </div>

      {/* Comprehensive Data Preview */}
      {parsedData.length > 0 && showPropertyList && (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200">
            <div className="flex items-center justify-between">
              <h2 className="text-xl font-semibold text-gray-900">
                Property Data Preview ({parsedData.length} properties)
              </h2>
              <div className="flex space-x-3">
                <div className="flex items-center space-x-2 bg-green-50 px-3 py-2 rounded-lg">
                  <CheckCircle className="w-4 h-4 text-green-600" />
                  <span className="text-sm font-medium text-green-800">
                    {parsedData.filter(p => p.isValid).length} Valid
                  </span>
                </div>
                <div className="flex items-center space-x-2 bg-red-50 px-3 py-2 rounded-lg">
                  <AlertCircle className="w-4 h-4 text-red-600" />
                  <span className="text-sm font-medium text-red-800">
                    {parsedData.filter(p => !p.isValid).length} Invalid
                  </span>
                </div>
              </div>
            </div>
          </div>
          
          <div className="p-6 overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200 text-xs">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Title</th>
                  <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Listing Type</th>
                  <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Property Type</th>
                  <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Price</th>
                  <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                  <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">City</th>
                  <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Validation</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {parsedData.map((property: any, index) => (
                  <tr key={index} className={`hover:bg-gray-50 ${property.isValid ? '' : 'bg-red-50'}`}>
                    <td className="px-3 py-2 text-xs text-gray-900">
                      <div className="max-w-40 truncate font-medium" title={property.title}>
                        {property.title || 'N/A'}
                      </div>
                    </td>
                    <td className="px-3 py-2 text-xs text-gray-900 capitalize">{property.listing_type || 'N/A'}</td>
                    <td className="px-3 py-2 text-xs text-gray-900 capitalize">{property.property_type || 'N/A'}</td>
                    <td className="px-3 py-2 text-xs text-gray-900 font-medium">
                      {(property.listing_type === 'sell' || property.listing_type === 'urgent sale' || property.listing_type === 'bidding') 
                        ? (property.sale_price ? `₹${(property.sale_price).toLocaleString()}` : '₹0')
                        : (property.monthly_rent ? `₹${property.monthly_rent.toLocaleString()}/mo` : '₹0')
                      }
                    </td>
                    <td className="px-3 py-2 text-xs text-gray-900 capitalize">{property.status || 'N/A'}</td>
                    <td className="px-3 py-2 text-xs text-gray-900">{property.city || 'N/A'}</td>
                    <td className="px-3 py-2 text-xs">
                      {property.isValid ? (
                        <span className="text-green-600 font-medium">Valid</span>
                      ) : (
                        <span className="text-red-600" title={property.errors.join(', ')}>
                          {property.errors.length} Errors
                        </span>
                      )}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}
    </div>
  );
};

export default BulkInventoryUpload;