import React, { useState, useRef } from 'react';
import { Upload, Building, FileText, CheckCircle, AlertCircle, Download, Eye, EyeOff, Trash2 } from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Progress } from '../ui/progress';
import { bulkUploadProperties, BulkPropertyUploadResponse, BulkPropertyData } from '../../api/properties';
import { parseCSV } from '../../utils/csvParser';
import { parseExcel as parseExcelFile } from '../../utils/excelParser';

interface ParsedRow extends BulkPropertyData {
  isValid: boolean;
  errors: string[];
  nearby_places?: any; // Allow array for backend
  nearby?: string; // For display
}

const BulkPropertyUpload: React.FC = () => {
  const [file, setFile] = useState<File | null>(null);
  const [parsedData, setParsedData] = useState<ParsedRow[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [showPropertyList, setShowPropertyList] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isDragOver, setIsDragOver] = useState(false);
  const [currentUserId, setCurrentUserId] = useState<string>('');
  const fileInputRef = useRef<HTMLInputElement>(null);

  React.useEffect(() => {
    // Get user ID from local storage
    try {
      const userStr = localStorage.getItem('user');
      if (userStr) {
        const user = JSON.parse(userStr);
        if (user && user.id) {
          setCurrentUserId(String(user.id));
        }
      }
    } catch (e) {
      console.error('Failed to get user ID from local storage', e);
    }
  }, []);

  // Helper function to get value from row with multiple possible keys
  const getValue = (row: any, keys: string[]) => {
    for (const key of keys) {
      if (row[key] !== undefined && row[key] !== null && row[key] !== '') {
        return String(row[key]).trim();
      }
    }
    return '';
  };

  const processRowData = (row: any, index: number): ParsedRow => {
    console.log(`Processing row ${index}:`, row);
    console.log('Available keys:', Object.keys(row));
    
    // Normalize basic strings
    const listingType = (getValue(row, ['Listing Type', 'listing_type', 'LISTING_TYPE', 'Type', 'ListingType', 'Sale/Rent']) || 'sell').toLowerCase();
    const propertyType = (getValue(row, ['Property Type', 'property_type', 'PROPERTY_TYPE', 'PropertyType', 'Category']) || 'apartment')
      .toLowerCase()
      .replace('pg/co-living', 'pg-coliving')
      .replace('/', '-');

    const nearbyStr = getValue(row, ['Nearby Places', 'Nearby', 'nearby', 'NEARBY', 'Connectivity']);
    const nearbyArray = nearbyStr ? nearbyStr.split(',').map(s => s.trim()).filter(s => s) : [];

    const property: BulkPropertyData & { [key: string]: any } = {
      // Basic Info - Exact CSV Headers
      fullname: getValue(row, ['Full Name', 'Owner Name', 'owner_name', 'OWNER_NAME', 'Name', 'OwnerName', 'Owner']),
      company: getValue(row, ['Company Name', 'Company', 'company', 'COMPANY', 'Agency', 'Broker']),
      property_manager_contact: getValue(row, ['Mobile Number', 'Contact', 'contact', 'CONTACT', 'Phone', 'Mobile', 'Contact Number']),
      email: getValue(row, ['Email Address', 'Email', 'email', 'EMAIL', 'E-mail']),
      
      // Property Details - Exact CSV Headers
      title: getValue(row, ['Property Title', 'Title', 'title', 'TITLE', 'Name', 'Property Name']),
      description: getValue(row, ['Property Description', 'Description', 'description', 'DESCRIPTION', 'Details']),
      listing_type: listingType,
      property_type: propertyType,
      
      // Specifications - Exact CSV Headers
      built_up_area: parseFloat(getValue(row, ['Built-up Area (sq ft)', 'Built Up Area', 'built_up_area', 'BUILT_UP_AREA', 'Area', 'Size', 'Built Area'])) || 0,
      carpet_area: parseFloat(getValue(row, ['Carpet Area (sq ft)', 'Carpet Area', 'carpet_area', 'CARPET_AREA', 'CarpetArea'])) || 0,
      floor_number: getValue(row, ['Floor Number', 'Floor', 'floor', 'FLOOR']),
      total_floors: getValue(row, ['Total Floors', 'total_floors', 'TOTAL_FLOORS']),
      bedrooms: parseInt(getValue(row, ['Bedrooms', 'bedrooms', 'BEDROOMS', 'BHK', 'Beds', 'Bedroom'])) || 0,
      bathrooms: parseInt(getValue(row, ['Bathrooms', 'bathrooms', 'BATHROOMS', 'Baths', 'Bathroom'])) || 0,
      balconies: parseInt(getValue(row, ['Balconies', 'balconies', 'BALCONIES', 'Balcony'])) || 0,
      facing: getValue(row, ['Facing', 'facing', 'FACING', 'Direction']).toLowerCase(),
      furnishing: (getValue(row, ['Furnishing', 'furnishing', 'FURNISHING', 'Furnished', 'Furniture']) || 'unfurnished').toLowerCase().replace(' ', '-'),
      property_age: getValue(row, ['Property Age', 'Age', 'age', 'AGE', 'Building Age']),
      amenities: (getValue(row, ['Amenities', 'amenities', 'AMENITIES', 'Features']) || '').split(',').map((a: string) => a.trim()).filter((a: string) => a),
      
      // Pricing - Handle CSV price format
      sale_price: 0,
      monthly_rent: 0,
      status: getValue(row, ['Possession Status', 'Status', 'status', 'STATUS', 'Availability']),
      
      // Location - Exact CSV Headers
      address: getValue(row, ['Property Address', 'Address', 'address', 'ADDRESS', 'Location', 'Full Address']),
      city: getValue(row, ['City', 'city', 'CITY']),
      locality: getValue(row, ['Locality', 'locality', 'LOCALITY', 'Area', 'Neighborhood']),
      state: getValue(row, ['State', 'state', 'STATE']) || 'Maharashtra',
      pincode: getValue(row, ['Pincode', 'pincode', 'PINCODE', 'Pin Code', 'ZIP', 'Postal Code']),
      landmark: getValue(row, ['Nearby Landmark', 'Landmark', 'landmark', 'LANDMARK', 'Near']),
      latitude: parseFloat(getValue(row, ['Latitude', 'latitude', 'LATITUDE', 'Lat'])) || 0,
      longitude: parseFloat(getValue(row, ['Longitude', 'longitude', 'LONGITUDE', 'Lng', 'Long'])) || 0,
      nearby_places: nearbyArray,
      nearby: nearbyStr, // Populate string for UI if needed
      
      // Media - Exact CSV Headers
      image_urls: (getValue(row, ['Property Images', 'Images', 'images', 'IMAGES', 'Photos', 'Image URL']) || '').split(',').map((u: string) => u.trim()).filter((u: string) => u),
      virtual_tour_url: getValue(row, ['Property Videos', 'Videos', 'videos', 'VIDEOS', 'Video URL']),
      
      // System fields
      listed_by_id: currentUserId || '1'
    };
    
    console.log(`Parsed property ${index}:`, property);

    // Parse price from CSV format
    const priceText = getValue(row, ['Sale/Rent Price (₹)', 'Sale Price', 'sale_price', 'SALE_PRICE', 'Price', 'SalePrice', 'Cost']);
    console.log(`Price text for row ${index}:`, priceText);
    
    if (priceText) {
      // Remove currency symbols and extract number
      let cleanPrice = priceText.replace(/[₹,₹]/g, '').trim();
      
      if (cleanPrice.includes('/month') || cleanPrice.includes('/mo')) {
        // It's a rental price
        cleanPrice = cleanPrice.replace(/\/month|\/mo/g, '').trim();
        property.monthly_rent = parseFloat(cleanPrice) || 0;
        property.listing_type = 'rent';
      } else if (cleanPrice.includes('Lakhs') || cleanPrice.includes('L')) {
        // It's a sale price in Lakhs
        cleanPrice = cleanPrice.replace(/Lakhs|L/g, '').trim();
        property.sale_price = (parseFloat(cleanPrice) || 0) * 100000;
      } else if (cleanPrice.includes('Cr') || cleanPrice.includes('Crores')) {
        // It's a sale price in Crores
        cleanPrice = cleanPrice.replace(/Cr|Crores/g, '').trim();
        property.sale_price = (parseFloat(cleanPrice) || 0) * 10000000;
      } else {
        // Direct number
        const numPrice = parseFloat(cleanPrice) || 0;
        if (property.listing_type === 'rent') {
          property.monthly_rent = numPrice;
        } else {
          property.sale_price = numPrice;
        }
      }
    }
    
    console.log(`Final prices - Sale: ${property.sale_price}, Rent: ${property.monthly_rent}`);

    // Normalize status values to match API enum
    const statusMap: { [key: string]: string } = {
      'under construction': 'under_review',
      'ready to move': 'active',
      'available immediately': 'active',
      'ready': 'active',
      'available': 'active',
      'sold': 'sold',
      'rented': 'rented'
    };
    
    if (property.status) {
      const normalizedStatus = property.status.toLowerCase();
      property.status = statusMap[normalizedStatus] || 'active';
    } else {
      property.status = 'active';
    }
    
    // Ensure required numeric fields are not null/undefined
    property.built_up_area = property.built_up_area || 1000; // Default if missing
    property.carpet_area = property.carpet_area || property.built_up_area * 0.8;
    property.bedrooms = property.bedrooms || 1;
    property.bathrooms = property.bathrooms || 1;
    property.floor_number = property.floor_number || '1';
    property.total_floors = property.total_floors || '10';
    
    // Ensure coordinates are valid numbers
    if (!property.latitude || property.latitude === 0) property.latitude = 19.0760;
    if (!property.longitude || property.longitude === 0) property.longitude = 72.8777;
    
    // Keep amenities as array for backend
    if (!Array.isArray(property.amenities)) {
      property.amenities = [];
    }
    
    // Ensure all required fields have valid values
    if (!property.title) property.title = 'Property Title';
    if (!property.fullname) property.fullname = 'Property Owner';
    if (!property.property_manager_contact) property.property_manager_contact = '9999999999';
    if (!property.email) property.email = 'owner@example.com';
    if (!property.address) property.address = 'Property Address';
    if (!property.city) property.city = 'Mumbai';
    if (!property.locality) property.locality = 'Central Mumbai';
    if (!property.state) property.state = 'Maharashtra';
    if (!property.pincode) property.pincode = '400001';
    if (!property.description) property.description = 'Property description';
    
    // Create complete property object with correct database field names
    const cleanProperty = {
      ...property,
      nearby_places: property.nearby_places, // Array for backend
      nearby: nearbyStr, // String for UI
      listed_by_id: currentUserId || '1'
    };
    
    console.log(`Final processed property:`, cleanProperty);
    
    const errors: string[] = [];
    
    // Only validate critical fields that would cause API errors
    if (!property.title || property.title.trim() === '') errors.push('Missing Property Title');
    if (!property.address || property.address.trim() === '') errors.push('Missing Property Address');
    if (!property.city || property.city.trim() === '') errors.push('Missing City');
    
    // Price validation - only fail if both are missing
    if (property.listing_type === 'rent' && (!property.monthly_rent || property.monthly_rent <= 0)) {
      errors.push('Missing Rent Price');
    }
    if ((property.listing_type === 'sell' || property.listing_type === 'bidding') && (!property.sale_price || property.sale_price <= 0)) {
      errors.push('Missing Sale Price');
    }
    
    console.log(`Validation errors for row ${index}:`, errors);

    return {
      ...cleanProperty,
      isValid: errors.length === 0,
      errors
    };
  };

  const validateFile = (file: File): boolean => {
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    const maxSize = 10 * 1024 * 1024; // 10MB
    
    if (!['csv', 'xlsx'].includes(fileExtension || '')) {
      alert('Please select a CSV or Excel (.xlsx) file');
      return false;
    }
    
    if (file.size > maxSize) {
      alert('File size must be less than 10MB');
      return false;
    }
    
    return true;
  };

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = event.target.files?.[0];
    if (selectedFile && validateFile(selectedFile)) {
      setFile(selectedFile);
      parseFile(selectedFile);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(false);
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(false);
    
    const droppedFile = e.dataTransfer.files[0];
    if (droppedFile && validateFile(droppedFile)) {
      setFile(droppedFile);
      parseFile(droppedFile);
    }
  };

  const clearFile = () => {
    setFile(null);
    setParsedData([]);
    setShowPropertyList(false);
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const parseFile = async (file: File) => {
    setIsLoading(true);
    setUploadProgress(10);
    
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    
    try {
      if (fileExtension === 'csv') {
        await parseCSVFile(file);
      } else if (fileExtension === 'xlsx') {
        await parseExcel(file);
      }
      setUploadProgress(100);
    } catch (error) {
      console.error('File parsing error:', error);
      alert('Error parsing file. Please check the file format.');
    } finally {
      setIsLoading(false);
      setTimeout(() => setUploadProgress(0), 1000);
    }
  };

  const parseCSVFile = (file: File): Promise<void> => {
    return new Promise((resolve, reject) => {
      console.log('Starting CSV parsing for file:', file.name);
      const reader = new FileReader();
      
      reader.onload = (e) => {
        try {
          setUploadProgress(30);
          const text = e.target?.result as string;
          console.log('CSV text length:', text.length);
          console.log('First 500 chars:', text.substring(0, 500));
          
          const result = parseCSV(text);
          setUploadProgress(60);
      
          console.log('CSV parse result:', result);
          if (result.errors.length > 0) {
            console.error('CSV parsing errors:', result.errors);
          }
          
          const data: ParsedRow[] = result.data.map((row, index) => processRowData(row, index));
          
          setParsedData(data);
          setUploadProgress(90);
          console.log('CSV parsed successfully:', data.length, 'rows');
          console.log('Parsed data sample:', data.slice(0, 2));
          resolve();
        } catch (error) {
          console.error('CSV parsing error:', error);
          reject(error);
        }
      };
      
      reader.onerror = () => reject(new Error('Failed to read file'));
      reader.readAsText(file);
    });
  };

  const parseExcel = async (file: File) => {
    console.log('Starting Excel parsing for file:', file.name);
    try {
      setUploadProgress(30);
      const result = await parseExcelFile(file);
      setUploadProgress(60);
      
      console.log('Excel parse result:', result);
      if (result.errors.length > 0) {
        console.error('Excel parsing errors:', result.errors);
        alert('Error parsing Excel file: ' + result.errors.join(', '));
        return;
      }
      
      const data: ParsedRow[] = result.data.map((row, index) => processRowData(row, index));
      
      setParsedData(data);
      setUploadProgress(90);
      console.log('Excel parsed successfully:', data.length, 'rows');
      console.log('Parsed data sample:', data.slice(0, 2));
    } catch (error) {
      console.error('Excel parsing error:', error);
      throw error;
    }
  };

  const handleSubmit = async () => {
    if (!file) {
      alert('Please select a file first');
      return;
    }

    const validProperties = parsedData.filter(row => row.isValid);
    if (validProperties.length === 0) {
      alert('No valid properties to upload. Please check your data.');
      return;
    }

    const invalidProperties = parsedData.filter(row => !row.isValid);
    if (invalidProperties.length > 0) {
      const proceed = confirm(`Found ${invalidProperties.length} invalid properties. Do you want to upload only the ${validProperties.length} valid properties?`);
      if (!proceed) return;
    }

    setIsLoading(true);
    try {
      console.log('Sending properties to bulk upload API:', validProperties.length);
      
      // Call the API service
      const response = await bulkUploadProperties(validProperties);
      
      console.log('Bulk upload response:', response);
      
      if (response.success) {
        alert(`Successfully uploaded ${response.created_count} properties!`);
        clearFile();
      } else {
        alert(`Upload completed with some errors. Created: ${response.created_count}, Failed: ${response.failed_count}`);
        console.error('Upload errors:', response.errors);
        // Show specific error messages if available
        if (response.errors && response.errors.length > 0) {
            alert(`First Error: ${response.errors[0]}`);
        }
      }
    } catch (error: any) {
      console.error('Upload failed:', error);
      alert('Upload failed: ' + (error.message || 'Unknown error'));
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl shadow-sm p-6 border border-blue-100">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <div className="bg-[#0056D2] p-3 rounded-lg">
              <Upload className="w-8 h-8 text-white" />
            </div>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">Bulk Upload</h1>
              <p className="text-gray-600 mt-1">Upload CSV or Excel files for bulk property processing</p>
            </div>
          </div>
          <div className="text-right">
            <p className="text-sm text-gray-500">Supported formats</p>
            <div className="flex space-x-2 mt-1">
              <span className="bg-green-100 text-green-800 px-2 py-1 rounded text-xs font-medium">CSV</span>
              <span className="bg-blue-100 text-blue-800 px-2 py-1 rounded text-xs font-medium">XLSX</span>
            </div>
          </div>
        </div>
      </div>

      {/* Sample Format Info */}
      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <h3 className="text-sm font-medium text-yellow-800 mb-2">Supported Columns (case-insensitive):</h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-2 text-xs text-yellow-700">
          <div>
            <strong>Owner Info:</strong> Owner Name, Company, Contact, Email
          </div>
          <div>
            <strong>Property:</strong> Title, Description, Listing Type, Property Type, Built Up Area, Carpet Area, Floor, Bedrooms, Bathrooms, Balconies, Facing, Furnishing, Age, Amenities
          </div>
          <div>
            <strong>Location & Media:</strong> Address, City, Locality, State, Pincode, Landmark, Latitude, Longitude, Nearby, Images, Videos, Floor Plan, Documents
          </div>
        </div>
        <p className="text-xs text-yellow-600 mt-2">
          <strong>Required:</strong> Owner Name, Contact, Email, Title, Address, City, Locality, Pincode, Built Up Area, Bedrooms (for residential), Bathrooms (for residential), Price/Rent
        </p>
      </div>

      {/* File Upload */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Upload File</h2>
              <p className="text-sm text-gray-500 mt-1">Maximum file size: 10MB</p>
            </div>
            {file && (
              <div className="flex space-x-2">
                <Button
                  onClick={() => setShowPropertyList(!showPropertyList)}
                  variant="outline"
                  className="flex items-center space-x-2 hover:bg-blue-50"
                >
                  {showPropertyList ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                  <span>{showPropertyList ? 'Hide' : 'Preview'} ({parsedData.length})</span>
                </Button>
                <Button
                  onClick={clearFile}
                  variant="outline"
                  className="flex items-center space-x-2 text-red-600 hover:bg-red-50 hover:border-red-200"
                >
                  <Trash2 className="w-4 h-4" />
                  <span>Clear</span>
                </Button>
              </div>
            )}
          </div>
        </div>
        
        <div className="p-6">
          {!file ? (
            <div 
              className={`border-2 border-dashed rounded-xl p-12 text-center transition-all duration-200 ${
                isDragOver 
                  ? 'border-blue-400 bg-blue-50' 
                  : 'border-gray-300 hover:border-gray-400 hover:bg-gray-50'
              }`}
              onDragOver={handleDragOver}
              onDragLeave={handleDragLeave}
              onDrop={handleDrop}
            >
              <div className="space-y-4">
                <div className={`mx-auto w-16 h-16 rounded-full flex items-center justify-center ${
                  isDragOver ? 'bg-blue-100' : 'bg-gray-100'
                }`}>
                  <Upload className={`w-8 h-8 ${
                    isDragOver ? 'text-blue-600' : 'text-gray-400'
                  }`} />
                </div>
                <div>
                  <p className="text-xl font-medium text-gray-900 mb-2">
                    {isDragOver ? 'Drop your file here' : 'Drag & drop your file here'}
                  </p>
                  <p className="text-gray-500 mb-4">or</p>
                  <Button 
                    onClick={() => fileInputRef.current?.click()}
                    className="bg-[#0056D2] hover:bg-[#0041A3] text-white px-6 py-2"
                  >
                    Choose File
                  </Button>
                </div>
                <div className="text-sm text-gray-500 space-y-1">
                  <p>Supports CSV and Excel (.xlsx) files</p>
                </div>
              </div>
              <Input
                ref={fileInputRef}
                type="file"
                accept=".csv,.xlsx"
                onChange={handleFileChange}
                className="hidden"
              />
            </div>
          ) : (
            <div className="space-y-4">
              {/* File Info */}
              <div className="bg-gray-50 rounded-lg p-4 border">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-3">
                    <div className="bg-blue-100 p-2 rounded">
                      <FileText className="w-5 h-5 text-blue-600" />
                    </div>
                    <div>
                      <p className="font-medium text-gray-900">{file.name}</p>
                      <p className="text-sm text-gray-500">
                        {(file.size / 1024 / 1024).toFixed(2)} MB • {file.type || 'Unknown type'}
                      </p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-2">
                    {parsedData.length > 0 && (
                      <div className="flex items-center space-x-2">
                        <CheckCircle className="w-5 h-5 text-green-500" />
                        <span className="text-sm font-medium text-green-700">
                          {parsedData.filter(p => p.isValid).length} valid properties
                        </span>
                      </div>
                    )}
                  </div>
                </div>
              </div>
              
              {/* Progress Bar */}
              {(isLoading || uploadProgress > 0) && (
                <div className="space-y-2">
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-600">Processing file...</span>
                    <span className="text-gray-600">{uploadProgress}%</span>
                  </div>
                  <Progress value={uploadProgress} className="h-2" />
                </div>
              )}
            </div>
          )}
        </div>
        
        {/* Submit Button */}
        {file && !isLoading && (
          <div className="p-6 border-t border-gray-200 bg-gray-50 rounded-b-xl flex justify-end">
            <Button
              onClick={handleSubmit}
              className="bg-[#0056D2] hover:bg-[#0041A3] text-white px-8 py-2"
            >
              Upload {parsedData.filter(p => p.isValid).length} Properties
            </Button>
          </div>
        )}
      </div>

      {/* Comprehensive Data Preview */}
      {parsedData.length > 0 && showPropertyList && (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200">
            <div className="flex items-center justify-between">
              <h2 className="text-xl font-semibold text-gray-900">
                Property Data Preview ({parsedData.length} properties)
              </h2>
              <div className="flex space-x-3">
                <div className="flex items-center space-x-2 bg-green-50 px-3 py-2 rounded-lg">
                  <CheckCircle className="w-4 h-4 text-green-600" />
                  <span className="text-sm font-medium text-green-800">
                    {parsedData.filter(p => p.isValid).length} Valid
                  </span>
                </div>
                <div className="flex items-center space-x-2 bg-red-50 px-3 py-2 rounded-lg">
                  <AlertCircle className="w-4 h-4 text-red-600" />
                  <span className="text-sm font-medium text-red-800">
                    {parsedData.filter(p => !p.isValid).length} Invalid
                  </span>
                </div>
              </div>
            </div>
          </div>
          
          <div className="p-6">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200 text-xs">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Owner</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Company</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Contact</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Email</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Title</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Description</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Listing Type</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Property Type</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Built Area</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Carpet Area</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Floor</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Total Floors</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">BHK</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Bath</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Balcony</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Facing</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Furnishing</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Age</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Amenities</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Price</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Address</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">City</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Locality</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Pincode</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Landmark</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Lat</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Lng</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Nearby</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Images</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Videos</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Floor Plan</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Documents</th>
                    <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">Validation</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {parsedData.map((property: any, index) => (
                    <tr key={index} className={`hover:bg-gray-50 ${property.isValid ? '' : 'bg-red-50'}`}>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-24 truncate" title={property.fullname}>
                          {property.fullname || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-20 truncate" title={property.company}>
                          {property.company || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.property_manager_contact || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-32 truncate" title={property.email}>
                          {property.email || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-40 truncate font-medium" title={property.title}>
                          {property.title || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-48 truncate" title={property.description}>
                          {property.description || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900 capitalize">{property.listing_type || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900 capitalize">{property.property_type || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.built_up_area || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.carpet_area || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.floor || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.total_floors || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.bedrooms || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.bathrooms || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.balconies || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900 capitalize">{property.facing || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900 capitalize">{property.furnishing || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.age || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-32 truncate" title={Array.isArray(property.amenities) ? property.amenities.join(', ') : property.amenities}>
                          {Array.isArray(property.amenities) ? property.amenities.slice(0, 3).join(', ') + (property.amenities.length > 3 ? '...' : '') : property.amenities || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900 font-medium">
                        {(property.listing_type === 'sell' || property.listing_type === 'urgent sale' || property.listing_type === 'bidding') 
                          ? (property.sale_price && property.sale_price > 0 ? (property.sale_price > 10000000 ? `₹${(property.sale_price/10000000).toFixed(1)} Cr` : `₹${(property.sale_price/100000).toFixed(0)} L`) : '₹0')
                          : (property.monthly_rent && property.monthly_rent > 0 ? `₹${property.monthly_rent.toLocaleString()}/mo` : '₹0')
                        }
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900 capitalize">{property.status || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-40 truncate" title={property.address}>
                          {property.address || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.city || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.locality || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.pincode || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-24 truncate" title={property.landmark}>
                          {property.landmark || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.latitude || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">{property.longitude || 'N/A'}</td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        <div className="max-w-32 truncate" title={property.nearby}>
                          {property.nearby || 'N/A'}
                        </div>
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        {property.images ? (
                          <a href={property.images} target="_blank" rel="noopener noreferrer" className="text-blue-600 hover:underline">
                            View
                          </a>
                        ) : 'N/A'}
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        {property.videos ? (
                          <a href={property.videos} target="_blank" rel="noopener noreferrer" className="text-blue-600 hover:underline">
                            View
                          </a>
                        ) : 'N/A'}
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        {property.floor_plan ? (
                          <a href={property.floor_plan} target="_blank" rel="noopener noreferrer" className="text-blue-600 hover:underline">
                            View
                          </a>
                        ) : 'N/A'}
                      </td>
                      <td className="px-3 py-2 text-xs text-gray-900">
                        {property.documents ? (
                          <a href={property.documents} target="_blank" rel="noopener noreferrer" className="text-blue-600 hover:underline">
                            View
                          </a>
                        ) : 'N/A'}
                      </td>
                      <td className="px-3 py-2 text-xs">
                        {property.isValid ? (
                          <span className="text-green-600 font-medium">Valid</span>
                        ) : (
                          <span className="text-red-600" title={property.errors.join(', ')}>
                            {property.errors.length} Errors
                          </span>
                        )}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default BulkPropertyUpload;