import React, { useState, useRef } from 'react';
import { Upload, Building, FileText, CheckCircle, AlertCircle, Download, Eye, EyeOff, Trash2 } from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Progress } from '../ui/progress';
import { bulkUploadProperties, BulkPropertyData, BulkPropertyUploadResponse } from '../../api/properties';
import { parseCSV } from '../../utils/csvParser';
import { parseExcel as parseExcelFile } from '../../utils/excelParser';

interface ParsedRow extends BulkPropertyData {
  isValid: boolean;
  errors: string[];
}

const BulkPropertyUpload: React.FC = () => {
  const [file, setFile] = useState<File | null>(null);
  const [parsedData, setParsedData] = useState<ParsedRow[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [showPropertyList, setShowPropertyList] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isDragOver, setIsDragOver] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const validateFile = (file: File): boolean => {
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    const maxSize = 10 * 1024 * 1024; // 10MB
    
    if (!['csv', 'xlsx'].includes(fileExtension || '')) {
      alert('Please select a CSV or Excel (.xlsx) file');
      return false;
    }
    
    if (file.size > maxSize) {
      alert('File size must be less than 10MB');
      return false;
    }
    
    return true;
  };

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = event.target.files?.[0];
    if (selectedFile && validateFile(selectedFile)) {
      setFile(selectedFile);
      parseFile(selectedFile);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(false);
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    setIsDragOver(false);
    
    const droppedFile = e.dataTransfer.files[0];
    if (droppedFile && validateFile(droppedFile)) {
      setFile(droppedFile);
      parseFile(droppedFile);
    }
  };

  const clearFile = () => {
    setFile(null);
    setParsedData([]);
    setShowPropertyList(false);
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const parseFile = async (file: File) => {
    setIsLoading(true);
    setUploadProgress(10);
    
    const fileExtension = file.name.split('.').pop()?.toLowerCase();
    
    try {
      if (fileExtension === 'csv') {
        await parseCSVFile(file);
      } else if (fileExtension === 'xlsx') {
        await parseExcel(file);
      }
      setUploadProgress(100);
    } catch (error) {
      console.error('File parsing error:', error);
      alert('Error parsing file. Please check the file format.');
    } finally {
      setIsLoading(false);
      setTimeout(() => setUploadProgress(0), 1000);
    }
  };

  // Helper function to get value from row with multiple possible keys
  const getValue = (row: any, keys: string[]) => {
    for (const key of keys) {
      if (row[key] !== undefined && row[key] !== null && row[key] !== '') {
        return String(row[key]).trim();
      }
    }
    return '';
  };

  const processRowData = (row: any, index: number): ParsedRow => {
    console.log(`Processing row ${index}:`, row);
    console.log('Available keys:', Object.keys(row));
    
    const property: BulkPropertyData = {
      title: getValue(row, ['Title', 'title', 'TITLE', 'Property Title', 'Name', 'Property Name']),
      description: getValue(row, ['Description', 'description', 'DESCRIPTION', 'Details', 'Property Description']),
      listing_type: (getValue(row, ['Listing Type', 'listing_type', 'LISTING_TYPE', 'Type', 'ListingType', 'Sale/Rent']) || 'sell').toLowerCase(),
      property_type: (getValue(row, ['Property Type', 'property_type', 'PROPERTY_TYPE', 'PropertyType', 'Category']) || 'apartment').toLowerCase(),
      address: getValue(row, ['Address', 'address', 'ADDRESS', 'Location', 'Full Address']),
      city: getValue(row, ['City', 'city', 'CITY']),
      locality: getValue(row, ['Locality', 'locality', 'LOCALITY', 'Area', 'Neighborhood']),
      state: getValue(row, ['State', 'state', 'STATE']) || 'Karnataka',
      pincode: getValue(row, ['Pincode', 'pincode', 'PINCODE', 'Pin Code', 'ZIP', 'Postal Code']),
      bedrooms: parseInt(getValue(row, ['Bedrooms', 'bedrooms', 'BEDROOMS', 'BHK', 'Beds', 'Bedroom'])) || 0,
      bathrooms: parseInt(getValue(row, ['Bathrooms', 'bathrooms', 'BATHROOMS', 'Baths', 'Bathroom'])) || 0,
      built_up_area: parseFloat(getValue(row, ['Built Up Area', 'built_up_area', 'BUILT_UP_AREA', 'Area', 'Size', 'Built Area'])) || 0,
      carpet_area: parseFloat(getValue(row, ['Carpet Area', 'carpet_area', 'CARPET_AREA', 'CarpetArea'])) || 0,
      sale_price: parseFloat(getValue(row, ['Sale Price', 'sale_price', 'SALE_PRICE', 'Price', 'SalePrice', 'Cost'])) || 0,
      monthly_rent: parseFloat(getValue(row, ['Monthly Rent', 'monthly_rent', 'MONTHLY_RENT', 'Rent', 'MonthlyRent', 'Rental'])) || 0,
      furnishing: (getValue(row, ['Furnishing', 'furnishing', 'FURNISHING', 'Furnished', 'Furniture']) || 'unfurnished').toLowerCase(),
      amenities: (getValue(row, ['Amenities', 'amenities', 'AMENITIES', 'Features']) || '').split(',').map((a: string) => a.trim()).filter((a: string) => a),
      property_manager_contact: getValue(row, ['Contact', 'contact', 'CONTACT', 'Phone', 'Mobile', 'Contact Number']) || '9999999999',
      email: getValue(row, ['Email', 'email', 'EMAIL', 'E-mail', 'Email Address']) || 'admin@newageland.com',
      fullname: getValue(row, ['Owner Name', 'owner_name', 'OWNER_NAME', 'Name', 'OwnerName', 'Owner']) || 'Property Owner',
      listed_by_id: 'bulk_upload_admin'
    };
    
    console.log(`Parsed property ${index}:`, property);

    const errors: string[] = [];
    if (!property.title || property.title.trim() === '') errors.push('Missing Title');
    if (!property.address || property.address.trim() === '') errors.push('Missing Address');
    if (!property.city || property.city.trim() === '') errors.push('Missing City');
    if (!property.state || property.state.trim() === '') errors.push('Missing State');
    if (!property.pincode || property.pincode.trim() === '') errors.push('Missing Pincode');
    if (!property.bedrooms || property.bedrooms <= 0) errors.push('Invalid Bedrooms');
    if (!property.bathrooms || property.bathrooms <= 0) errors.push('Invalid Bathrooms');
    if (!property.built_up_area || property.built_up_area <= 0) errors.push('Invalid Built Up Area');
    if (property.listing_type === 'sell' && (!property.sale_price || property.sale_price <= 0)) {
      errors.push('Missing or invalid Sale Price');
    }
    if (property.listing_type === 'rent' && (!property.monthly_rent || property.monthly_rent <= 0)) {
      errors.push('Missing or invalid Monthly Rent');
    }
    
    console.log(`Validation errors for row ${index}:`, errors);

    return {
      ...property,
      isValid: errors.length === 0,
      errors
    };
  };

  const parseCSVFile = (file: File): Promise<void> => {
    return new Promise((resolve, reject) => {
      console.log('Starting CSV parsing for file:', file.name);
      const reader = new FileReader();
      
      reader.onload = (e) => {
        try {
          setUploadProgress(30);
          const text = e.target?.result as string;
          console.log('CSV text length:', text.length);
          console.log('First 500 chars:', text.substring(0, 500));
          
          const result = parseCSV(text);
          setUploadProgress(60);
      
          console.log('CSV parse result:', result);
          if (result.errors.length > 0) {
            console.error('CSV parsing errors:', result.errors);
          }
          
          const data: ParsedRow[] = result.data.map((row, index) => processRowData(row, index));
          
          setParsedData(data);
          setUploadProgress(90);
          console.log('CSV parsed successfully:', data.length, 'rows');
          console.log('Parsed data sample:', data.slice(0, 2));
          resolve();
        } catch (error) {
          console.error('CSV parsing error:', error);
          reject(error);
        }
      };
      
      reader.onerror = () => reject(new Error('Failed to read file'));
      reader.readAsText(file);
    });
  };

  const parseExcel = async (file: File) => {
    console.log('Starting Excel parsing for file:', file.name);
    try {
      setUploadProgress(30);
      const result = await parseExcelFile(file);
      setUploadProgress(60);
      
      console.log('Excel parse result:', result);
      if (result.errors.length > 0) {
        console.error('Excel parsing errors:', result.errors);
        alert('Error parsing Excel file: ' + result.errors.join(', '));
        return;
      }
      
      const data: ParsedRow[] = result.data.map((row, index) => processRowData(row, index));
      
      setParsedData(data);
      setUploadProgress(90);
      console.log('Excel parsed successfully:', data.length, 'rows');
      console.log('Parsed data sample:', data.slice(0, 2));
    } catch (error) {
      console.error('Excel parsing error:', error);
      throw error;
    }
  };

  const handleSubmit = async () => {
    if (!file) {
      alert('Please select a file first');
      return;
    }

    const validProperties = parsedData.filter(row => row.isValid);
    if (validProperties.length === 0) {
      alert('No valid properties to upload. Please check your data.');
      return;
    }

    const invalidProperties = parsedData.filter(row => !row.isValid);
    if (invalidProperties.length > 0) {
      const proceed = confirm(`Found ${invalidProperties.length} invalid properties. Do you want to upload only the ${validProperties.length} valid properties?`);
      if (!proceed) return;
    }

    setIsLoading(true);
    try {
      console.log('Uploading properties:', validProperties.length, 'properties');
      
      const result = await bulkUploadProperties(validProperties);
      
      if (result.success) {
        alert(`All properties uploaded successfully! Created: ${result.created_count} properties.`);
        setFile(null);
        setParsedData([]);
      } else {
        const errorDetails = result.errors.length > 0 ? `\n\nErrors:\n${result.errors.slice(0, 3).join('\n')}${result.errors.length > 3 ? '\n...and more' : ''}` : '';
        alert(`Upload completed with some issues. Created: ${result.created_count}, Failed: ${result.failed_count}${errorDetails}`);
        setFile(null);
        setParsedData([]);
      }
    } catch (error) {
      console.error('Upload error:', error);
      alert('Upload failed. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const canSubmit = file !== null && !isLoading;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl shadow-sm p-6 border border-blue-100">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <div className="bg-[#0056D2] p-3 rounded-lg">
              <Upload className="w-8 h-8 text-white" />
            </div>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">Bulk Upload</h1>
              <p className="text-gray-600 mt-1">Upload CSV or Excel files for bulk property processing</p>
            </div>
          </div>
          <div className="text-right">
            <p className="text-sm text-gray-500">Supported formats</p>
            <div className="flex space-x-2 mt-1">
              <span className="bg-green-100 text-green-800 px-2 py-1 rounded text-xs font-medium">CSV</span>
              <span className="bg-blue-100 text-blue-800 px-2 py-1 rounded text-xs font-medium">XLSX</span>
            </div>
          </div>
        </div>
      </div>

      {/* Sample Format Info */}
      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <h3 className="text-sm font-medium text-yellow-800 mb-2">Required Columns (case-insensitive):</h3>
        <p className="text-xs text-yellow-700">
          Title, Address, City, State, Pincode, Bedrooms, Bathrooms, Built Up Area, Listing Type (sell/rent), Property Type, Sale Price (for sell), Monthly Rent (for rent)
        </p>
      </div>

      {/* File Upload */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Upload File</h2>
              <p className="text-sm text-gray-500 mt-1">Maximum file size: 10MB</p>
            </div>
            {file && (
              <div className="flex space-x-2">
                <Button
                  onClick={() => setShowPropertyList(!showPropertyList)}
                  variant="outline"
                  className="flex items-center space-x-2 hover:bg-blue-50"
                >
                  {showPropertyList ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                  <span>{showPropertyList ? 'Hide' : 'Preview'} ({parsedData.length})</span>
                </Button>
                <Button
                  onClick={clearFile}
                  variant="outline"
                  className="flex items-center space-x-2 text-red-600 hover:bg-red-50 hover:border-red-200"
                >
                  <Trash2 className="w-4 h-4" />
                  <span>Clear</span>
                </Button>
              </div>
            )}
          </div>
        </div>
        
        <div className="p-6">
          {!file ? (
            <div 
              className={`border-2 border-dashed rounded-xl p-12 text-center transition-all duration-200 ${
                isDragOver 
                  ? 'border-blue-400 bg-blue-50' 
                  : 'border-gray-300 hover:border-gray-400 hover:bg-gray-50'
              }`}
              onDragOver={handleDragOver}
              onDragLeave={handleDragLeave}
              onDrop={handleDrop}
            >
              <div className="space-y-4">
                <div className={`mx-auto w-16 h-16 rounded-full flex items-center justify-center ${
                  isDragOver ? 'bg-blue-100' : 'bg-gray-100'
                }`}>
                  <Upload className={`w-8 h-8 ${
                    isDragOver ? 'text-blue-600' : 'text-gray-400'
                  }`} />
                </div>
                <div>
                  <p className="text-xl font-medium text-gray-900 mb-2">
                    {isDragOver ? 'Drop your file here' : 'Drag & drop your file here'}
                  </p>
                  <p className="text-gray-500 mb-4">or</p>
                  <Button 
                    onClick={() => fileInputRef.current?.click()}
                    className="bg-[#0056D2] hover:bg-[#0041A3] text-white px-6 py-2"
                  >
                    Choose File
                  </Button>
                </div>
                <div className="text-sm text-gray-500 space-y-1">
                  <p>Supports CSV and Excel (.xlsx) files</p>
                </div>
              </div>
              <Input
                ref={fileInputRef}
                type="file"
                accept=".csv,.xlsx"
                onChange={handleFileChange}
                className="hidden"
              />
            </div>
          ) : (
            <div className="space-y-4">
              {/* File Info */}
              <div className="bg-gray-50 rounded-lg p-4 border">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-3">
                    <div className="bg-blue-100 p-2 rounded">
                      <FileText className="w-5 h-5 text-blue-600" />
                    </div>
                    <div>
                      <p className="font-medium text-gray-900">{file.name}</p>
                      <p className="text-sm text-gray-500">
                        {(file.size / 1024 / 1024).toFixed(2)} MB • {file.type || 'Unknown type'}
                      </p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-2">
                    {parsedData.length > 0 && (
                      <div className="flex items-center space-x-2">
                        <CheckCircle className="w-5 h-5 text-green-500" />
                        <span className="text-sm font-medium text-green-700">
                          {parsedData.filter(p => p.isValid).length} valid properties
                        </span>
                      </div>
                    )}
                  </div>
                </div>
              </div>
              
              {/* Progress Bar */}
              {(isLoading || uploadProgress > 0) && (
                <div className="space-y-2">
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-600">Processing file...</span>
                    <span className="text-gray-600">{uploadProgress}%</span>
                  </div>
                  <Progress value={uploadProgress} className="h-2" />
                </div>
              )}
            </div>
          )}
        </div>
      </div>

      {/* Data Preview */}
      {parsedData.length > 0 && showPropertyList && (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200">
            <div className="flex items-center justify-between">
              <h2 className="text-xl font-semibold text-gray-900">
                Data Preview ({parsedData.length} properties)
              </h2>
              <div className="flex space-x-3">
                <div className="flex items-center space-x-2 bg-green-50 px-3 py-2 rounded-lg">
                  <CheckCircle className="w-4 h-4 text-green-600" />
                  <span className="text-sm font-medium text-green-800">
                    {parsedData.filter(p => p.isValid).length} Valid
                  </span>
                </div>
                <div className="flex items-center space-x-2 bg-red-50 px-3 py-2 rounded-lg">
                  <AlertCircle className="w-4 h-4 text-red-600" />
                  <span className="text-sm font-medium text-red-800">
                    {parsedData.filter(p => !p.isValid).length} Invalid
                  </span>
                </div>
              </div>
            </div>
          </div>
          
          <div className="p-6">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Title</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Address</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">City</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Bedrooms</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Price</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {parsedData.slice(0, 10).map((property, index) => (
                    <tr key={index} className={`hover:bg-gray-50 ${property.isValid ? '' : 'bg-red-50'}`}>
                      <td className="px-6 py-4 text-sm text-gray-900">
                        <div className="font-medium truncate max-w-xs" title={property.title}>
                          {property.title || 'N/A'}
                        </div>
                      </td>
                      <td className="px-6 py-4 text-sm text-gray-900">
                        <div className="truncate max-w-xs" title={property.address}>
                          {property.address || 'N/A'}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">{property.city || 'N/A'}</td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">{property.bedrooms || 0}</td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 font-medium">
                        {property.listing_type === 'sell' 
                          ? `₹${property.sale_price?.toLocaleString() || 0}` 
                          : `₹${property.monthly_rent?.toLocaleString() || 0}/mo`
                        }
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 capitalize">{property.property_type}</td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        {property.isValid ? (
                          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                            <CheckCircle className="w-3 h-3 mr-1" />
                            Valid
                          </span>
                        ) : (
                          <span 
                            className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800 cursor-help" 
                            title={property.errors.join(', ')}
                          >
                            <AlertCircle className="w-3 h-3 mr-1" />
                            Invalid
                          </span>
                        )}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
              {parsedData.length > 10 && (
                <div className="mt-4 text-center">
                  <p className="text-sm text-gray-500">
                    Showing first 10 of {parsedData.length} properties
                  </p>
                </div>
              )}
            </div>
          </div>
        </div>
      )}

      {/* Submit Button */}
      {file && parsedData.length > 0 && (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6">
            <div className="text-center space-y-4">
              <div className="flex justify-center space-x-6 text-sm">
                <div className="flex items-center space-x-2">
                  <div className="w-3 h-3 bg-green-500 rounded-full"></div>
                  <span className="text-gray-600">
                    {parsedData.filter(p => p.isValid).length} properties ready to upload
                  </span>
                </div>
                {parsedData.filter(p => !p.isValid).length > 0 && (
                  <div className="flex items-center space-x-2">
                    <div className="w-3 h-3 bg-red-500 rounded-full"></div>
                    <span className="text-gray-600">
                      {parsedData.filter(p => !p.isValid).length} properties will be skipped
                    </span>
                  </div>
                )}
              </div>
              
              <Button
                onClick={handleSubmit}
                disabled={!canSubmit || isLoading || parsedData.filter(p => p.isValid).length === 0}
                className="bg-[#0056D2] hover:bg-[#0041A3] text-white px-12 py-3 text-lg font-semibold rounded-lg disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-200 shadow-lg hover:shadow-xl"
              >
                {isLoading ? (
                  <div className="flex items-center space-x-2">
                    <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    <span>Uploading Properties...</span>
                  </div>
                ) : (
                  <div className="flex items-center space-x-2">
                    <Upload className="w-5 h-5" />
                    <span>Upload {parsedData.filter(p => p.isValid).length} Properties</span>
                  </div>
                )}
              </Button>
              
              <p className="text-xs text-gray-500 max-w-md mx-auto">
                By uploading, you confirm that all property information is accurate and you have the right to list these properties.
              </p>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default BulkPropertyUpload;