import React, { useState, useEffect } from 'react';
import {
  TrendingUp, TrendingDown, DollarSign, Users, Building, CreditCard,
  Calendar, ArrowUpRight, ArrowDownRight, Activity, Target, Percent,
  PieChart, BarChart3, LineChart, RefreshCw, Download, Filter
} from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';

interface RevenueMetrics {
  total_revenue: number;
  revenue_growth: number;
  monthly_revenue: number;
  quarterly_revenue: number;
  annual_revenue: number;
  avg_transaction_value: number;
  revenue_by_source: Record<string, number>;
  revenue_trend: Array<{ date: string; revenue: number; transactions: number }>;
}

interface BusinessMetrics {
  total_transactions: number;
  completed_transactions: number;
  pending_transactions: number;
  failed_transactions: number;
  success_rate: number;
  conversion_rate: number;
  active_properties: number;
  sold_properties: number;
  avg_days_to_close: number;
  customer_lifetime_value: number;
}

export function BusinessRevenueDashboard() {
  const [revenueMetrics, setRevenueMetrics] = useState<RevenueMetrics | null>(null);
  const [businessMetrics, setBusinessMetrics] = useState<BusinessMetrics | null>(null);
  const [loading, setLoading] = useState(true);
  const [timeRange, setTimeRange] = useState<'7d' | '30d' | '90d' | '1y'>('30d');

  useEffect(() => {
    fetchDashboardData();
  }, [timeRange]);

  const fetchDashboardData = async () => {
    setLoading(true);
    // Use mock data directly
    setRevenueMetrics(getMockRevenueData());
    setBusinessMetrics(getMockBusinessData());
    setLoading(false);
  };

  const getMockRevenueData = (): RevenueMetrics => ({
    total_revenue: 125000000,
    revenue_growth: 23.5,
    monthly_revenue: 8500000,
    quarterly_revenue: 24500000,
    annual_revenue: 98000000,
    avg_transaction_value: 4250000,
    revenue_by_source: {
      'Property Sales': 75000000,
      'Rental Income': 28000000,
      'Service Charges': 15000000,
      'Consultation Fees': 7000000
    },
    revenue_trend: [
      { date: '2024-01-01', revenue: 6500000, transactions: 45 },
      { date: '2024-01-08', revenue: 7200000, transactions: 52 },
      { date: '2024-01-15', revenue: 8100000, transactions: 58 },
      { date: '2024-01-22', revenue: 7800000, transactions: 54 },
      { date: '2024-01-29', revenue: 8900000, transactions: 62 }
    ]
  });

  const getMockBusinessData = (): BusinessMetrics => ({
    total_transactions: 1847,
    completed_transactions: 1654,
    pending_transactions: 156,
    failed_transactions: 37,
    success_rate: 89.5,
    conversion_rate: 12.8,
    active_properties: 2890,
    sold_properties: 1245,
    avg_days_to_close: 42,
    customer_lifetime_value: 8750000
  });

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(2)}Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(2)}L`;
    return `₹${amount.toLocaleString('en-IN')}`;
  };

  const formatPercentage = (value: number) => {
    const sign = value >= 0 ? '+' : '';
    return `${sign}${value.toFixed(1)}%`;
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Business & Revenue Dashboard</h1>
          <p className="text-gray-600 mt-1">Real-time financial insights and business metrics</p>
        </div>
        <div className="flex gap-3">
          <select
            value={timeRange}
            onChange={(e) => setTimeRange(e.target.value as any)}
            className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          >
            <option value="7d">Last 7 Days</option>
            <option value="30d">Last 30 Days</option>
            <option value="90d">Last 90 Days</option>
            <option value="1y">Last Year</option>
          </select>
          <button className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 flex items-center gap-2">
            <Download className="w-4 h-4" />
            Export
          </button>
          <button onClick={fetchDashboardData} className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 flex items-center gap-2">
            <RefreshCw className="w-4 h-4" />
            Refresh
          </button>
        </div>
      </div>

      {/* Key Revenue Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <Card className="border-l-4 border-l-blue-600">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-blue-100 rounded-lg">
                <DollarSign className="w-6 h-6 text-blue-600" />
              </div>
              <span className={`flex items-center gap-1 text-sm font-semibold ${revenueMetrics!.revenue_growth >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                {revenueMetrics!.revenue_growth >= 0 ? <TrendingUp className="w-4 h-4" /> : <TrendingDown className="w-4 h-4" />}
                {formatPercentage(revenueMetrics!.revenue_growth)}
              </span>
            </div>
            <h3 className="text-sm font-medium text-gray-600 mb-1">Total Revenue</h3>
            <p className="text-3xl font-bold text-gray-900">{formatCurrency(revenueMetrics!.total_revenue)}</p>
            <p className="text-xs text-gray-500 mt-2">vs previous period</p>
          </CardContent>
        </Card>

        <Card className="border-l-4 border-l-green-600">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-green-100 rounded-lg">
                <Calendar className="w-6 h-6 text-green-600" />
              </div>
              <span className="text-sm font-semibold text-green-600">MTD</span>
            </div>
            <h3 className="text-sm font-medium text-gray-600 mb-1">Monthly Revenue</h3>
            <p className="text-3xl font-bold text-gray-900">{formatCurrency(revenueMetrics!.monthly_revenue)}</p>
            <p className="text-xs text-gray-500 mt-2">Current month</p>
          </CardContent>
        </Card>

        <Card className="border-l-4 border-l-purple-600">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-purple-100 rounded-lg">
                <CreditCard className="w-6 h-6 text-purple-600" />
              </div>
              <span className="text-sm font-semibold text-purple-600">AVG</span>
            </div>
            <h3 className="text-sm font-medium text-gray-600 mb-1">Avg Transaction Value</h3>
            <p className="text-3xl font-bold text-gray-900">{formatCurrency(revenueMetrics!.avg_transaction_value)}</p>
            <p className="text-xs text-gray-500 mt-2">Per transaction</p>
          </CardContent>
        </Card>

        <Card className="border-l-4 border-l-orange-600">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-orange-100 rounded-lg">
                <Target className="w-6 h-6 text-orange-600" />
              </div>
              <span className="text-sm font-semibold text-orange-600">CLV</span>
            </div>
            <h3 className="text-sm font-medium text-gray-600 mb-1">Customer Lifetime Value</h3>
            <p className="text-3xl font-bold text-gray-900">{formatCurrency(businessMetrics!.customer_lifetime_value)}</p>
            <p className="text-xs text-gray-500 mt-2">Average per customer</p>
          </CardContent>
        </Card>
      </div>

      {/* Business Performance Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-indigo-100 rounded-lg">
                <Activity className="w-6 h-6 text-indigo-600" />
              </div>
            </div>
            <h3 className="text-sm font-medium text-gray-600 mb-1">Total Transactions</h3>
            <p className="text-3xl font-bold text-gray-900">{businessMetrics!.total_transactions.toLocaleString()}</p>
            <div className="mt-3 flex items-center gap-2 text-xs">
              <span className="text-green-600 font-medium">{businessMetrics!.completed_transactions} completed</span>
              <span className="text-gray-400">•</span>
              <span className="text-yellow-600 font-medium">{businessMetrics!.pending_transactions} pending</span>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-green-100 rounded-lg">
                <Percent className="w-6 h-6 text-green-600" />
              </div>
            </div>
            <h3 className="text-sm font-medium text-gray-600 mb-1">Success Rate</h3>
            <p className="text-3xl font-bold text-gray-900">{businessMetrics!.success_rate}%</p>
            <div className="mt-3 w-full bg-gray-200 rounded-full h-2">
              <div className="bg-green-600 h-2 rounded-full" style={{ width: `${businessMetrics!.success_rate}%` }}></div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-blue-100 rounded-lg">
                <Building className="w-6 h-6 text-blue-600" />
              </div>
            </div>
            <h3 className="text-sm font-medium text-gray-600 mb-1">Properties Sold</h3>
            <p className="text-3xl font-bold text-gray-900">{businessMetrics!.sold_properties.toLocaleString()}</p>
            <p className="text-xs text-gray-500 mt-2">
              {((businessMetrics!.sold_properties / businessMetrics!.active_properties) * 100).toFixed(1)}% of active inventory
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div className="p-3 bg-yellow-100 rounded-lg">
                <Calendar className="w-6 h-6 text-yellow-600" />
              </div>
            </div>
            <h3 className="text-sm font-medium text-gray-600 mb-1">Avg Days to Close</h3>
            <p className="text-3xl font-bold text-gray-900">{businessMetrics!.avg_days_to_close}</p>
            <p className="text-xs text-gray-500 mt-2">Average closing time</p>
          </CardContent>
        </Card>
      </div>

      {/* Revenue Breakdown & Trend */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Revenue by Source */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <PieChart className="w-5 h-5" />
              Revenue by Source
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {Object.entries(revenueMetrics!.revenue_by_source).map(([source, amount], index) => {
                const percentage = (amount / revenueMetrics!.total_revenue) * 100;
                const colors = ['bg-blue-600', 'bg-green-600', 'bg-purple-600', 'bg-orange-600'];
                return (
                  <div key={source}>
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-sm font-medium text-gray-700">{source}</span>
                      <span className="text-sm font-bold text-gray-900">{formatCurrency(amount)}</span>
                    </div>
                    <div className="w-full bg-gray-200 rounded-full h-3">
                      <div className={`${colors[index]} h-3 rounded-full transition-all`} style={{ width: `${percentage}%` }}></div>
                    </div>
                    <p className="text-xs text-gray-500 mt-1">{percentage.toFixed(1)}% of total revenue</p>
                  </div>
                );
              })}
            </div>
          </CardContent>
        </Card>

        {/* Revenue Trend */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <LineChart className="w-5 h-5" />
              Revenue Trend
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {revenueMetrics!.revenue_trend.map((item, index) => (
                <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
                  <div>
                    <p className="text-sm font-medium text-gray-900">{new Date(item.date).toLocaleDateString('en-IN', { month: 'short', day: 'numeric' })}</p>
                    <p className="text-xs text-gray-500">{item.transactions} transactions</p>
                  </div>
                  <div className="text-right">
                    <p className="text-lg font-bold text-gray-900">{formatCurrency(item.revenue)}</p>
                    {index > 0 && (
                      <p className={`text-xs flex items-center gap-1 ${item.revenue >= revenueMetrics!.revenue_trend[index - 1].revenue ? 'text-green-600' : 'text-red-600'}`}>
                        {item.revenue >= revenueMetrics!.revenue_trend[index - 1].revenue ? <ArrowUpRight className="w-3 h-3" /> : <ArrowDownRight className="w-3 h-3" />}
                        {Math.abs(((item.revenue - revenueMetrics!.revenue_trend[index - 1].revenue) / revenueMetrics!.revenue_trend[index - 1].revenue) * 100).toFixed(1)}%
                      </p>
                    )}
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Quarterly & Annual Overview */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <Card className="bg-gradient-to-br from-blue-50 to-blue-100">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-sm font-medium text-blue-900">Quarterly Revenue</h3>
              <BarChart3 className="w-5 h-5 text-blue-600" />
            </div>
            <p className="text-4xl font-bold text-blue-900">{formatCurrency(revenueMetrics!.quarterly_revenue)}</p>
            <p className="text-sm text-blue-700 mt-2">Q1 2024 Performance</p>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-50 to-green-100">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-sm font-medium text-green-900">Annual Revenue</h3>
              <TrendingUp className="w-5 h-5 text-green-600" />
            </div>
            <p className="text-4xl font-bold text-green-900">{formatCurrency(revenueMetrics!.annual_revenue)}</p>
            <p className="text-sm text-green-700 mt-2">YTD Performance</p>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-sm font-medium text-purple-900">Conversion Rate</h3>
              <Target className="w-5 h-5 text-purple-600" />
            </div>
            <p className="text-4xl font-bold text-purple-900">{businessMetrics!.conversion_rate}%</p>
            <p className="text-sm text-purple-700 mt-2">Lead to customer conversion</p>
          </CardContent>
        </Card>
      </div>

      {/* Footer */}
      <div className="text-center py-4 border-t border-gray-200">
        <p className="text-xs text-gray-500">
          Last updated: <span className="font-medium">{new Date().toLocaleString('en-IN')}</span>
        </p>
        <p className="text-xs text-gray-400 mt-1">Demo data - Connect to database for live metrics</p>
      </div>
    </div>
  );
}
