import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Button } from '../ui/button';
import { Badge } from '../ui/badge';
import { getCampaigns, deleteCampaign, approveCampaign } from '../../api/campaigns';
import { toast } from 'sonner';
import { 
  Eye, 
  CheckCircle, 
  XCircle, 
  Clock, 
  TrendingUp, 
  Users, 
  IndianRupee,
  Calendar,
  MapPin,
  Building
} from 'lucide-react';

interface Campaign {
  id: string;
  campaignName: string;
  campaignObjective: string;
  propertyType: string;
  propertyLocation: string;
  budgetAmount: number;
  budgetType: string;
  startDate: string;
  endDate: string;
  status: string;
  agentName: string;
  agentContact: string;
  platforms: string[];
  created_at: string;
}

export function CampaignManagement() {
  const [campaigns, setCampaigns] = useState<Campaign[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedCampaign, setSelectedCampaign] = useState<Campaign | null>(null);

  useEffect(() => {
    fetchCampaigns();
  }, []);

  const fetchCampaigns = async () => {
    try {
      setLoading(true);
      const response = await getCampaigns({ admin: true });
      setCampaigns(response.campaigns || []);
    } catch (error) {
      console.error('Error fetching campaigns:', error);
      toast.error('Failed to fetch campaigns');
    } finally {
      setLoading(false);
    }
  };

  const getStatusBadge = (status: string) => {
    const statusConfig = {
      draft: { color: 'bg-gray-100 text-gray-800', label: 'Draft' },
      pending: { color: 'bg-yellow-100 text-yellow-800', label: 'Pending Approval' },
      approved: { color: 'bg-green-100 text-green-800', label: 'Approved' },
      rejected: { color: 'bg-red-100 text-red-800', label: 'Rejected' },
      active: { color: 'bg-blue-100 text-blue-800', label: 'Active' },
      completed: { color: 'bg-purple-100 text-purple-800', label: 'Completed' }
    };
    
    const config = statusConfig[status as keyof typeof statusConfig] || statusConfig.draft;
    return <Badge className={config.color}>{config.label}</Badge>;
  };

  const handleApprove = async (campaignId: string) => {
    try {
      await approveCampaign(campaignId);
      toast.success('Campaign approved successfully');
      fetchCampaigns();
    } catch (error: any) {
      const message =
        error?.response?.data?.message ||
        error?.response?.data?.detail ||
        error?.message ||
        'Failed to approve campaign';
      toast.error(message);
    }
  };

  const handleReject = async (campaignId: string) => {
    try {
      await deleteCampaign(campaignId);
      toast.success('Campaign rejected and deleted');
      fetchCampaigns();
    } catch (error) {
      toast.error('Failed to reject campaign');
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className={`${selectedCampaign ? 'blur-sm' : ''} transition-all duration-300`}>
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Campaign Management</h1>
            <p className="text-gray-600 mt-1">Review and approve user-submitted campaigns</p>
          </div>
          <div className="flex gap-2">
            <Button onClick={fetchCampaigns} variant="outline">
              Refresh
            </Button>
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mt-6 mb-6">
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center">
                <Clock className="h-8 w-8 text-yellow-600" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Pending Approval</p>
                  <p className="text-2xl font-bold text-gray-900">
                    {campaigns.filter(c => c.status === 'draft' || c.status === 'pending').length}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center">
                <CheckCircle className="h-8 w-8 text-green-600" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Approved</p>
                  <p className="text-2xl font-bold text-gray-900">
                    {campaigns.filter(c => c.status === 'approved' || c.status === 'active').length}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center">
                <TrendingUp className="h-8 w-8 text-blue-600" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Campaigns</p>
                  <p className="text-2xl font-bold text-gray-900">{campaigns.length}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center">
                <IndianRupee className="h-8 w-8 text-purple-600" />
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">Total Budget</p>
                  <p className="text-2xl font-bold text-gray-900">
                    ₹{campaigns.reduce((sum, c) => sum + c.budgetAmount, 0).toLocaleString()}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Campaigns List */}
        <Card>
          <CardHeader>
            <CardTitle>Campaign Submissions</CardTitle>
          </CardHeader>
          <CardContent>
            {campaigns.length === 0 ? (
              <div className="text-center py-8">
                <TrendingUp className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                <p className="text-gray-500">No campaigns found</p>
              </div>
            ) : (
              <div className="space-y-4">
                {campaigns.map((campaign) => (
                  <div key={campaign.id} className="border rounded-lg p-4 hover:bg-gray-50 transition-colors">
                    <div className="flex justify-between items-start">
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-2">
                          <h3 className="text-lg font-semibold text-gray-900">{campaign.campaignName}</h3>
                          {getStatusBadge(campaign.status)}
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm text-gray-600">
                          <div className="flex items-center gap-2">
                            <Building className="h-4 w-4" />
                            <span>{campaign.propertyType} in {campaign.propertyLocation}</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <IndianRupee className="h-4 w-4" />
                            <span>₹{campaign.budgetAmount.toLocaleString()} ({campaign.budgetType})</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <Calendar className="h-4 w-4" />
                            <span>{campaign.startDate} to {campaign.endDate}</span>
                          </div>
                        </div>

                        <div className="mt-2 flex items-center gap-4 text-sm text-gray-600">
                          <div className="flex items-center gap-2">
                            <Users className="h-4 w-4" />
                            <span>{campaign.agentName} ({campaign.agentContact})</span>
                          </div>
                          <div className="flex items-center gap-2">
                            <TrendingUp className="h-4 w-4" />
                            <span>{campaign.campaignObjective.replace('_', ' ')}</span>
                          </div>
                        </div>

                        <div className="mt-2">
                          <span className="text-xs text-gray-500">
                            Platforms: {campaign.platforms?.join(', ') || 'Not specified'}
                          </span>
                        </div>
                      </div>

                      <div className="flex gap-2 ml-4">
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => setSelectedCampaign(campaign)}
                        >
                          <Eye className="h-4 w-4 mr-1" />
                          View
                        </Button>
                        
                        {(campaign.status === 'draft' || campaign.status === 'pending') && (
                          <>
                            <Button
                              size="sm"
                              onClick={() => handleApprove(campaign.id)}
                              style={{ backgroundColor: '#16a34a', color: 'white' }}
                              className="hover:bg-green-700 border-green-600"
                            >
                              <CheckCircle className="h-4 w-4 mr-1" />
                              Approve
                            </Button>
                            <Button
                              size="sm"
                              variant="destructive"
                              onClick={() => handleReject(campaign.id)}
                            >
                              <XCircle className="h-4 w-4 mr-1" />
                              Reject
                            </Button>
                          </>
                        )}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>
      </div>

      {/* Campaign Details Modal */}
      {selectedCampaign && (
        <div className="fixed inset-0 bg-gray-500 bg-opacity-30 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-2xl w-full mx-4 max-h-[80vh] overflow-y-auto shadow-xl">
            <div className="flex justify-between items-center mb-4">
              <h2 className="text-xl font-bold">Campaign Details</h2>
              <Button variant="outline" onClick={() => setSelectedCampaign(null)}>
                Close
              </Button>
            </div>
            
            <div className="space-y-4">
              <div>
                <h3 className="font-semibold text-gray-900">{selectedCampaign.campaignName}</h3>
                <p className="text-gray-600">{selectedCampaign.campaignObjective.replace('_', ' ')}</p>
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-gray-600">Property Type</label>
                  <p className="text-gray-900">{selectedCampaign.propertyType}</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-600">Location</label>
                  <p className="text-gray-900">{selectedCampaign.propertyLocation}</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-600">Budget</label>
                  <p className="text-gray-900">₹{selectedCampaign.budgetAmount.toLocaleString()} ({selectedCampaign.budgetType})</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-600">Status</label>
                  <div className="mt-1">{getStatusBadge(selectedCampaign.status)}</div>
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-600">Agent Name</label>
                  <p className="text-gray-900">{selectedCampaign.agentName}</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-600">Agent Contact</label>
                  <p className="text-gray-900">{selectedCampaign.agentContact}</p>
                </div>
              </div>
              
              <div>
                <label className="text-sm font-medium text-gray-600">Campaign Duration</label>
                <p className="text-gray-900">{selectedCampaign.startDate} to {selectedCampaign.endDate}</p>
              </div>
              
              <div>
                <label className="text-sm font-medium text-gray-600">Platforms</label>
                <p className="text-gray-900">{selectedCampaign.platforms?.join(', ') || 'Not specified'}</p>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
