import React, { useState, useEffect } from 'react';
import { Shield, FileText, CheckCircle, AlertCircle, Users, Plus, Eye, Mail, AlertTriangle } from 'lucide-react';
import axios from 'axios';

interface ConsentForm {
  id: number;
  title: string;
  content: string;
  version: string;
  is_active: boolean;
  created_at: string;
}

interface UserCompliance {
  user_id: number;
  email: string;
  user_type: string;
  total_progress: number;
  document_progress: number;
  consent_progress: number;
  status: string;
  missing_documents: number;
  missing_consents: number;
}

export function ComplianceManagement() {
  const [activeTab, setActiveTab] = useState<'forms' | 'users' | 'documents' | 'breaches'>('forms');
  const [consentForms, setConsentForms] = useState<ConsentForm[]>([]);
  const [userCompliance, setUserCompliance] = useState<UserCompliance[]>([]);
  const [filteredCompliance, setFilteredCompliance] = useState<UserCompliance[]>([]);
  const [documentRequirements, setDocumentRequirements] = useState<any[]>([]);
  const [showAddDoc, setShowAddDoc] = useState(false);
  const [editingDoc, setEditingDoc] = useState<any>(null);
  const [newDoc, setNewDoc] = useState({ user_type: 'buyer', document_type: '', document_name: '', description: '', is_required: true });
  const [roleFilter, setRoleFilter] = useState<string>('all');
  const [loading, setLoading] = useState(false);
  const [showCreateForm, setShowCreateForm] = useState(false);
  const [showReport, setShowReport] = useState(false);
  const [selectedFormReport, setSelectedFormReport] = useState<any>(null);
  const [newForm, setNewForm] = useState({ title: '', content: '', version: '1.0', type: 'privacy' });
  const [breachType, setBreachType] = useState<string>('password');
  const [breachUsers, setBreachUsers] = useState<any[]>([]);
  const [selectedUsers, setSelectedUsers] = useState<Set<string>>(new Set());
  const [showBreachReport, setShowBreachReport] = useState(false);
  const [breachReport, setBreachReport] = useState<any>(null);
  const [breachSubTab, setBreachSubTab] = useState<'send' | 'report'>('send');
  const [showSendBreachModal, setShowSendBreachModal] = useState(false);
  const [selectedBreachType, setSelectedBreachType] = useState<string>('password');

  const API_BASE = 'http://localhost:8090/api';

  useEffect(() => {
    if (activeTab === 'forms') {
      fetchConsentForms();
    } else if (activeTab === 'users') {
      fetchUserCompliance();
    } else if (activeTab === 'documents') {
      fetchDocumentRequirements();
    } else if (activeTab === 'breaches') {
      if (breachSubTab === 'report') {
        viewBreachReport();
      } else {
        fetchBreachUsers();
      }
    }
  }, [activeTab, breachSubTab]);

  useEffect(() => {
    if (activeTab === 'breaches') {
      fetchBreachUsers();
    }
  }, [breachType]);

  const fetchConsentForms = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      const response = await axios.get(`${API_BASE}/compliance/admin/consent-forms`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setConsentForms(response.data || []);
    } catch (error) {
      console.error('Failed to fetch consent forms:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchUserCompliance = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      console.log('Fetching user compliance with token:', token ? 'Token exists' : 'No token');
      const response = await axios.get(`${API_BASE}/compliance/admin/user-compliance`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      console.log('User compliance response:', response.data);
      console.log('Users array:', response.data.users);
      console.log('Users length:', response.data.users?.length);
      setUserCompliance(response.data.users || []);
      setFilteredCompliance(response.data.users || []);
    } catch (error) {
      console.error('Failed to fetch user compliance:', error);
      if (axios.isAxiosError(error)) {
        console.error('Error response:', error.response?.data);
        console.error('Error status:', error.response?.status);
      }
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    if (roleFilter === 'all') {
      setFilteredCompliance(userCompliance);
    } else {
      setFilteredCompliance(userCompliance.filter(u => u.user_type === roleFilter));
    }
  }, [roleFilter, userCompliance]);

  const createConsentForm = async () => {
    // Validate all fields
    if (!newForm.title.trim()) {
      alert('Please enter a title');
      return;
    }
    if (!newForm.content.trim()) {
      alert('Please enter content');
      return;
    }
    if (!newForm.version.trim()) {
      alert('Please enter a version');
      return;
    }

    try {
      const token = localStorage.getItem('access_token');
      const payload = {
        ...newForm,
        is_required: true
      };
      await axios.post(`${API_BASE}/compliance/admin/consent-forms`, payload, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setShowCreateForm(false);
      setNewForm({ title: '', content: '', version: '1.0', type: 'privacy' });
      fetchConsentForms();
      alert('Compliance rule created successfully!');
    } catch (error) {
      console.error('Failed to create consent form:', error);
      alert('Failed to create compliance rule');
    }
  };

  const fetchDocumentRequirements = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      const response = await axios.get(`${API_BASE}/compliance/admin/document-requirements`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setDocumentRequirements(response.data || []);
    } catch (error) {
      console.error('Failed to fetch document requirements:', error);
    } finally {
      setLoading(false);
    }
  };

  const viewReport = async (formId: number) => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await axios.get(`${API_BASE}/compliance/admin/consent-forms/${formId}/report`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      console.log('Report data:', response.data);
      setSelectedFormReport(response.data);
      setShowReport(true);
    } catch (error) {
      console.error('Failed to fetch report:', error);
      alert('Failed to load report. Please check console for details.');
    }
  };

  const createDocRequirement = async () => {
    try {
      const token = localStorage.getItem('access_token');
      await axios.post(`${API_BASE}/compliance/admin/document-requirements`, newDoc, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setShowAddDoc(false);
      setNewDoc({ user_type: 'buyer', document_type: '', document_name: '', description: '', is_required: true });
      fetchDocumentRequirements();
    } catch (error) {
      console.error('Failed to create document requirement:', error);
    }
  };

  const updateDocRequirement = async () => {
    try {
      const token = localStorage.getItem('access_token');
      await axios.put(`${API_BASE}/compliance/admin/document-requirements/${editingDoc.id}`, editingDoc, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setEditingDoc(null);
      fetchDocumentRequirements();
    } catch (error) {
      console.error('Failed to update document requirement:', error);
    }
  };

  const deleteDocRequirement = async (id: number) => {
    if (!window.confirm('Delete this document requirement?')) return;
    try {
      const token = localStorage.getItem('access_token');
      await axios.delete(`${API_BASE}/compliance/admin/document-requirements/${id}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      fetchDocumentRequirements();
    } catch (error) {
      console.error('Failed to delete document requirement:', error);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'fully_compliant': return 'text-green-600 bg-green-50';
      case 'almost_complete': return 'text-blue-600 bg-blue-50';
      case 'partially_compliant': return 'text-yellow-600 bg-yellow-50';
      default: return 'text-red-600 bg-red-50';
    }
  };

  const getStatusIcon = (status: string) => {
    return status === 'fully_compliant' ? <CheckCircle className="w-4 h-4" /> : <AlertCircle className="w-4 h-4" />;
  };

  const getProgressColor = (progress: number) => {
    if (progress >= 80) return 'from-green-500 to-green-600';
    if (progress >= 50) return 'from-yellow-500 to-yellow-600';
    return 'from-red-500 to-red-600';
  };

  const fetchBreachUsers = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      const response = await axios.get(`${API_BASE}/compliance/admin/breach-users?type=${breachType}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setBreachUsers(response.data.users || []);
      setSelectedUsers(new Set());
    } catch (error) {
      console.error('Failed to fetch breach users:', error);
    } finally {
      setLoading(false);
    }
  };

  const toggleUserSelection = (userId: string) => {
    const newSelected = new Set(selectedUsers);
    if (newSelected.has(userId)) {
      newSelected.delete(userId);
    } else {
      newSelected.add(userId);
    }
    setSelectedUsers(newSelected);
  };

  const toggleAllUsers = () => {
    if (selectedUsers.size === breachUsers.length) {
      setSelectedUsers(new Set());
    } else {
      setSelectedUsers(new Set(breachUsers.map(u => u.user_id)));
    }
  };

  const sendBreachNotifications = async () => {
    if (selectedUsers.size === 0) {
      alert('Please select at least one user');
      return;
    }

    const breachMessages = {
      password: 'SECURITY ALERT: We detected 5+ failed login attempts on your account. If this was not you, please reset your password immediately and contact support.',
      security: 'SECURITY ALERT: Unauthorized access attempt detected on your account. If this was not you, please contact the support team or change your password.',
      document: 'COMPLIANCE ALERT: Your uploaded document verification failed (below 80% match). Please re-upload a clear, valid document to maintain compliance.'
    };

    const message = breachMessages[selectedBreachType as keyof typeof breachMessages];
    
    if (window.confirm(`Send breach notification to ${selectedUsers.size} user(s)?\n\n${message}`)) {
      try {
        const token = localStorage.getItem('access_token');
        console.log('Sending breach notification:', {
          user_ids: Array.from(selectedUsers),
          breach_type: selectedBreachType,
          message: message
        });
        await axios.post(`${API_BASE}/compliance/admin/send-breach-notification`, {
          user_ids: Array.from(selectedUsers),
          breach_type: selectedBreachType,
          message: message
        }, {
          headers: { Authorization: `Bearer ${token}` }
        });
        alert('Breach notifications sent successfully!');
        setSelectedUsers(new Set());
        setShowSendBreachModal(false);
        fetchBreachUsers();
      } catch (error) {
        console.error('Failed to send breach notifications:', error);
        if (axios.isAxiosError(error)) {
          console.error('Error response:', error.response?.data);
          alert(`Failed to send breach notifications: ${error.response?.data?.detail || error.message}`);
        } else {
          alert('Failed to send breach notifications');
        }
      }
    }
  };

  const openSendBreachModal = () => {
    setSelectedBreachType('password');
    setShowSendBreachModal(true);
    // Fetch users for the default breach type
    fetchBreachUsersForType('password');
  };

  const fetchBreachUsersForType = async (type: string) => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      const response = await axios.get(`${API_BASE}/compliance/admin/breach-users?type=${type}`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      setBreachUsers(response.data.users || []);
      setSelectedUsers(new Set());
    } catch (error) {
      console.error('Failed to fetch breach users:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleBreachTypeChange = (type: string) => {
    setSelectedBreachType(type);
    fetchBreachUsersForType(type);
  };

  const viewBreachReport = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('access_token');
      const response = await axios.get(`${API_BASE}/compliance/admin/breach-notification-report`, {
        headers: { Authorization: `Bearer ${token}` }
      });
      console.log('Breach report data:', response.data);
      setBreachReport(response.data);
    } catch (error) {
      console.error('Failed to fetch breach report:', error);
      if (axios.isAxiosError(error)) {
        console.error('Error response:', error.response?.data);
      }
    } finally {
      setLoading(false);
    }
  };

  const downloadBreachReportPDF = () => {
    if (!breachReport) return;
    
    const printWindow = window.open('', '_blank');
    if (!printWindow) return;
    
    const html = `
      <!DOCTYPE html>
      <html>
      <head>
        <title>Breach Notification Report</title>
        <style>
          body { font-family: Arial, sans-serif; padding: 20px; }
          h1 { color: #dc2626; }
          .stats { display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; margin: 20px 0; }
          .stat-card { border: 2px solid #e5e7eb; padding: 15px; border-radius: 8px; }
          .stat-label { font-size: 12px; color: #6b7280; font-weight: bold; }
          .stat-value { font-size: 28px; font-weight: bold; margin-top: 5px; }
          table { width: 100%; border-collapse: collapse; margin-top: 20px; }
          th, td { border: 1px solid #e5e7eb; padding: 10px; text-align: left; }
          th { background: #f3f4f6; font-weight: bold; }
          .badge { padding: 4px 8px; border-radius: 4px; font-size: 11px; font-weight: bold; }
          .badge-password { background: #fee2e2; color: #991b1b; }
          .badge-security { background: #fed7aa; color: #9a3412; }
          .badge-document { background: #fef3c7; color: #92400e; }
        </style>
      </head>
      <body>
        <h1>Breach Notification Report</h1>
        <p>Generated on: ${new Date().toLocaleString()}</p>
        <div class="stats">
          <div class="stat-card">
            <div class="stat-label">Total Notifications</div>
            <div class="stat-value">${breachReport.statistics.total_notifications}</div>
          </div>
          <div class="stat-card">
            <div class="stat-label">Unique Users</div>
            <div class="stat-value">${breachReport.statistics.unique_users}</div>
          </div>
          <div class="stat-card">
            <div class="stat-label">Password Breaches</div>
            <div class="stat-value">${breachReport.statistics.password_breaches}</div>
          </div>
          <div class="stat-card">
            <div class="stat-label">Security Breaches</div>
            <div class="stat-value">${breachReport.statistics.security_breaches}</div>
          </div>
        </div>
        <table>
          <thead>
            <tr>
              <th>Date</th>
              <th>Email</th>
              <th>Role</th>
              <th>Breach Type</th>
              <th>Status</th>
            </tr>
          </thead>
          <tbody>
            ${breachReport.notifications.map((n: any) => `
              <tr>
                <td>${new Date(n.sent_at).toLocaleString()}</td>
                <td>${n.email}</td>
                <td>${n.user_type}</td>
                <td><span class="badge badge-${n.breach_type}">${n.breach_type.toUpperCase()}</span></td>
                <td>${n.acknowledged ? 'Acknowledged' : 'Sent'}</td>
              </tr>
            `).join('')}
          </tbody>
        </table>
      </body>
      </html>
    `;
    
    printWindow.document.write(html);
    printWindow.document.close();
    setTimeout(() => {
      printWindow.print();
    }, 250);
  };

  const sendReminderEmail = async (user: UserCompliance) => {
    const roleMessages = {
      buyer: `Dear User,Your compliance is at ${user.total_progress}%. Please upload the following documents:- ID Card- Income Proof- Address ProofAlso, accept all required consent forms.Thank you.`,
      seller: `Dear User,Your compliance is at ${user.total_progress}%. Please upload the following documents:- Property Deed- Tax Clearance Certificate- NOC (No Objection Certificate)Also, accept all required consent forms.Thank you.`,
      agent: `Dear User,Your compliance is at ${user.total_progress}%. Please upload the following documents:- Agent License- Professional Certification- Insurance CertificateAlso, accept all required consent forms.Thank you.`
    };

    const message = roleMessages[user.user_type as keyof typeof roleMessages] || roleMessages.buyer;
    
    if (window.confirm(`Send reminder to ${user.email}?${message}`)) {
      try {
        const token = localStorage.getItem('access_token');
        await axios.post(`${API_BASE}/compliance/admin/send-reminder`, {
          user_id: user.user_id,
          email: user.email,
          user_type: user.user_type,
          progress: user.total_progress,
          message: message
        }, {
          headers: { Authorization: `Bearer ${token}` }
        });
        alert('Reminder email sent successfully!');
      } catch (error) {
        console.error('Failed to send reminder:', error);
        alert('Failed to send reminder email');
      }
    }
  };

  return (
    <>
    <div className="space-y-6 min-h-screen bg-gray-50 p-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-600 to-indigo-600 rounded-xl shadow-lg p-8">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold mb-2">Compliance Management</h1>
            <p className="text-blue-100">Manage compliance rules and track user compliance status</p>
          </div>
          {activeTab === 'forms' && (
            <button
              onClick={() => setShowCreateForm(true)}
              className="flex items-center gap-2 px-6 py-3 bg-white text-blue-600 rounded-lg hover:bg-blue-50 font-semibold shadow-lg transition-all"
            >
              <Plus className="w-5 h-5" />
              Create Compliance Rule
            </button>
          )}
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-xl shadow-md overflow-hidden">
        <div className="flex border-b border-gray-200">
          <button
            onClick={() => setActiveTab('forms')}
            className={`flex-1 px-6 py-4 font-semibold text-sm transition-all ${
              activeTab === 'forms'
                ? 'bg-blue-50 text-blue-600 border-b-2 border-blue-600'
                : 'text-gray-600 hover:bg-gray-50'
            }`}
          >
            <div className="flex items-center justify-center gap-2">
              <FileText className="w-5 h-5" />
              Compliance Rules
            </div>
          </button>
          <button
            onClick={() => setActiveTab('users')}
            className={`flex-1 px-6 py-4 font-semibold text-sm transition-all ${
              activeTab === 'users'
                ? 'bg-blue-50 text-blue-600 border-b-2 border-blue-600'
                : 'text-gray-600 hover:bg-gray-50'
            }`}
          >
            <div className="flex items-center justify-center gap-2">
              <Users className="w-5 h-5" />
              User Compliance
            </div>
          </button>
          <button
            onClick={() => setActiveTab('breaches')}
            className={`flex-1 px-6 py-4 font-semibold text-sm transition-all ${
              activeTab === 'breaches'
                ? 'bg-blue-50 text-blue-600 border-b-2 border-blue-600'
                : 'text-gray-600 hover:bg-gray-50'
            }`}
          >
            <div className="flex items-center justify-center gap-2">
              <AlertTriangle className="w-5 h-5" />
              Breach Notifications
            </div>
          </button>
        </div>

        {/* Content */}
        <div className="p-6">
          {loading ? (
            <div className="flex justify-center py-12">
              <div className="animate-spin rounded-full h-12 w-12 border-4 border-blue-600 border-t-transparent"></div>
            </div>
          ) : activeTab === 'forms' ? (
            <div className="grid gap-4">
              {consentForms.length === 0 ? (
                <div className="text-center py-12">
                  <FileText className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                  <p className="text-gray-500 text-lg">No compliance rules created yet</p>
                  <p className="text-gray-400 text-sm mt-2">Click "Create Compliance Rule" to get started</p>
                </div>
              ) : (
                consentForms.map((form) => (
                  <div key={form.id} className="bg-gradient-to-r from-white to-gray-50 rounded-xl shadow-md hover:shadow-lg transition-shadow p-6 border border-gray-200">
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-3">
                          <Shield className="w-6 h-6 text-blue-600" />
                          <h3 className="text-xl font-bold text-gray-900">{form.title}</h3>
                          <span className="px-3 py-1 text-xs font-semibold bg-gray-100 text-gray-700 rounded-full">
                            v{form.version}
                          </span>
                          {form.is_active && (
                            <span className="px-3 py-1 text-xs font-semibold bg-green-100 text-green-700 rounded-full flex items-center gap-1">
                              <CheckCircle className="w-3 h-3" />
                              Active
                            </span>
                          )}
                        </div>
                        <p className="text-gray-600 leading-relaxed line-clamp-2">{form.content}</p>
                        <p className="text-sm text-gray-500 mt-3 flex items-center gap-2">
                          <span className="font-medium">Created:</span>
                          {new Date(form.created_at).toLocaleDateString('en-US', { 
                            year: 'numeric', 
                            month: 'long', 
                            day: 'numeric' 
                          })}
                        </p>
                      </div>
                      <button 
                        onClick={() => viewReport(form.id)}
                        className="px-4 py-2 rounded-lg transition-colors font-semibold flex items-center gap-2"
                        style={{ backgroundColor: '#2563eb', color: 'white' }}
                      >
                        <Eye className="w-4 h-4" />
                        View Report
                      </button>
                    </div>
                  </div>
                ))
              )}
            </div>
          ) : activeTab === 'users' ? (
            <div>
              <div className="mb-4 flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <label className="text-sm font-semibold text-gray-700">Filter by Role:</label>
                  <select
                    value={roleFilter}
                    onChange={(e) => setRoleFilter(e.target.value)}
                    className="px-4 py-2 border-2 border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 font-medium"
                  >
                    <option value="all">All Users</option>
                    <option value="buyer">Buyers</option>
                    <option value="seller">Sellers</option>
                    <option value="agent">Agents</option>
                  </select>
                </div>
                <div className="flex items-center gap-2 px-4 py-2 bg-blue-50 rounded-lg border border-blue-200">
                  <Users className="w-5 h-5 text-blue-600" />
                  <span className="text-sm font-bold text-blue-900">Total: {filteredCompliance.length} users</span>
                </div>
              </div>
              {console.log('Rendering users tab, filteredCompliance:', filteredCompliance)}
              <div className="bg-white rounded-xl shadow-lg overflow-hidden border-2 border-gray-200">
              {filteredCompliance.length === 0 ? (
                <div className="text-center py-12">
                  <Users className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                  <p className="text-gray-500 text-lg">No user compliance data available</p>
                </div>
              ) : (
                <table className="min-w-full divide-y divide-gray-200">
                  <thead className="bg-gradient-to-r from-gray-50 to-gray-100">
                    <tr>
                      <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">User</th>
                      <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Role</th>
                      <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Progress</th>
                      <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Docs</th>
                      <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Consents</th>
                      <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase tracking-wider">Status</th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {filteredCompliance.map((user) => (
                      <tr key={user.user_id} className="hover:bg-blue-50 transition-colors">
                        <td className="px-6 py-4">
                          <div className="flex items-center gap-3">
                            <div className="w-10 h-10 rounded-full flex items-center justify-center text-white font-bold shadow flex-shrink-0" style={{ backgroundColor: '#2563eb' }}>
                              <span className="text-sm">{(user.email || 'U').charAt(0).toUpperCase()}</span>
                            </div>
                            <div className="text-sm font-semibold text-gray-900">{user.email || 'No email'}</div>
                          </div>
                        </td>
                        <td className="px-6 py-4">
                          <span className={`px-3 py-1 text-xs font-bold rounded-md ${
                            user.user_type === 'buyer' ? 'bg-blue-100 text-blue-700' :
                            user.user_type === 'seller' ? 'bg-green-100 text-green-700' :
                            user.user_type === 'agent' ? 'bg-purple-100 text-purple-700' : 'bg-gray-100 text-gray-700'
                          }`}>
                            {user.user_type}
                          </span>
                        </td>
                        <td className="px-6 py-4">
                          <div className="flex items-center gap-2">
                            <div className="w-32 bg-gray-200 rounded-full h-4">
                              <div
                                className={`bg-gradient-to-r ${getProgressColor(user.total_progress)} h-4 rounded-full transition-all`}
                                style={{ width: `${user.total_progress}%` }}
                              ></div>
                            </div>
                            <span className="text-sm font-bold text-gray-700 min-w-[35px]">{user.total_progress}%</span>
                          </div>
                        </td>
                        <td className="px-6 py-4">
                          <span className="text-sm font-bold text-gray-700">{user.document_progress}%</span>
                        </td>
                        <td className="px-6 py-4">
                          <span className="text-sm font-bold text-gray-700">{user.consent_progress}%</span>
                        </td>
                        <td className="px-6 py-4">
                          <span className={`px-2 py-1 text-xs font-bold rounded-md ${getStatusColor(user.status)}`}>
                            {user.status.replace('_', ' ')}
                          </span>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              )}
            </div>
            </div>
          ) : activeTab === 'documents' ? (
            <div className="space-y-4">
              <button
                onClick={() => setShowAddDoc(true)}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-semibold flex items-center gap-2"
              >
                <Plus className="w-4 h-4" />
                Add Document Requirement
              </button>
              <div className="grid gap-4">
                {['buyer', 'seller', 'agent'].map(role => (
                  <div key={role} className="bg-white rounded-lg border border-gray-200 p-4">
                    <h3 className="text-lg font-bold text-gray-900 mb-3 capitalize">{role} Documents</h3>
                    <div className="space-y-2">
                      {documentRequirements.filter(d => d.user_type === role).map(doc => (
                        <div key={doc.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                          <div className="flex-1">
                            <div className="font-semibold text-gray-900">{doc.document_name}</div>
                            <div className="text-sm text-gray-600">{doc.description}</div>
                            <div className="text-xs text-gray-500 mt-1">Type: {doc.document_type}</div>
                          </div>
                          <div className="flex items-center gap-2">
                            {doc.is_required && <span className="px-2 py-1 text-xs bg-red-100 text-red-700 rounded-full">Required</span>}
                            <button onClick={() => setEditingDoc(doc)} className="px-3 py-1 text-blue-600 hover:bg-blue-50 rounded text-sm font-semibold">
                              Edit
                            </button>
                            <button onClick={() => deleteDocRequirement(doc.id)} className="px-3 py-1 text-red-600 hover:bg-red-50 rounded text-sm font-semibold">
                              Delete
                            </button>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                ))}
              </div>
            </div>
          ) : activeTab === 'breaches' ? (
            <div className="space-y-6">
              <div className="bg-gradient-to-r from-red-50 to-orange-50 border-2 border-red-200 rounded-xl p-6">
                <div className="flex items-start gap-4">
                  <AlertTriangle className="w-8 h-8 text-red-600 flex-shrink-0 mt-1" />
                  <div>
                    <h3 className="text-lg font-bold text-red-900 mb-2">Breach Notification System</h3>
                    <p className="text-red-700 text-sm">Manage security breach notifications and view reports</p>
                  </div>
                </div>
              </div>

              {/* Sub-tabs for Breach Notifications */}
              <div className="bg-white rounded-xl shadow-md overflow-hidden">
                <div className="flex border-b border-gray-200">
                  <button
                    onClick={() => setBreachSubTab('send')}
                    className={`flex-1 px-6 py-3 font-semibold text-sm transition-all ${
                      breachSubTab === 'send'
                        ? 'bg-red-50 text-red-600 border-b-2 border-red-600'
                        : 'text-gray-600 hover:bg-gray-50'
                    }`}
                  >
                    <div className="flex items-center justify-center gap-2">
                      <Mail className="w-4 h-4" />
                      Send Breach Notification
                    </div>
                  </button>
                  <button
                    onClick={() => setBreachSubTab('report')}
                    className={`flex-1 px-6 py-3 font-semibold text-sm transition-all ${
                      breachSubTab === 'report'
                        ? 'bg-red-50 text-red-600 border-b-2 border-red-600'
                        : 'text-gray-600 hover:bg-gray-50'
                    }`}
                  >
                    <div className="flex items-center justify-center gap-2">
                      <Eye className="w-4 h-4" />
                      View Report
                    </div>
                  </button>
                </div>
              </div>

              {breachSubTab === 'send' ? (
                <div className="bg-white rounded-xl border-2 border-gray-200 p-8">
                  <div className="text-center max-w-2xl mx-auto">
                    <div className="bg-gradient-to-br from-red-50 to-orange-50 rounded-full w-24 h-24 flex items-center justify-center mx-auto mb-6">
                      <Mail className="w-12 h-12 text-red-600" />
                    </div>
                    <h3 className="text-2xl font-bold text-gray-900 mb-3">Send Breach Notification</h3>
                    <p className="text-gray-600 mb-8">Notify users about security breaches, failed login attempts, or document verification issues</p>
                    <button
                      onClick={openSendBreachModal}
                      className="px-8 py-4 rounded-xl font-bold shadow-lg hover:shadow-xl transition-all flex items-center gap-3 mx-auto text-lg"
                      style={{ backgroundColor: '#dc2626', color: 'white' }}
                      onMouseEnter={(e) => e.currentTarget.style.backgroundColor = '#b91c1c'}
                      onMouseLeave={(e) => e.currentTarget.style.backgroundColor = '#dc2626'}
                    >
                      <AlertTriangle className="w-6 h-6" />
                      Send Breach Notification
                    </button>
                    <div className="mt-8 grid grid-cols-3 gap-4 text-left">
                      <div className="bg-red-50 rounded-lg p-4 border border-red-200">
                        <div className="flex items-center gap-2 mb-2">
                          <div className="w-2 h-2 bg-red-600 rounded-full"></div>
                          <span className="text-xs font-bold text-red-900 uppercase">Password</span>
                        </div>
                        <p className="text-xs text-red-700">Failed login attempts</p>
                      </div>
                      <div className="bg-orange-50 rounded-lg p-4 border border-orange-200">
                        <div className="flex items-center gap-2 mb-2">
                          <div className="w-2 h-2 bg-orange-600 rounded-full"></div>
                          <span className="text-xs font-bold text-orange-900 uppercase">Security</span>
                        </div>
                        <p className="text-xs text-orange-700">Unauthorized access</p>
                      </div>
                      <div className="bg-yellow-50 rounded-lg p-4 border border-yellow-200">
                        <div className="flex items-center gap-2 mb-2">
                          <div className="w-2 h-2 bg-yellow-600 rounded-full"></div>
                          <span className="text-xs font-bold text-yellow-900 uppercase">Document</span>
                        </div>
                        <p className="text-xs text-yellow-700">Verification failed</p>
                      </div>
                    </div>
                  </div>
                </div>
              ) : (
                <div className="bg-white rounded-xl border-2 border-gray-200 p-6">
                  {loading ? (
                    <div className="flex justify-center py-12">
                      <div className="animate-spin rounded-full h-12 w-12 border-4 border-red-600 border-t-transparent"></div>
                    </div>
                  ) : breachReport && breachReport.notifications && breachReport.notifications.length > 0 ? (
                    <div className="space-y-6">
                      <div className="flex items-center justify-between mb-6">
                        <div>
                          <h3 className="text-lg font-bold text-gray-900">Breach Notification History</h3>
                          <p className="text-sm text-gray-600 mt-1">Complete record of all breach notifications</p>
                        </div>
                        <button
                          onClick={downloadBreachReportPDF}
                          className="px-6 py-3 bg-red-600 hover:bg-red-700 text-white rounded-lg font-bold shadow-lg transition-all flex items-center gap-2"
                        >
                          <FileText className="w-4 h-4" />
                          Download PDF
                        </button>
                      </div>
                      
                      <div className="grid grid-cols-4 gap-4">
                        <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl p-6 border-2 border-blue-200">
                          <div className="text-sm font-bold text-blue-700 mb-2">Total Notifications</div>
                          <div className="text-4xl font-bold text-blue-900">{breachReport.statistics.total_notifications}</div>
                        </div>
                        <div className="bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl p-6 border-2 border-purple-200">
                          <div className="text-sm font-bold text-purple-700 mb-2">Unique Users</div>
                          <div className="text-4xl font-bold text-purple-900">{breachReport.statistics.unique_users}</div>
                        </div>
                        <div className="bg-gradient-to-br from-red-50 to-red-100 rounded-xl p-6 border-2 border-red-200">
                          <div className="text-sm font-bold text-red-700 mb-2">Password Breaches</div>
                          <div className="text-4xl font-bold text-red-900">{breachReport.statistics.password_breaches}</div>
                        </div>
                        <div className="bg-gradient-to-br from-orange-50 to-orange-100 rounded-xl p-6 border-2 border-orange-200">
                          <div className="text-sm font-bold text-orange-700 mb-2">Security Breaches</div>
                          <div className="text-4xl font-bold text-orange-900">{breachReport.statistics.security_breaches}</div>
                        </div>
                      </div>

                      <div className="grid gap-3 max-h-96 overflow-y-auto">
                        {breachReport.notifications.map((notif: any) => (
                          <div key={notif.id} className="bg-white border-2 border-gray-200 rounded-xl p-4 hover:shadow-lg transition-shadow">
                            <div className="flex items-start justify-between">
                              <div className="flex-1">
                                <div className="flex items-center gap-3 mb-2">
                                  <div className="w-10 h-10 rounded-full flex items-center justify-center text-white font-bold" style={{ backgroundColor: '#dc2626' }}>
                                    {notif.email.charAt(0).toUpperCase()}
                                  </div>
                                  <div>
                                    <div className="font-bold text-gray-900">{notif.email}</div>
                                    <div className="text-xs text-gray-500">{new Date(notif.sent_at).toLocaleString()}</div>
                                  </div>
                                </div>
                                <div className="flex items-center gap-2 mt-3">
                                  <span className={`px-3 py-1 text-xs font-bold rounded-full flex items-center gap-1 ${
                                    notif.breach_type === 'password' ? 'bg-red-100 text-red-700' :
                                    notif.breach_type === 'security' ? 'bg-orange-100 text-orange-700' :
                                    'bg-yellow-100 text-yellow-700'
                                  }`}>
                                    {notif.breach_type === 'password' && '🔒'}
                                    {notif.breach_type === 'security' && '⚠️'}
                                    {notif.breach_type === 'document' && '📄'}
                                    {notif.breach_type.toUpperCase()}
                                  </span>
                                  <span className="px-3 py-1 text-xs font-bold rounded-full bg-blue-100 text-blue-700 capitalize flex items-center gap-1">
                                    👤 {notif.user_type}
                                  </span>
                                  <span className={`px-3 py-1 text-xs font-bold rounded-full flex items-center gap-1 ${
                                    notif.acknowledged ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-700'
                                  }`}>
                                    {notif.acknowledged ? '✓ Acknowledged' : '📤 Sent'}
                                  </span>
                                </div>
                              </div>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  ) : (
                    <div className="text-center py-12">
                      <AlertTriangle className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                      <p className="text-gray-500 text-lg font-semibold">No breach notifications sent yet</p>
                      <p className="text-gray-400 text-sm mt-2">Send breach notifications from the "Send Breach Notification" tab</p>
                    </div>
                  )}
                </div>
              )}
            </div>
          ) : null}
        </div>
      </div>

      {/* Send Breach Notification Modal */}
      {showSendBreachModal && (
        <div className="fixed inset-0 flex items-center justify-center p-4" style={{ zIndex: 9999, backgroundColor: 'rgba(0, 0, 0, 0.5)' }}>
          <div className="bg-white rounded-2xl shadow-2xl w-full max-w-5xl max-h-[90vh] overflow-y-auto">
            <div className="bg-gradient-to-r from-red-600 to-orange-600 p-6 rounded-t-2xl sticky top-0">
              <h2 className="text-2xl font-bold text-white">Send Breach Notification</h2>
              <p className="text-red-100 mt-1">Select breach type and affected users</p>
            </div>
            <div className="p-6 space-y-6">
              <div>
                <label className="block text-sm font-bold text-gray-700 mb-3">Select Breach Type:</label>
                <select
                  value={selectedBreachType}
                  onChange={(e) => handleBreachTypeChange(e.target.value)}
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-lg focus:ring-2 focus:ring-red-500 focus:border-red-500 font-semibold text-gray-900"
                >
                  <option value="password">Password Related Breaches (5+ Failed Login Attempts)</option>
                  <option value="security">Security Related Breaches (Unauthorized Access Attempts)</option>
                  <option value="document">Document Upload Breaches (Below 80% Match)</option>
                </select>
              </div>

              {loading ? (
                <div className="flex justify-center py-12">
                  <div className="animate-spin rounded-full h-12 w-12 border-4 border-red-600 border-t-transparent"></div>
                </div>
              ) : breachUsers.length === 0 ? (
                <div className="bg-green-50 border-2 border-green-200 rounded-xl p-12 text-center">
                  <CheckCircle className="w-16 h-16 text-green-500 mx-auto mb-4" />
                  <p className="text-green-900 text-lg font-semibold">No users affected by this breach type</p>
                  <p className="text-green-700 text-sm mt-2">All users are secure</p>
                </div>
              ) : (
                <div className="border-2 border-gray-200 rounded-xl overflow-hidden">
                  <div className="bg-gradient-to-r from-red-50 to-orange-50 p-4 border-b-2 border-red-200 flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <AlertTriangle className="w-5 h-5 text-red-600" />
                      <span className="font-bold text-red-900">Affected Users: {breachUsers.length} | Selected: {selectedUsers.size}</span>
                    </div>
                  </div>
                  <div className="max-h-96 overflow-y-auto">
                    <table className="min-w-full divide-y divide-gray-200">
                      <thead className="bg-gray-50 sticky top-0">
                        <tr>
                          {selectedBreachType !== 'security' && (
                            <th className="px-6 py-4 text-left">
                              <input
                                type="checkbox"
                                checked={selectedUsers.size === breachUsers.length && breachUsers.length > 0}
                                onChange={toggleAllUsers}
                                className="w-4 h-4 text-red-600 rounded"
                              />
                            </th>
                          )}
                          <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">User</th>
                          <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Email</th>
                          <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Role</th>
                          <th className="px-6 py-4 text-left text-xs font-bold text-gray-700 uppercase">Breach Details</th>
                        </tr>
                      </thead>
                      <tbody className="bg-white divide-y divide-gray-200">
                        {breachUsers.map((user) => (
                          <tr key={user.user_id} className={`hover:bg-red-50 transition-colors ${selectedUsers.has(user.user_id) ? 'bg-red-50' : ''}`}>
                            {selectedBreachType !== 'security' && (
                              <td className="px-6 py-4">
                                <input
                                  type="checkbox"
                                  checked={selectedUsers.has(user.user_id)}
                                  onChange={() => toggleUserSelection(user.user_id)}
                                  className="w-4 h-4 text-red-600 rounded"
                                />
                              </td>
                            )}
                            <td className="px-6 py-4">
                              <div className="flex items-center gap-3">
                                <div className="w-10 h-10 rounded-full flex items-center justify-center text-white font-bold shadow" style={{ backgroundColor: '#dc2626' }}>
                                  {(user.email || 'U').charAt(0).toUpperCase()}
                                </div>
                              </div>
                            </td>
                            <td className="px-6 py-4">
                              <div className="text-sm font-semibold text-gray-900">{user.email || 'No email'}</div>
                            </td>
                            <td className="px-6 py-4">
                              <span className="px-3 py-1 text-xs font-bold rounded-md bg-gray-100 text-gray-700 capitalize">
                                {user.user_type || 'N/A'}
                              </span>
                            </td>
                            <td className="px-6 py-4">
                              <div className="text-sm text-gray-700">
                                {selectedBreachType === 'password' && (
                                  <span className="flex items-center gap-2">
                                    <span className="px-2 py-1 bg-red-100 text-red-700 rounded text-xs font-bold">
                                      {user.failed_attempts || 5}+ Failed Attempts
                                    </span>
                                    <span className="text-gray-500 text-xs">
                                      Last: {user.last_attempt ? new Date(user.last_attempt).toLocaleString() : 'Recently'}
                                    </span>
                                  </span>
                                )}
                                {selectedBreachType === 'security' && (
                                  <span className="flex items-center gap-2">
                                    <span className="px-2 py-1 bg-orange-100 text-orange-700 rounded text-xs font-bold">
                                      Unauthorized Access
                                    </span>
                                    <span className="text-gray-500 text-xs">
                                      IP: {user.ip_address || 'Unknown'}
                                    </span>
                                  </span>
                                )}
                                {selectedBreachType === 'document' && (
                                  <span className="flex items-center gap-2">
                                    <span className="px-2 py-1 bg-yellow-100 text-yellow-700 rounded text-xs font-bold">
                                      {user.match_percentage || 65}% Match
                                    </span>
                                    <span className="text-gray-500 text-xs">
                                      Doc: {user.document_type || 'ID Card'}
                                    </span>
                                  </span>
                                )}
                              </div>
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              )}
            </div>
            <div className="flex gap-3 p-6 bg-gray-50 rounded-b-2xl">
              {selectedBreachType !== 'security' && (
                <button
                  onClick={sendBreachNotifications}
                  disabled={selectedUsers.size === 0}
                  className="flex-1 px-6 py-3 bg-red-600 hover:bg-red-700 disabled:bg-gray-400 text-white rounded-lg font-bold shadow-lg transition-all flex items-center justify-center gap-2"
                >
                  <Mail className="w-5 h-5" />
                  Send Notification to {selectedUsers.size} User(s)
                </button>
              )}
              <button
                onClick={() => { setShowSendBreachModal(false); setSelectedUsers(new Set()); }}
                className="flex-1 px-6 py-3 border-2 border-gray-300 text-gray-700 rounded-lg font-semibold hover:bg-gray-100 transition-all"
              >
                {selectedBreachType === 'security' ? 'Close' : 'Cancel'}
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Create Form Modal */}
      {showCreateForm && (
        <div className="fixed inset-0 flex items-center justify-center p-4" style={{ zIndex: 9999, backgroundColor: 'rgba(0, 0, 0, 0.5)' }}>
          <div className="bg-white rounded-2xl shadow-2xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
            <div className="bg-gradient-to-r from-blue-600 to-indigo-600 p-6 rounded-t-2xl sticky top-0">
              <h2 className="text-2xl font-bold text-white">Create Compliance Rule</h2>
            </div>
            <div className="p-6 space-y-4" style={{ backgroundColor: '#ffffff' }}>
              <div>
                <label className="block text-sm font-semibold mb-2" style={{ color: '#1f2937' }}>Compliance Type</label>
                <select
                  value={newForm.type}
                  onChange={(e) => setNewForm({ ...newForm, type: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  style={{ color: '#111827', backgroundColor: 'white' }}
                >
                  <option value="terms">Terms & Conditions</option>
                  <option value="privacy">Privacy Policy</option>
                  <option value="data_sharing">Data Sharing</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-semibold mb-2" style={{ color: '#1f2937' }}>Title *</label>
                <input
                  type="text"
                  value={newForm.title}
                  onChange={(e) => setNewForm({ ...newForm, title: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  style={{ color: '#111827', backgroundColor: 'white' }}
                  placeholder="e.g., Privacy Policy"
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-semibold mb-2" style={{ color: '#374151' }}>Content *</label>
                <textarea
                  value={newForm.content}
                  onChange={(e) => setNewForm({ ...newForm, content: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg h-40 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  style={{ color: '#111827', backgroundColor: 'white' }}
                  placeholder="Enter compliance rule content..."
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-semibold mb-2" style={{ color: '#374151' }}>Version *</label>
                <input
                  type="text"
                  value={newForm.version}
                  onChange={(e) => setNewForm({ ...newForm, version: e.target.value })}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  style={{ color: '#111827', backgroundColor: 'white' }}
                  placeholder="1.0"
                  required
                />
              </div>
            </div>
            <div className="flex gap-3 p-6 bg-gray-50 rounded-b-2xl">
              <button
                onClick={createConsentForm}
                className="flex-1 px-6 py-3 rounded-lg font-semibold shadow-lg transition-all"
                style={{ backgroundColor: '#2563eb', color: 'white' }}
              >
                Create Rule
              </button>
              <button
                onClick={() => setShowCreateForm(false)}
                className="flex-1 px-6 py-3 rounded-lg font-semibold transition-all"
                style={{ backgroundColor: 'white', border: '2px solid #d1d5db', color: '#374151' }}
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Report Modal */}
      {showReport && selectedFormReport && (
        <div className="fixed inset-0 flex items-center justify-center p-4" style={{ zIndex: 9999, backgroundColor: 'rgba(0, 0, 0, 0.5)' }}>
          <div className="bg-white rounded-2xl shadow-2xl w-full max-w-4xl max-h-[90vh] overflow-y-auto">
            <div className="bg-gradient-to-r from-blue-600 to-indigo-600 p-6 rounded-t-2xl sticky top-0">
              <h2 className="text-2xl font-bold text-white">{selectedFormReport.title} - Report</h2>
              <p className="text-blue-100 mt-1">Version {selectedFormReport.version}</p>
            </div>
            <div className="p-6">
              <div className="grid grid-cols-3 gap-4 mb-6">
                <div className="bg-blue-50 rounded-lg p-4">
                  <div className="text-sm text-blue-600 font-semibold">Total Responses</div>
                  <div className="text-3xl font-bold text-blue-900 mt-1">{selectedFormReport.total_responses}</div>
                </div>
                <div className="bg-green-50 rounded-lg p-4">
                  <div className="text-sm text-green-600 font-semibold">Accepted</div>
                  <div className="text-3xl font-bold text-green-900 mt-1">{selectedFormReport.accepted_count}</div>
                </div>
                <div className="bg-red-50 rounded-lg p-4">
                  <div className="text-sm text-red-600 font-semibold">Rejected</div>
                  <div className="text-3xl font-bold text-red-900 mt-1">{selectedFormReport.rejected_count}</div>
                </div>
              </div>
              <div className="border border-gray-200 rounded-lg overflow-hidden">
                {selectedFormReport.users && selectedFormReport.users.length > 0 ? (
                  <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                      <tr>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Email</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Phone</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Role</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Status</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Date</th>
                      </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                      {selectedFormReport.users.map((user: any) => (
                        <tr key={user.user_id} className="hover:bg-gray-50">
                          <td className="px-4 py-3 text-sm text-gray-900">{user.email}</td>
                          <td className="px-4 py-3 text-sm text-gray-600">{user.phone_number || 'N/A'}</td>
                          <td className="px-4 py-3 text-sm">
                            <span className="px-2 py-1 text-xs font-semibold bg-blue-100 text-blue-700 rounded-full capitalize">
                              {user.role}
                            </span>
                          </td>
                          <td className="px-4 py-3 text-sm">
                            {user.accepted ? (
                              <span className="px-2 py-1 text-xs font-semibold bg-green-100 text-green-700 rounded-full flex items-center gap-1 w-fit">
                                <CheckCircle className="w-3 h-3" />Accepted
                              </span>
                            ) : (
                              <span className="px-2 py-1 text-xs font-semibold bg-red-100 text-red-700 rounded-full flex items-center gap-1 w-fit">
                                <AlertCircle className="w-3 h-3" />Rejected
                              </span>
                            )}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {user.accepted_at ? new Date(user.accepted_at).toLocaleDateString() : 'N/A'}
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                ) : (
                  <div className="text-center py-12">
                    <Users className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                    <p className="text-gray-500 text-lg">No users have responded to this consent form yet</p>
                  </div>
                )}
              </div>
            </div>
            <div className="flex gap-3 p-6 bg-gray-50 rounded-b-2xl">
              <button
                onClick={() => setShowReport(false)}
                className="flex-1 px-6 py-3 rounded-lg font-semibold transition-all"
                style={{ backgroundColor: '#2563eb', color: 'white' }}
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Breach Report Modal */}
      {showBreachReport && breachReport && (
        <div className="fixed inset-0 flex items-center justify-center p-4" style={{ zIndex: 9999, backgroundColor: 'rgba(0, 0, 0, 0.5)' }}>
          <div className="bg-white rounded-2xl shadow-2xl w-full max-w-6xl max-h-[90vh] overflow-y-auto">
            <div className="bg-gradient-to-r from-red-600 to-orange-600 p-6 rounded-t-2xl sticky top-0">
              <h2 className="text-2xl font-bold text-white">Breach Notification Report</h2>
              <p className="text-red-100 mt-1">Complete history of all breach notifications sent</p>
            </div>
            <div className="p-6">
              <div className="grid grid-cols-4 gap-4 mb-6">
                <div className="bg-blue-50 rounded-lg p-4">
                  <div className="text-sm text-blue-600 font-semibold">Total Notifications</div>
                  <div className="text-3xl font-bold text-blue-900 mt-1">{breachReport.statistics.total_notifications}</div>
                </div>
                <div className="bg-purple-50 rounded-lg p-4">
                  <div className="text-sm text-purple-600 font-semibold">Unique Users</div>
                  <div className="text-3xl font-bold text-purple-900 mt-1">{breachReport.statistics.unique_users}</div>
                </div>
                <div className="bg-red-50 rounded-lg p-4">
                  <div className="text-sm text-red-600 font-semibold">Password Breaches</div>
                  <div className="text-3xl font-bold text-red-900 mt-1">{breachReport.statistics.password_breaches}</div>
                </div>
                <div className="bg-orange-50 rounded-lg p-4">
                  <div className="text-sm text-orange-600 font-semibold">Security Breaches</div>
                  <div className="text-3xl font-bold text-orange-900 mt-1">{breachReport.statistics.security_breaches}</div>
                </div>
              </div>
              <div className="border border-gray-200 rounded-lg overflow-hidden">
                {breachReport.notifications && breachReport.notifications.length > 0 ? (
                  <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                      <tr>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Date</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Email</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Role</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Breach Type</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Sent By</th>
                        <th className="px-4 py-3 text-left text-xs font-bold text-gray-600 uppercase">Status</th>
                      </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                      {breachReport.notifications.map((notif: any) => (
                        <tr key={notif.id} className="hover:bg-gray-50">
                          <td className="px-4 py-3 text-sm text-gray-600">
                            {new Date(notif.sent_at).toLocaleString()}
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-900 font-semibold">{notif.email}</td>
                          <td className="px-4 py-3 text-sm">
                            <span className="px-2 py-1 text-xs font-semibold bg-blue-100 text-blue-700 rounded-full capitalize">
                              {notif.user_type}
                            </span>
                          </td>
                          <td className="px-4 py-3 text-sm">
                            <span className={`px-2 py-1 text-xs font-bold rounded-full ${
                              notif.breach_type === 'password' ? 'bg-red-100 text-red-700' :
                              notif.breach_type === 'security' ? 'bg-orange-100 text-orange-700' :
                              'bg-yellow-100 text-yellow-700'
                            }`}>
                              {notif.breach_type}
                            </span>
                          </td>
                          <td className="px-4 py-3 text-sm text-gray-600">{notif.sent_by || 'Admin'}</td>
                          <td className="px-4 py-3 text-sm">
                            <span className={`px-2 py-1 text-xs font-semibold rounded-full ${
                              notif.acknowledged ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-700'
                            }`}>
                              {notif.acknowledged ? 'Acknowledged' : 'Sent'}
                            </span>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                ) : (
                  <div className="text-center py-12">
                    <AlertTriangle className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                    <p className="text-gray-500 text-lg">No breach notifications sent yet</p>
                  </div>
                )}
              </div>
            </div>
            <div className="flex gap-3 p-6 bg-gray-50 rounded-b-2xl">
              <button
                onClick={() => setShowBreachReport(false)}
                className="flex-1 px-6 py-3 rounded-lg font-semibold transition-all"
                style={{ backgroundColor: '#dc2626', color: 'white' }}
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Add/Edit Document Modal */}
      {(showAddDoc || editingDoc) && (
        <div className="fixed inset-0 flex items-center justify-center p-4" style={{ zIndex: 9999, backgroundColor: 'rgba(0, 0, 0, 0.5)' }}>
          <div className="bg-white rounded-2xl shadow-2xl w-full max-w-2xl">
            <div className="bg-gradient-to-r from-blue-600 to-indigo-600 p-6 rounded-t-2xl">
              <h2 className="text-2xl font-bold text-white">{editingDoc ? 'Edit' : 'Add'} Document Requirement</h2>
            </div>
            <div className="p-6 space-y-4">
              <div>
                <label className="block text-sm font-semibold mb-2 text-gray-700">User Role</label>
                <select
                  value={editingDoc ? editingDoc.user_type : newDoc.user_type}
                  onChange={(e) => editingDoc ? setEditingDoc({...editingDoc, user_type: e.target.value}) : setNewDoc({...newDoc, user_type: e.target.value})}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg"
                  disabled={!!editingDoc}
                >
                  <option value="buyer">Buyer</option>
                  <option value="seller">Seller</option>
                  <option value="agent">Agent</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-semibold mb-2 text-gray-700">Document Type (ID)</label>
                <input
                  type="text"
                  value={editingDoc ? editingDoc.document_type : newDoc.document_type}
                  onChange={(e) => editingDoc ? setEditingDoc({...editingDoc, document_type: e.target.value}) : setNewDoc({...newDoc, document_type: e.target.value})}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg"
                  placeholder="e.g., id_card"
                  disabled={!!editingDoc}
                />
              </div>
              <div>
                <label className="block text-sm font-semibold mb-2 text-gray-700">Document Name</label>
                <input
                  type="text"
                  value={editingDoc ? editingDoc.document_name : newDoc.document_name}
                  onChange={(e) => editingDoc ? setEditingDoc({...editingDoc, document_name: e.target.value}) : setNewDoc({...newDoc, document_name: e.target.value})}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg"
                  placeholder="e.g., ID Card"
                />
              </div>
              <div>
                <label className="block text-sm font-semibold mb-2 text-gray-700">Description</label>
                <textarea
                  value={editingDoc ? editingDoc.description : newDoc.description}
                  onChange={(e) => editingDoc ? setEditingDoc({...editingDoc, description: e.target.value}) : setNewDoc({...newDoc, description: e.target.value})}
                  className="w-full px-4 py-3 border border-gray-300 rounded-lg h-24"
                  placeholder="Document description"
                />
              </div>
              <div className="flex items-center gap-2">
                <input
                  type="checkbox"
                  checked={editingDoc ? editingDoc.is_required : newDoc.is_required}
                  onChange={(e) => editingDoc ? setEditingDoc({...editingDoc, is_required: e.target.checked}) : setNewDoc({...newDoc, is_required: e.target.checked})}
                  className="w-4 h-4"
                />
                <label className="text-sm font-semibold text-gray-700">Required Document</label>
              </div>
            </div>
            <div className="flex gap-3 p-6 bg-gray-50 rounded-b-2xl">
              <button
                onClick={editingDoc ? updateDocRequirement : createDocRequirement}
                className="flex-1 px-6 py-3 bg-blue-600 text-white rounded-lg font-semibold"
              >
                {editingDoc ? 'Update' : 'Create'}
              </button>
              <button
                onClick={() => { setShowAddDoc(false); setEditingDoc(null); }}
                className="flex-1 px-6 py-3 border-2 border-gray-300 text-gray-700 rounded-lg font-semibold"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
    </>
  );
}





