import React, { useState } from "react";

// TypeScript interfaces
interface FAQ {
  id: number;
  question: string;
  answer: string;
  status: 'published' | 'draft';
  views: number;
}

interface Policy {
  id: number;
  title: string;
  description: string;
  status: 'published' | 'draft';
}

interface Testimonial {
  id: number;
  name: string;
  role: string;
  message: string;
  status: 'published' | 'draft';
}

interface Blog {
  id: number;
  title: string;
  description: string;
  category: 'INVESTMENT' | 'DOCUMENTATION' | 'PRICING';
  status: 'published' | 'draft';
  views: number;
  date: string;
}

type ContentItem = FAQ | Policy | Testimonial | Blog;

export function ContentCMS() {
  const [activeTab, setActiveTab] = useState('faqs');
  const [faqs, setFaqs] = useState<FAQ[]>([
    { id: 1, question: 'How do I list my property?', answer: 'Step-by-step guide for property listing with photos, documents, and pricing', status: 'published', views: 1234 },
    { id: 2, question: 'What documents are required?', answer: 'Complete list of required documents for property verification and legal compliance', status: 'draft', views: 0 }
  ]);
  const [policies, setPolicies] = useState<Policy[]>([
    { id: 1, title: 'Privacy Policy', description: 'Comprehensive data protection and privacy guidelines for user information security', status: 'published' },
    { id: 2, title: 'Terms of Service', description: 'Platform usage terms, conditions, and user responsibilities for service access', status: 'published' },
    { id: 3, title: 'Refund Policy', description: 'Payment processing, refund guidelines, and financial transaction policies', status: 'draft' }
  ]);
  const [testimonials, setTestimonials] = useState<Testimonial[]>([
    { id: 1, name: 'Amit Singh', role: 'Property Seller', message: 'Sold my 3BHK apartment in just 2 weeks through New Age Land. The team\'s support was exceptional and the process was seamless!', status: 'published' },
    { id: 2, name: 'Priya Sharma', role: 'Property Buyer', message: 'Found my dream home in Bangalore with amazing support from the team. The virtual tours saved so much time!', status: 'published' },
    { id: 3, name: 'Rajesh Kumar', role: 'Real Estate Agent', message: 'Great platform for connecting with serious buyers and sellers. My sales increased by 40% in 3 months!', status: 'published' }
  ]);
  const [blogs, setBlogs] = useState<Blog[]>(() => {
    const defaultBlogs: Blog[] = [
      { id: 1, title: 'Top 10 Real Estate Investment Tips for 2024', description: 'Expert advice on property investment strategies, market trends, and profitable opportunities in Indian real estate market', category: 'INVESTMENT', status: 'published', views: 2341, date: 'Jan 15, 2024' },
      { id: 2, title: 'Understanding Property Documentation in India', description: 'Complete guide to property papers, legal requirements, and documentation process for buyers and sellers', category: 'DOCUMENTATION', status: 'published', views: 856, date: 'Jan 10, 2024' },
      { id: 3, title: 'How to Price Your Property Right', description: 'Market analysis and pricing strategies for sellers to maximize returns and attract genuine buyers', category: 'PRICING', status: 'published', views: 1876, date: 'Jan 5, 2024' }
    ];
    const savedBlogs = localStorage.getItem('cms-blogs');
    if (!savedBlogs) {
      localStorage.setItem('cms-blogs', JSON.stringify(defaultBlogs));
      return defaultBlogs;
    }
    const parsedBlogs = JSON.parse(savedBlogs);
    return parsedBlogs.map((blog: any) => ({
      ...blog,
      status: blog.status === 'published' ? 'published' : 'draft'
    })) as Blog[];
  });
  const [showModal, setShowModal] = useState(false);
  const [modalType, setModalType] = useState('');
  const [editingItem, setEditingItem] = useState<ContentItem | null>(null);
  const [editingId, setEditingId] = useState<number | null>(null);
  const [editingType, setEditingType] = useState('');

  const handleAddNew = (type: string) => {
    setModalType(`add-${type.toLowerCase()}`);
    setEditingItem(null);
    setShowModal(true);
  };

  const handleEdit = (type: string, id: number) => {
    setEditingId(id);
    setEditingType(type.toLowerCase());
  };

  const handleSaveInline = (type: string, id: number, formData: Record<string, string>) => {
    if (type === 'faq') {
      setFaqs(faqs.map(f => f.id === id ? { ...f, ...formData } : f));
    } else if (type === 'policy') {
      setPolicies(policies.map(p => p.id === id ? { ...p, ...formData } : p));
    } else if (type === 'blog') {
      const updatedBlogs = blogs.map(b => b.id === id ? { ...b, ...formData } : b);
      setBlogs(updatedBlogs);
      localStorage.setItem('cms-blogs', JSON.stringify(updatedBlogs));
    }
    setEditingId(null);
    setEditingType('');
  };

  const handleCancelInline = () => {
    setEditingId(null);
    setEditingType('');
  };

  const handleEditModal = (type: string, id: number) => {
    let item: ContentItem | undefined;
    if (type === 'FAQ') item = faqs.find(f => f.id === id);
    else if (type === 'Policy') item = policies.find(p => p.id === id);
    else if (type === 'Testimonial') item = testimonials.find(t => t.id === id);
    else if (type === 'Blog') item = blogs.find(b => b.id === id);
    
    setModalType(`edit-${type.toLowerCase()}`);
    setEditingItem(item || null);
    setShowModal(true);
  };

  const handleCloseModal = () => {
    setShowModal(false);
    setModalType('');
    setEditingItem(null);
  };

  const handleSave = (formData: Record<string, string>) => {
    if (modalType.includes('faq')) {
      if (modalType.includes('add')) {
        const newFaq: FAQ = { 
          id: Date.now(), 
          question: formData.question || '', 
          answer: formData.answer || '', 
          status: 'draft', 
          views: 0 
        };
        setFaqs([...faqs, newFaq]);
      } else {
        setFaqs(faqs.map(f => f.id === editingItem?.id ? { ...f, ...formData } : f));
      }
    } else if (modalType.includes('policy')) {
      if (modalType.includes('add')) {
        const newPolicy: Policy = { 
          id: Date.now(), 
          title: formData.title || '', 
          description: formData.description || '', 
          status: 'draft' 
        };
        setPolicies([...policies, newPolicy]);
      } else {
        setPolicies(policies.map(p => p.id === editingItem?.id ? { ...p, ...formData } : p));
      }
    } else if (modalType.includes('testimonial')) {
      if (modalType.includes('add')) {
        const newTestimonial: Testimonial = { 
          id: Date.now(), 
          name: formData.name || '', 
          role: formData.role || '', 
          message: formData.message || '', 
          status: 'draft' 
        };
        setTestimonials([...testimonials, newTestimonial]);
      } else {
        setTestimonials(testimonials.map(t => t.id === editingItem?.id ? { ...t, ...formData } : t));
      }
    } else if (modalType.includes('blog')) {
      if (modalType.includes('add')) {
        const newBlog: Blog = { 
          id: Date.now(), 
          title: formData.title || '', 
          description: formData.description || '', 
          category: (formData.category as Blog['category']) || 'INVESTMENT', 
          status: 'draft', 
          views: 0, 
          date: new Date().toLocaleDateString() 
        };
        const updatedBlogs = [...blogs, newBlog];
        setBlogs(updatedBlogs);
        localStorage.setItem('cms-blogs', JSON.stringify(updatedBlogs));
      } else {
        const updatedBlogs = blogs.map(b => b.id === editingItem?.id ? { ...b, ...formData } : b);
        setBlogs(updatedBlogs);
        localStorage.setItem('cms-blogs', JSON.stringify(updatedBlogs));
      }
    }
    handleCloseModal();
  };

  const handleStatusToggle = (type: string, id: number) => {
    if (type === 'faqs') {
      setFaqs(faqs.map(faq => 
        faq.id === id ? { ...faq, status: faq.status === 'published' ? 'draft' : 'published' } : faq
      ));
    } else if (type === 'policies') {
      setPolicies(policies.map(policy => 
        policy.id === id ? { ...policy, status: policy.status === 'published' ? 'draft' : 'published' } : policy
      ));
    } else if (type === 'testimonials') {
      setTestimonials(testimonials.map(testimonial => 
        testimonial.id === id ? { ...testimonial, status: testimonial.status === 'published' ? 'draft' : 'published' } : testimonial
      ));
    } else if (type === 'blogs') {
      const updatedBlogs = blogs.map(blog => 
        blog.id === id ? { ...blog, status: (blog.status === 'published' ? 'draft' : 'published') as Blog['status'] } : blog
      );
      setBlogs(updatedBlogs);
      localStorage.setItem('cms-blogs', JSON.stringify(updatedBlogs));
    }
  };

  return (
    <div className="space-y-6">
      <h1 className="text-2xl font-bold text-gray-900">Content & CMS</h1>
      
      {/* Content Tabs */}
      <div className="bg-white rounded-lg shadow">
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            <button 
              onClick={() => setActiveTab('faqs')}
              className={`border-b-2 py-4 px-1 text-sm font-medium ${
                activeTab === 'faqs' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700'
              }`}
            >
              FAQs
            </button>
            <button 
              onClick={() => setActiveTab('policies')}
              className={`border-b-2 py-4 px-1 text-sm font-medium ${
                activeTab === 'policies' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700'
              }`}
            >
              Policies
            </button>
            <button 
              onClick={() => setActiveTab('testimonials')}
              className={`border-b-2 py-4 px-1 text-sm font-medium ${
                activeTab === 'testimonials' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700'
              }`}
            >
              Testimonials
            </button>
            <button 
              onClick={() => setActiveTab('blogs')}
              className={`border-b-2 py-4 px-1 text-sm font-medium ${
                activeTab === 'blogs' ? 'border-blue-500 text-blue-600' : 'border-transparent text-gray-500 hover:text-gray-700'
              }`}
            >
              Blogs
            </button>
          </nav>
        </div>
        
        {/* Tab Content */}
        <div className="p-6">
          {activeTab === 'faqs' && (
            <div>
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                    <span className="text-blue-600 font-bold text-lg">?</span>
                  </div>
                  <h2 className="text-xl font-bold text-gray-900">Frequently Asked Questions</h2>
                </div>
                <button 
                  onClick={() => handleAddNew('FAQ')}
                  className="bg-gradient-to-r from-blue-600 to-blue-700 text-white px-6 py-2 rounded-lg text-sm font-medium shadow-md hover:shadow-lg transition-all"
                >
                  + Add New FAQ
                </button>
              </div>
              <div className="grid gap-4">
                {faqs.map((faq) => (
                  <div key={faq.id} className={`bg-gradient-to-r p-6 rounded-r-lg shadow-sm border-l-4 ${
                    faq.status === 'published' 
                      ? 'from-blue-50 to-indigo-50 border-blue-500' 
                      : 'from-amber-50 to-orange-50 border-amber-500'
                  }`}>
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        {editingId === faq.id && editingType === 'faq' ? (
                          <form id={`faq-edit-${faq.id}`} className="space-y-3">
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">Question</label>
                              <input 
                                name="question" 
                                defaultValue={faq.question} 
                                className="w-full p-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" 
                                required 
                              />
                            </div>
                            <div>
                              <label className="block text-sm font-medium text-gray-700 mb-1">Answer</label>
                              <textarea 
                                name="answer" 
                                defaultValue={faq.answer} 
                                className="w-full p-2 border border-gray-300 rounded-lg h-20 focus:ring-2 focus:ring-blue-500 focus:border-transparent" 
                                required
                              ></textarea>
                            </div>
                          </form>
                        ) : (
                          <>
                            <h3 className="font-semibold text-gray-900 text-lg mb-2">🏠 {faq.question}</h3>
                            <p className="text-gray-700 mb-3">{faq.answer}</p>
                            <div className="flex items-center gap-2 text-sm text-gray-600">
                              <span>📅 {faq.status === 'published' ? 'Last updated' : 'Draft saved'}: Jan 15, 2024</span>
                              <span>•</span>
                              <span>👀 {faq.views} views</span>
                            </div>
                          </>
                        )}
                      </div>
                      <div className="flex items-center gap-3 ml-4">
                        <button 
                          onClick={() => handleStatusToggle('faqs', faq.id)}
                          className={`px-3 py-1 rounded-full text-xs font-medium ${
                            faq.status === 'published' 
                              ? 'bg-green-500 text-white' 
                              : 'bg-amber-500 text-white'
                          }`}
                        >
                          {faq.status === 'published' ? '✓ Published' : '📝 Draft'}
                        </button>
                        {editingId === faq.id && editingType === 'faq' ? (
                          <div className="flex gap-2">
                            <button 
                              onClick={() => {
                                const form = document.getElementById(`faq-edit-${faq.id}`) as HTMLFormElement;
                                if (form) {
                                  const formData = new FormData(form);
                                  handleSaveInline('faq', faq.id, {
                                    question: formData.get('question') as string,
                                    answer: formData.get('answer') as string
                                  });
                                }
                              }}
                              className="bg-purple-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-purple-700 transition-colors"
                            >
                              Save
                            </button>
                            <button 
                              onClick={handleCancelInline}
                              className="bg-gray-300 text-gray-700 px-4 py-2 rounded-lg text-sm font-medium hover:bg-gray-400 transition-colors"
                            >
                              Cancel
                            </button>
                          </div>
                        ) : (
                          <button 
                            onClick={() => handleEdit('FAQ', faq.id)}
                            className={`bg-white px-4 py-2 rounded-lg text-sm font-medium border transition-colors ${
                              faq.status === 'published'
                                ? 'text-blue-600 border-blue-200 hover:bg-blue-50'
                                : 'text-amber-600 border-amber-200 hover:bg-amber-50'
                            }`}
                          >
                            Edit
                          </button>
                        )}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}

          {activeTab === 'policies' && (
            <div>
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-purple-100 rounded-full flex items-center justify-center">
                    <span className="text-purple-600 font-bold text-lg">⚖️</span>
                  </div>
                  <h2 className="text-xl font-bold text-gray-900">Legal Policies & Terms</h2>
                </div>
                <button 
                  onClick={() => handleAddNew('Policy')}
                  className="bg-gradient-to-r from-purple-600 to-purple-700 text-white px-6 py-2 rounded-lg text-sm font-medium shadow-md hover:shadow-lg transition-all"
                >
                  + Add New Policy
                </button>
              </div>
              <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                {policies.map((policy, index) => {
                  const colors = [
                    { border: 'border-green-200', bg: 'bg-green-100', text: 'text-green-600', icon: '🔒', btnBg: 'bg-green-100', btnText: 'text-green-700', btnHover: 'hover:bg-green-200', statusBg: 'bg-green-500' },
                    { border: 'border-blue-200', bg: 'bg-blue-100', text: 'text-blue-600', icon: '📋', btnBg: 'bg-blue-100', btnText: 'text-blue-700', btnHover: 'hover:bg-blue-200', statusBg: 'bg-blue-500' },
                    { border: 'border-orange-200', bg: 'bg-orange-100', text: 'text-orange-600', icon: '💰', btnBg: 'bg-orange-100', btnText: 'text-orange-700', btnHover: 'hover:bg-orange-200', statusBg: 'bg-orange-500' }
                  ];
                  const color = colors[index] || colors[0];
                  return (
                    <div key={policy.id} className={`bg-white border-2 ${color.border} rounded-xl p-6 shadow-lg hover:shadow-xl transition-all`}>
                      <div className="flex items-center gap-3 mb-4">
                        <div className={`w-12 h-12 ${color.bg} rounded-full flex items-center justify-center`}>
                          <span className="text-2xl">{color.icon}</span>
                        </div>
                        <div className="flex-1">
                          {editingId === policy.id && editingType === 'policy' ? (
                            <form id={`policy-edit-${policy.id}`} className="space-y-2">
                              <input 
                                name="title" 
                                defaultValue={policy.title} 
                                className="w-full p-2 border border-gray-300 rounded-lg text-lg font-bold focus:ring-2 focus:ring-purple-500 focus:border-transparent" 
                                required 
                              />
                            </form>
                          ) : (
                            <>
                              <h3 className="font-bold text-gray-900 text-lg">{policy.title}</h3>
                              <p className={`text-sm ${color.text} font-medium`}>{policy.title.split(' ')[0]} {policy.title.split(' ')[1] || ''}</p>
                            </>
                          )}
                        </div>
                      </div>
                      {editingId === policy.id && editingType === 'policy' ? (
                        <div className="mb-4">
                          <textarea 
                            name="description" 
                            defaultValue={policy.description} 
                            className="w-full p-2 border border-gray-300 rounded-lg h-20 focus:ring-2 focus:ring-purple-500 focus:border-transparent" 
                            required
                            form={`policy-edit-${policy.id}`}
                          ></textarea>
                        </div>
                      ) : (
                        <p className="text-gray-600 mb-4">{policy.description}</p>
                      )}
                      <div className="flex items-center justify-between">
                        <button 
                          onClick={() => handleStatusToggle('policies', policy.id)}
                          className={`${policy.status === 'published' ? color.statusBg : 'bg-orange-500'} text-white px-3 py-1 rounded-full text-xs font-bold`}
                        >
                          {policy.status === 'published' ? '✓ LIVE' : '📝 DRAFT'}
                        </button>
                        {editingId === policy.id && editingType === 'policy' ? (
                          <div className="flex gap-2">
                            <button 
                              onClick={() => {
                                const form = document.getElementById(`policy-edit-${policy.id}`) as HTMLFormElement;
                                if (form) {
                                  const formData = new FormData(form);
                                  handleSaveInline('policy', policy.id, {
                                    title: formData.get('title') as string,
                                    description: formData.get('description') as string
                                  });
                                }
                              }}
                              className="bg-purple-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-purple-700 transition-colors"
                            >
                              Save
                            </button>
                            <button 
                              onClick={handleCancelInline}
                              className="bg-gray-300 text-gray-700 px-4 py-2 rounded-lg text-sm font-medium hover:bg-gray-400 transition-colors"
                            >
                              Cancel
                            </button>
                          </div>
                        ) : (
                          <button 
                            onClick={() => handleEdit('Policy', policy.id)}
                            className={`${color.btnBg} ${color.btnText} px-4 py-2 rounded-lg text-sm font-medium ${color.btnHover} transition-colors`}
                          >
                            {policy.status === 'published' ? 'Edit Policy' : 'Complete'}
                          </button>
                        )}
                      </div>
                    </div>
                  );
                })}
              </div>
            </div>
          )}

          {activeTab === 'testimonials' && (
            <div>
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-yellow-100 rounded-full flex items-center justify-center">
                    <span className="text-yellow-600 font-bold text-lg">⭐</span>
                  </div>
                  <h2 className="text-xl font-bold text-gray-900">Success Stories & Testimonials</h2>
                </div>
                <button 
                  onClick={() => handleAddNew('Testimonial')}
                  className="bg-gradient-to-r from-yellow-500 to-orange-500 text-white px-6 py-2 rounded-lg text-sm font-medium shadow-md hover:shadow-lg transition-all"
                >
                  + Add New Story
                </button>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {testimonials.map((testimonial, index) => {
                  const colors = [
                    { bg: 'from-green-50 to-emerald-50', border: 'border-green-200', avatar: 'from-green-400 to-green-600', text: 'text-green-600', statusBg: 'bg-green-500', btnBg: 'bg-green-100', btnText: 'text-green-700', btnHover: 'hover:bg-green-200' },
                    { bg: 'from-blue-50 to-indigo-50', border: 'border-blue-200', avatar: 'from-blue-400 to-blue-600', text: 'text-blue-600', statusBg: 'bg-blue-500', btnBg: 'bg-blue-100', btnText: 'text-blue-700', btnHover: 'hover:bg-blue-200' },
                    { bg: 'from-purple-50 to-pink-50', border: 'border-purple-200', avatar: 'from-purple-400 to-purple-600', text: 'text-purple-600', statusBg: 'bg-purple-500', btnBg: 'bg-purple-100', btnText: 'text-purple-700', btnHover: 'hover:bg-purple-200' }
                  ];
                  const color = colors[index] || colors[0];
                  const initials = testimonial.name.split(' ').map(n => n[0]).join('');
                  const roleEmoji = testimonial.role.includes('Seller') ? '🏠' : testimonial.role.includes('Buyer') ? '🏡' : '🏢';
                  return (
                    <div key={testimonial.id} className={`bg-gradient-to-br ${color.bg} border ${color.border} rounded-2xl p-6 shadow-lg hover:shadow-xl transition-all transform hover:-translate-y-1`}>
                      <div className="flex items-center gap-4 mb-4">
                        <div className={`w-16 h-16 bg-gradient-to-br ${color.avatar} rounded-full flex items-center justify-center text-white font-bold text-xl`}>{initials}</div>
                        <div>
                          <h4 className="font-bold text-gray-900 text-lg">{testimonial.name}</h4>
                          <p className={`${color.text} font-medium text-sm`}>{roleEmoji} {testimonial.role}</p>
                          <div className="flex text-yellow-400 text-sm mt-1">
                            ⭐⭐⭐⭐⭐
                          </div>
                        </div>
                      </div>
                      <div className="bg-white rounded-lg p-4 mb-4 shadow-sm">
                        <p className="text-gray-700 italic">"{testimonial.message}"</p>
                      </div>
                    </div>
                  );
                })}
              </div>
            </div>
          )}

          {activeTab === 'blogs' && (
            <div>
              <div className="flex items-center justify-between mb-6">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-indigo-100 rounded-full flex items-center justify-center">
                    <span className="text-indigo-600 font-bold text-lg">📝</span>
                  </div>
                  <h2 className="text-xl font-bold text-gray-900">Blog Posts & Articles</h2>
                </div>
                <button 
                  onClick={() => handleAddNew('Blog')}
                  className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white px-6 py-2 rounded-lg text-sm font-medium shadow-md hover:shadow-lg transition-all"
                >
                  + Create New Blog
                </button>
              </div>
              <div className="grid gap-6">
                {blogs.map((blog, index) => {
                  const colors = [
                    { bg: 'from-green-400 to-emerald-500', icon: '💰', categoryBg: 'bg-green-100', categoryText: 'text-green-700', statusBg: 'bg-green-500', btnBg: 'bg-green-100', btnText: 'text-green-700', btnHover: 'hover:bg-green-200', accent: 'text-green-600' },
                    { bg: 'from-blue-400 to-indigo-500', icon: '📄', categoryBg: 'bg-blue-100', categoryText: 'text-blue-700', statusBg: 'bg-orange-500', btnBg: 'bg-orange-100', btnText: 'text-orange-700', btnHover: 'hover:bg-orange-200', accent: 'text-blue-600' },
                    { bg: 'from-purple-400 to-pink-500', icon: '💲', categoryBg: 'bg-purple-100', categoryText: 'text-purple-700', statusBg: 'bg-purple-500', btnBg: 'bg-purple-100', btnText: 'text-purple-700', btnHover: 'hover:bg-purple-200', accent: 'text-purple-600' }
                  ];
                  const color = colors[index] || colors[0];
                  return (
                    <div key={blog.id} className="bg-white border border-gray-200 rounded-2xl overflow-hidden shadow-lg hover:shadow-xl transition-all">
                      <div className="flex">
                        <div className={`w-48 h-32 bg-gradient-to-br ${color.bg} flex items-center justify-center text-white text-4xl font-bold`}>{color.icon}</div>
                        <div className="flex-1 p-6">
                          <div className="flex items-start justify-between">
                            <div className="flex-1">
                              {editingId === blog.id && editingType === 'blog' ? (
                                <form id={`blog-edit-${blog.id}`} className="space-y-3">
                                  <div>
                                    <select 
                                      name="category" 
                                      defaultValue={blog.category} 
                                      className="px-3 py-1 rounded-full text-xs font-bold border border-gray-300 focus:ring-2 focus:ring-purple-500 focus:border-transparent"
                                      required
                                    >
                                      <option value="INVESTMENT">INVESTMENT</option>
                                      <option value="DOCUMENTATION">DOCUMENTATION</option>
                                      <option value="PRICING">PRICING</option>
                                    </select>
                                  </div>
                                  <input 
                                    name="title" 
                                    defaultValue={blog.title} 
                                    className="w-full p-2 border border-gray-300 rounded-lg text-xl font-bold focus:ring-2 focus:ring-purple-500 focus:border-transparent" 
                                    required 
                                  />
                                  <textarea 
                                    name="description" 
                                    defaultValue={blog.description} 
                                    className="w-full p-2 border border-gray-300 rounded-lg h-20 focus:ring-2 focus:ring-purple-500 focus:border-transparent" 
                                    required
                                  ></textarea>
                                </form>
                              ) : (
                                <>
                                  <div className="flex items-center gap-2 mb-2">
                                    <span className={`${color.categoryBg} ${color.categoryText} px-3 py-1 rounded-full text-xs font-bold`}>{blog.category}</span>
                                    <span className="text-gray-400 text-xs">•</span>
                                    <span className="text-gray-500 text-xs">📅 {blog.date}</span>
                                    <span className="text-gray-400 text-xs">•</span>
                                    <span className="text-gray-500 text-xs">👀 {blog.views} views</span>
                                  </div>
                                  <h3 className="font-bold text-gray-900 text-xl mb-2">{blog.title}</h3>
                                  <p className="text-gray-600 mb-3">{blog.description}</p>
                                </>
                              )}
                              <div className="flex items-center gap-4 text-sm">
                                <span className={`${color.accent} font-medium`}>📈 {blog.category === 'INVESTMENT' ? 'Trending' : blog.category === 'DOCUMENTATION' ? 'Guide' : 'Strategy'}</span>
                                <span className="text-gray-500">✍️ By {blog.category === 'INVESTMENT' ? 'Admin' : blog.category === 'DOCUMENTATION' ? 'Legal' : 'Market'} Team</span>
                              </div>
                            </div>
                            <div className="flex flex-col gap-2 ml-4">
                              <button 
                                onClick={() => handleStatusToggle('blogs', blog.id)}
                                className={`${blog.status === 'published' ? color.statusBg : 'bg-orange-500'} text-white px-3 py-1 rounded-full text-xs font-bold`}
                              >
                                {blog.status === 'published' ? '✓ PUBLISHED' : '📝 DRAFT'}
                              </button>
                              {editingId === blog.id && editingType === 'blog' ? (
                                <div className="flex gap-2">
                                  <button 
                                    onClick={() => {
                                      const form = document.getElementById(`blog-edit-${blog.id}`) as HTMLFormElement;
                                      if (form) {
                                        const formData = new FormData(form);
                                        handleSaveInline('blog', blog.id, {
                                          title: formData.get('title') as string,
                                          description: formData.get('description') as string,
                                          category: formData.get('category') as string
                                        });
                                      }
                                    }}
                                    className="bg-purple-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-purple-700 transition-colors"
                                  >
                                    Save
                                  </button>
                                  <button 
                                    onClick={handleCancelInline}
                                    className="bg-gray-300 text-gray-700 px-4 py-2 rounded-lg text-sm font-medium hover:bg-gray-400 transition-colors"
                                  >
                                    Cancel
                                  </button>
                                </div>
                              ) : (
                                <button 
                                  onClick={() => handleEdit('Blog', blog.id)}
                                  className={`${color.btnBg} ${color.btnText} px-4 py-2 rounded-lg text-sm font-medium ${color.btnHover} transition-colors`}
                                >
                                  {blog.status === 'published' ? 'Edit Post' : 'Continue'}
                                </button>
                              )}
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                  );
                })}
              </div>
            </div>
          )}
        </div>
      </div>

      {/* Modal */}
      {showModal && (
        <div className="fixed inset-0 bg-white bg-opacity-20 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg w-full max-w-md max-h-[90vh] shadow-2xl flex flex-col">
            <div className="flex items-center justify-between p-6 pb-4 border-b">
              <h3 className="text-lg font-bold">
                {modalType.includes('add') ? 'Add New' : 'Edit'} {modalType.includes('faq') ? 'FAQ' : modalType.includes('policy') ? 'Policy' : modalType.includes('testimonial') ? 'Testimonial' : 'Blog'}
              </h3>
              <button onClick={handleCloseModal} className="text-gray-500 hover:text-gray-700">✕</button>
            </div>
            <div className="flex-1 overflow-y-auto p-6">
            
            {modalType.includes('faq') && (
              <form id="faq-form" onSubmit={(e) => {
                e.preventDefault();
                const formData = new FormData(e.target as HTMLFormElement);
                handleSave({
                  question: formData.get('question') as string,
                  answer: formData.get('answer') as string
                });
              }}>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Question</label>
                  <input name="question" defaultValue={(editingItem as FAQ)?.question || ''} className="w-full p-2 border rounded" required />
                </div>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Answer</label>
                  <textarea name="answer" defaultValue={(editingItem as FAQ)?.answer || ''} className="w-full p-2 border rounded h-24" required></textarea>
                </div>
              </form>
            )}
            
            {modalType.includes('policy') && (
              <form id="policy-form" onSubmit={(e) => {
                e.preventDefault();
                const formData = new FormData(e.target as HTMLFormElement);
                handleSave({
                  title: formData.get('title') as string,
                  description: formData.get('description') as string
                });
              }}>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Policy Title</label>
                  <input name="title" defaultValue={(editingItem as Policy)?.title || ''} className="w-full p-2 border rounded" required />
                </div>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Description</label>
                  <textarea name="description" defaultValue={(editingItem as Policy)?.description || ''} className="w-full p-2 border rounded h-24" required></textarea>
                </div>
              </form>
            )}
            
            {modalType.includes('testimonial') && (
              <form id="testimonial-form" onSubmit={(e) => {
                e.preventDefault();
                const formData = new FormData(e.target as HTMLFormElement);
                handleSave({
                  name: formData.get('name') as string,
                  role: formData.get('role') as string,
                  message: formData.get('message') as string
                });
              }}>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Name</label>
                  <input name="name" defaultValue={(editingItem as Testimonial)?.name || ''} className="w-full p-2 border rounded" required />
                </div>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Role</label>
                  <select name="role" defaultValue={(editingItem as Testimonial)?.role || ''} className="w-full p-2 border rounded" required>
                    <option value="">Select Role</option>
                    <option value="Property Seller">Property Seller</option>
                    <option value="Property Buyer">Property Buyer</option>
                    <option value="Real Estate Agent">Real Estate Agent</option>
                  </select>
                </div>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Testimonial Message</label>
                  <textarea name="message" defaultValue={(editingItem as Testimonial)?.message || ''} className="w-full p-2 border rounded h-24" required></textarea>
                </div>
              </form>
            )}
            
            {modalType.includes('blog') && (
              <form id="blog-form" onSubmit={(e) => {
                e.preventDefault();
                const formData = new FormData(e.target as HTMLFormElement);
                handleSave({
                  title: formData.get('title') as string,
                  description: formData.get('description') as string,
                  category: formData.get('category') as string
                });
              }}>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Blog Title</label>
                  <input name="title" defaultValue={(editingItem as Blog)?.title || ''} className="w-full p-2 border rounded" required />
                </div>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Category</label>
                  <select name="category" defaultValue={(editingItem as Blog)?.category || ''} className="w-full p-2 border rounded" required>
                    <option value="">Select Category</option>
                    <option value="INVESTMENT">Investment</option>
                    <option value="DOCUMENTATION">Documentation</option>
                    <option value="PRICING">Pricing</option>
                  </select>
                </div>
                <div className="mb-4">
                  <label className="block text-sm font-medium mb-2">Description</label>
                  <textarea name="description" defaultValue={(editingItem as Blog)?.description || ''} className="w-full p-2 border rounded h-24" required></textarea>
                </div>
              </form>
            )}
            </div>
            <div className="border-t p-6 pt-4">
              <div className="flex gap-2">
                <button 
                  type="submit" 
                  form={modalType.includes('faq') ? 'faq-form' : modalType.includes('policy') ? 'policy-form' : modalType.includes('testimonial') ? 'testimonial-form' : 'blog-form'}
                  className={`px-4 py-2 rounded text-white ${
                    modalType.includes('faq') ? 'bg-blue-600' : 
                    modalType.includes('policy') ? 'bg-purple-600' : 
                    modalType.includes('testimonial') ? 'bg-yellow-600' : 
                    'bg-indigo-600'
                  }`}
                >
                  Save
                </button>
                <button type="button" onClick={handleCloseModal} className="bg-gray-300 px-4 py-2 rounded">Cancel</button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}