import React, { useState, useEffect } from 'react';
import { Shield, CheckCircle, AlertTriangle, XCircle, Users, TrendingUp, Filter } from 'lucide-react';

interface UserQuality {
  user_id: string;
  name: string;
  email: string;
  role: 'buyer' | 'seller' | 'agent';
  governance_score: number;
  risk_level: 'Low' | 'Medium' | 'High';
  completion_details: {
    criteria: string;
    completed: boolean;
    weight: number;
  }[];
  missing_fields: string[];
  last_updated: string;
}

export function ContentQualityGovernance() {
  const [users, setUsers] = useState<UserQuality[]>([]);
  const [filteredUsers, setFilteredUsers] = useState<UserQuality[]>([]);
  const [selectedRole, setSelectedRole] = useState<string>('all');
  const [selectedRisk, setSelectedRisk] = useState<string>('all');
  const [stats, setStats] = useState({
    total: 0,
    low_risk: 0,
    medium_risk: 0,
    high_risk: 0,
    avg_score: 0
  });

  useEffect(() => {
    fetchUserQualityData();
  }, []);

  useEffect(() => {
    applyFilters();
  }, [selectedRole, selectedRisk, users]);

  const fetchUserQualityData = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/admin/quality/users', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await response.json();
      
      if (data.success) {
        setUsers(data.users);
        calculateStats(data.users);
      }
    } catch (error) {
      console.error('Error fetching quality data:', error);
    }
  };

  const calculateStats = (data: UserQuality[]) => {
    const total = data.length;
    const low_risk = data.filter(u => u.risk_level === 'Low').length;
    const medium_risk = data.filter(u => u.risk_level === 'Medium').length;
    const high_risk = data.filter(u => u.risk_level === 'High').length;
    const avg_score = data.reduce((sum, u) => sum + u.governance_score, 0) / total;

    setStats({ total, low_risk, medium_risk, high_risk, avg_score: Math.round(avg_score) });
  };

  const applyFilters = () => {
    let filtered = users;

    if (selectedRole !== 'all') {
      filtered = filtered.filter(u => u.role === selectedRole);
    }

    if (selectedRisk !== 'all') {
      filtered = filtered.filter(u => u.risk_level === selectedRisk);
    }

    setFilteredUsers(filtered);
  };

  const getRiskColor = (risk: string) => {
    switch (risk) {
      case 'Low': return 'bg-green-100 text-green-700 border-green-300';
      case 'Medium': return 'bg-yellow-100 text-yellow-700 border-yellow-300';
      case 'High': return 'bg-red-100 text-red-700 border-red-300';
      default: return 'bg-gray-100 text-gray-700 border-gray-300';
    }
  };

  const getScoreColor = (score: number) => {
    if (score >= 80) return 'text-green-600';
    if (score >= 50) return 'text-yellow-600';
    return 'text-red-600';
  };

  return (
    <div className="space-y-6">
      <div className="bg-white border-b-4 border-purple-600 rounded-2xl shadow-xl p-8">
        <div className="flex items-center gap-4">
          <div className="p-4 bg-purple-600 rounded-xl">
            <Shield className="w-8 h-8 text-white" />
          </div>
          <div>
            <h2 className="text-3xl font-bold text-gray-900">Content & Data Quality Governance</h2>
            <p className="text-gray-600 text-lg">Monitor user data completeness and quality scores</p>
          </div>
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-blue-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Total Users</p>
              <p className="text-3xl font-bold text-gray-900">{stats.total}</p>
            </div>
            <Users className="w-10 h-10 text-blue-500" />
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-green-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Low Risk</p>
              <p className="text-3xl font-bold text-green-600">{stats.low_risk}</p>
            </div>
            <CheckCircle className="w-10 h-10 text-green-500" />
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-yellow-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Medium Risk</p>
              <p className="text-3xl font-bold text-yellow-600">{stats.medium_risk}</p>
            </div>
            <AlertTriangle className="w-10 h-10 text-yellow-500" />
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-red-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">High Risk</p>
              <p className="text-3xl font-bold text-red-600">{stats.high_risk}</p>
            </div>
            <XCircle className="w-10 h-10 text-red-500" />
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-purple-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Avg Score</p>
              <p className="text-3xl font-bold text-purple-600">{stats.avg_score}%</p>
            </div>
            <TrendingUp className="w-10 h-10 text-purple-500" />
          </div>
        </div>
      </div>

      <div className="bg-white rounded-xl shadow-md p-6">
        <div className="flex items-center gap-4">
          <Filter className="w-5 h-5 text-gray-600" />
          <div className="flex gap-4 flex-1">
            <select
              value={selectedRole}
              onChange={(e) => setSelectedRole(e.target.value)}
              className="px-4 py-2 border-2 border-gray-300 rounded-lg font-semibold"
            >
              <option value="all">All Roles</option>
              <option value="seller">Seller</option>
              <option value="buyer">Buyer</option>
              <option value="agent">Agent</option>
            </select>

            <select
              value={selectedRisk}
              onChange={(e) => setSelectedRisk(e.target.value)}
              className="px-4 py-2 border-2 border-gray-300 rounded-lg font-semibold"
            >
              <option value="all">All Risk Levels</option>
              <option value="Low">Low Risk</option>
              <option value="Medium">Medium Risk</option>
              <option value="High">High Risk</option>
            </select>
          </div>
        </div>
      </div>

      <div className="space-y-4">
        {filteredUsers.map((user) => (
          <div
            key={user.user_id}
            className="bg-gradient-to-br from-white to-purple-50 rounded-xl shadow-lg p-6 border-2 border-purple-200 hover:border-purple-400 hover:shadow-2xl transition-all"
          >
            <div className="flex items-start justify-between mb-4">
              <div className="flex-1">
                <div className="flex items-center gap-3 mb-2">
                  <h3 className="text-xl font-bold text-gray-900">{user.name}</h3>
                  <span className="px-3 py-1 text-xs font-semibold rounded-full bg-blue-100 text-blue-700">
                    {user.role.toUpperCase()}
                  </span>
                  <span className={`px-3 py-1 text-xs font-semibold rounded-full border ${getRiskColor(user.risk_level)}`}>
                    {user.risk_level} Risk
                  </span>
                </div>
                <p className="text-sm text-gray-600">{user.email}</p>
              </div>
              <div className="text-right">
                <p className="text-sm text-gray-600 mb-1">Governance Score</p>
                <p className={`text-4xl font-bold ${getScoreColor(user.governance_score)}`}>
                  {user.governance_score}%
                </p>
              </div>
            </div>

            <div className="mb-4">
              <div className="w-full bg-gray-200 rounded-full h-3">
                <div
                  className={`h-3 rounded-full transition-all ${
                    user.governance_score >= 80 ? 'bg-green-500' :
                    user.governance_score >= 50 ? 'bg-yellow-500' : 'bg-red-500'
                  }`}
                  style={{ width: `${user.governance_score}%` }}
                />
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-3 mb-4">
              {user.completion_details.filter(detail => detail.completed).map((detail, idx) => (
                <div
                  key={idx}
                  className="flex items-center gap-2 p-3 rounded-lg border bg-green-50 border-green-200"
                >
                  <div className="w-6 h-6 rounded-full bg-green-500 flex items-center justify-center flex-shrink-0">
                    <svg className="w-4 h-4 text-white" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="flex-1">
                    <p className="text-sm font-semibold text-green-900">
                      {detail.criteria}
                    </p>
                  </div>
                  <span className="text-xs font-bold text-green-700">
                    {detail.weight}%
                  </span>
                </div>
              ))}
            </div>

            {user.missing_fields.length > 0 && (
              <div className="bg-gradient-to-br from-orange-50 to-red-50 border-2 border-orange-300 rounded-xl p-5 shadow-md">
                <div className="flex items-center gap-2 mb-4">
                  <div className="w-8 h-8 rounded-full bg-orange-500 flex items-center justify-center">
                    <svg className="w-5 h-5 text-white" fill="currentColor" viewBox="0 0 20 20">
                      <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <p className="text-base font-bold text-orange-900">Missing Fields</p>
                </div>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                  {user.missing_fields.map((field, idx) => (
                    <div key={idx} className="flex items-center gap-3 bg-white border-l-4 border-orange-500 rounded-lg p-3 shadow-sm hover:shadow-md transition-shadow">
                      <div className="w-6 h-6 rounded-full bg-orange-100 flex items-center justify-center flex-shrink-0">
                        <svg className="w-4 h-4 text-orange-600" fill="currentColor" viewBox="0 0 20 20">
                          <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 9.586 8.707 8.293z" clipRule="evenodd" />
                        </svg>
                      </div>
                      <span className="text-sm font-semibold text-gray-800">{field}</span>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </div>
        ))}
      </div>
    </div>
  );
}
