import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../ui/card";
import { Button } from "../ui/button";
import { 
  Users, Building, Shield, UserCheck, Home, TrendingUp, 
  Calendar, RefreshCw, Activity, FileText, Download, CreditCard, ArrowUpRight
} from "lucide-react";
import { getAdminDashboardStats, getAdminUserStats, getAdminPropertyStats, AdminDashboardStats, AdminUserStats, AdminPropertyStats } from "../../api/admin";
import { toast } from "sonner";
import { transactionApi } from "../../api/transactions";
import { Transaction, TransactionStatus } from "../../types/transaction";
import { reportService } from "../../services/reportService";
import { adminMarketTrendsApi } from "../../api/publicMarketTrends";

export function DashboardOverview() {
  const [dashboardStats, setDashboardStats] = useState<AdminDashboardStats | null>(null);
  const [userStats, setUserStats] = useState<AdminUserStats | null>(null);
  const [propertyStats, setPropertyStats] = useState<AdminPropertyStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [recentTransactions, setRecentTransactions] = useState<Transaction[]>([]);
  const [transactionsLoading, setTransactionsLoading] = useState(true);
  const [reportLoading, setReportLoading] = useState(false);

  const fetchDashboardStats = async () => {
    try {
      setLoading(true);
      
      // Check authentication before making requests
      const token = localStorage.getItem('access_token');
      if (!token) {
        toast.error('Authentication required. Please log in.');
        return;
      }
      
      const [dashboardData, userData, propertyData] = await Promise.all([
        getAdminDashboardStats(),
        getAdminUserStats(),
        getAdminPropertyStats()
      ]);
      setDashboardStats(dashboardData);
      setUserStats(userData);
      setPropertyStats(propertyData);
    } catch (error) {
      console.error('Error fetching dashboard stats:', error);
      
      if (error.message.includes('401') || error.message.includes('Unauthorized')) {
        toast.error('Session expired. Please log in again.');
        localStorage.removeItem('access_token');
        localStorage.removeItem('refresh_token');
        window.location.href = '/login';
      } else {
        toast.error('Failed to fetch dashboard statistics');
      }
    } finally {
      setLoading(false);
    }
  };

  const fetchRecentTransactions = async () => {
    try {
      setTransactionsLoading(true);
      
      // Check if user is authenticated before making the request
      const token = localStorage.getItem('access_token');
      if (!token) {
        console.warn('No access token found, skipping transaction fetch');
        setRecentTransactions([]);
        return;
      }
      
      const transactions = await transactionApi.list({ limit: 5 });
      setRecentTransactions(transactions);
    } catch (error) {
      console.error('Error fetching recent transactions:', error);
      
      // If it's an authentication error, don't show error toast
      if (!error.message.includes('Authentication')) {
        toast.error('Failed to load recent transactions');
      }
      
      setRecentTransactions([]);
    } finally {
      setTransactionsLoading(false);
    }
  };

  useEffect(() => {
    fetchDashboardStats();
    fetchRecentTransactions();
  }, []);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-[#2B256D]"></div>
      </div>
    );
  }

  if (!dashboardStats || !userStats || !propertyStats) {
    return (
      <div className="text-center py-12">
        <p className="text-gray-500 mb-4 text-sm">Failed to load dashboard statistics</p>
        <Button onClick={fetchDashboardStats} className="bg-[#2B256D] hover:bg-[#1f1a5a] text-sm px-4 py-2">
          <RefreshCw className="w-4 h-4 mr-2" />
          Retry
        </Button>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900">Dashboard Overview</h1>
          <p className="text-sm text-gray-600 mt-1">Real-time platform statistics and insights</p>
        </div>
        <div className="flex gap-3">
          <Button 
            onClick={async () => {
              try {
                setReportLoading(true);
                const filename = await reportService.downloadDashboardReport(userStats, propertyStats);
                toast.success(`Report downloaded: ${filename}`, {
                  description: "Your dashboard report has been generated successfully."
                });
              } catch (error) {
                console.error('Failed to generate report:', error);
                toast.error('Failed to generate report', {
                  description: error.message || 'An error occurred while generating the report.'
                });
              } finally {
                setReportLoading(false);
              }
            }}
            disabled={reportLoading}
            variant="outline" 
            className="border-[#2B256D] text-[#2B256D] hover:bg-[#2B256D] hover:text-white disabled:opacity-50 disabled:cursor-not-allowed"
          >
            <FileText className="w-4 h-4 mr-2" />
            {reportLoading ? 'Generating...' : 'Generate Report'}
          </Button>
          <Button 
            onClick={() => {
              try {
                const filename = reportService.exportToCSV(userStats, propertyStats);
                toast.success(`Data exported: ${filename}`, {
                  description: "Your dashboard data has been exported successfully."
                });
              } catch (error) {
                console.error('Failed to export data:', error);
                toast.error('Failed to export data', {
                  description: error.message || 'An error occurred while exporting the data.'
                });
              }
            }}
            variant="outline" 
            className="border-gray-300 text-gray-700 hover:bg-gray-50"
          >
            <Download className="w-4 h-4 mr-2" />
            Export Data
          </Button>
        </div>
      </div>

      {/* User Statistics */}
      <div>
        <h2 className="text-lg font-medium text-gray-900 mb-4">User Statistics</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Total Users</p>
                  <p className="text-2xl font-semibold text-gray-900 mt-1">{userStats.total_users.toLocaleString()}</p>
                  <p className="text-xs text-[#2B256D] mt-1">+{userStats.new_users_this_month} this month</p>
                </div>
                <div className="p-2 bg-[#2B256D]/10 rounded-lg">
                  <Users className="w-5 h-5 text-[#2B256D]" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Active Users</p>
                  <p className="text-2xl font-semibold text-gray-900 mt-1">{userStats.active_users.toLocaleString()}</p>
                  <p className="text-xs text-green-600 mt-1">+{userStats.new_users_today} today</p>
                </div>
                <div className="p-2 bg-green-100 rounded-lg">
                  <UserCheck className="w-5 h-5 text-green-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Verified Users</p>
                  <p className="text-2xl font-semibold text-gray-900 mt-1">{userStats.verified_users?.toLocaleString() || 0}</p>
                  <p className="text-xs text-blue-600 mt-1">
                    {userStats.total_users > 0 && userStats.verified_users ? 
                      Math.round((userStats.verified_users / userStats.total_users) * 100) : 0}% verified
                  </p>
                </div>
                <div className="p-2 bg-blue-100 rounded-lg">
                  <Shield className="w-5 h-5 text-blue-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Complete Profiles</p>
                  <p className="text-2xl font-semibold text-gray-900 mt-1">{userStats.users_with_complete_profile?.toLocaleString() || 0}</p>
                  <p className="text-xs text-purple-600 mt-1">
                    {userStats.total_users > 0 && userStats.users_with_complete_profile ? 
                      Math.round((userStats.users_with_complete_profile / userStats.total_users) * 100) : 0}% complete
                  </p>
                </div>
                <div className="p-2 bg-purple-100 rounded-lg">
                  <Users className="w-5 h-5 text-purple-600" />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Property Statistics */}
      <div>
        <h2 className="text-lg font-medium text-gray-900 mb-4">Property Statistics</h2>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Total Properties</p>
                  <p className="text-2xl font-semibold text-gray-900 mt-1">{propertyStats.total_properties.toLocaleString()}</p>
                  <p className="text-xs text-gray-600 mt-1">+{propertyStats.new_properties_this_month || 0} this month</p>
                </div>
                <div className="p-2 bg-gray-100 rounded-lg">
                  <Building className="w-5 h-5 text-gray-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Active Properties</p>
                  <p className="text-2xl font-semibold text-gray-900 mt-1">{propertyStats.active_properties.toLocaleString()}</p>
                  <p className="text-xs text-green-600 mt-1">+{propertyStats.new_properties_today || 0} today</p>
                </div>
                <div className="p-2 bg-green-100 rounded-lg">
                  <Home className="w-5 h-5 text-green-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Sold Properties</p>
                  <p className="text-2xl font-semibold text-gray-900 mt-1">{propertyStats.sold_properties.toLocaleString()}</p>
                  <p className="text-xs text-blue-600 mt-1">
                    {propertyStats.total_properties > 0 ? 
                      Math.round((propertyStats.sold_properties / propertyStats.total_properties) * 100) : 0}% sold
                  </p>
                </div>
                <div className="p-2 bg-blue-100 rounded-lg">
                  <TrendingUp className="w-5 h-5 text-blue-600" />
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Rented Properties</p>
                  <p className="text-2xl font-semibold text-gray-900 mt-1">{propertyStats.rented_properties.toLocaleString()}</p>
                  <p className="text-xs text-orange-600 mt-1">
                    {propertyStats.total_properties > 0 ? 
                      Math.round((propertyStats.rented_properties / propertyStats.total_properties) * 100) : 0}% rented
                  </p>
                </div>
                <div className="p-2 bg-orange-100 rounded-lg">
                  <Building className="w-5 h-5 text-orange-600" />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Property Breakdown */}
      {(Object.keys(propertyStats.properties_by_type || {}).length > 0 || 
        Object.keys(propertyStats.properties_by_city || {}).length > 0) && (
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {Object.keys(propertyStats.properties_by_type || {}).length > 0 && (
            <Card className="border border-gray-200 shadow-sm">
              <CardHeader className="">
                <CardTitle className="text-base font-medium text-gray-900">Properties by Type</CardTitle>
              </CardHeader>
              <CardContent className="pt-0">
                <div className="space-y-3">
                  {Object.entries(propertyStats.properties_by_type || {}).map(([type, count]) => (
                    <div key={type} className="flex items-center justify-between">
                      <span className="text-sm text-gray-700 capitalize">{type.replace('_', ' ')}</span>
                      <div className="flex items-center gap-3">
                        <div className="w-20 bg-gray-200 rounded-full h-1.5">
                          <div 
                            className="bg-[#2B256D] h-1.5 rounded-full transition-all" 
                            style={{ width: `${propertyStats.total_properties > 0 ? (count / propertyStats.total_properties) * 100 : 0}%` }}
                          ></div>
                        </div>
                        <span className="text-sm font-medium text-gray-900 w-6 text-right">{count}</span>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}

          {Object.keys(propertyStats.properties_by_city || {}).length > 0 && (
            <Card className="border border-gray-200 shadow-sm">
              <CardHeader className="pb-3">
                <CardTitle className="text-base font-medium text-gray-900">Properties by City</CardTitle>
              </CardHeader>
              <CardContent className="pt-0">
                <div className="space-y-3">
                  {Object.entries(propertyStats.properties_by_city || {}).slice(0, 6).map(([city, count]) => (
                    <div key={city} className="flex items-center justify-between">
                      <span className="text-sm text-gray-700">{city}</span>
                      <div className="flex items-center gap-3">
                        <div className="w-20 bg-gray-200 rounded-full h-1.5">
                          <div 
                            className="bg-green-600 h-1.5 rounded-full transition-all" 
                            style={{ width: `${propertyStats.total_properties > 0 ? (count / propertyStats.total_properties) * 100 : 0}%` }}
                          ></div>
                        </div>
                        <span className="text-sm font-medium text-gray-900 w-6 text-right">{count}</span>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      )}

      {/* User Roles Breakdown */}
      {Object.keys(userStats.users_by_role || {}).length > 0 && (
        <Card className="border border-gray-200 shadow-sm">
          <CardHeader className="pb-3">
            <CardTitle className="text-base font-medium text-gray-900">Users by Role</CardTitle>
          </CardHeader>
          <CardContent className="pt-0">
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              {Object.entries(userStats.users_by_role || {}).map(([role, count]) => (
                <div key={role} className="text-center p-3 bg-gray-50 rounded-lg">
                  <p className="text-xl font-semibold text-gray-900">{count}</p>
                  <p className="text-xs text-gray-600 mt-1 capitalize">{role.replace('_', ' ')}</p>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Recent Transactions Widget */}
      <Card className="border border-gray-200 shadow-sm">
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2 text-base font-medium text-gray-900">
              <CreditCard className="w-4 h-4" />
              Recent Transactions
            </CardTitle>
            <Button 
              variant="ghost" 
              size="sm"
              onClick={() => toast.info('Full transaction history coming soon!')}
              className="text-xs text-blue-600 hover:text-blue-700"
            >
              View All
              <ArrowUpRight className="w-3 h-3 ml-1" />
            </Button>
          </div>
        </CardHeader>
        <CardContent className="pt-0">
          {transactionsLoading ? (
            <div className="flex items-center justify-center py-8">
              <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-[#2B256D]"></div>
            </div>
          ) : recentTransactions.length === 0 ? (
            <div className="text-center py-8">
              <CreditCard className="w-12 h-12 text-gray-300 mx-auto mb-2" />
              <p className="text-sm text-gray-500">No recent transactions</p>
            </div>
          ) : (
            <div className="space-y-3">
              {recentTransactions.map((transaction) => (
                <div 
                  key={transaction.id} 
                  className="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors"
                >
                  <div className="flex items-center gap-3">
                    <div className={`p-2 rounded-full ${
                      transaction.transaction_status === TransactionStatus.COMPLETED ? 'bg-green-100' :
                      transaction.transaction_status === TransactionStatus.INITIATED || transaction.transaction_status === TransactionStatus.PROCESSING ? 'bg-yellow-100' :
                      transaction.transaction_status === TransactionStatus.FAILED ? 'bg-red-100' :
                      'bg-gray-100'
                    }`}>
                      <CreditCard className={`w-4 h-4 ${
                        transaction.transaction_status === TransactionStatus.COMPLETED ? 'text-green-600' :
                        transaction.transaction_status === TransactionStatus.INITIATED || transaction.transaction_status === TransactionStatus.PROCESSING ? 'text-yellow-600' :
                        transaction.transaction_status === TransactionStatus.FAILED ? 'text-red-600' :
                        'text-gray-600'
                      }`} />
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-900">
                        {transaction.transaction_type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}
                      </p>
                      <p className="text-xs text-gray-500">
                        {new Date(transaction.created_at).toLocaleDateString()} • {transaction.payment_method}
                      </p>
                    </div>
                  </div>
                  <div className="text-right">
                    <p className="text-sm font-semibold text-gray-900">
                      ₹{transaction.amount.toLocaleString()}
                    </p>
                    <p className={`text-xs capitalize ${
                      transaction.transaction_status === TransactionStatus.COMPLETED ? 'text-green-600' :
                      transaction.transaction_status === TransactionStatus.INITIATED || transaction.transaction_status === TransactionStatus.PROCESSING ? 'text-yellow-600' :
                      transaction.transaction_status === TransactionStatus.FAILED ? 'text-red-600' :
                      'text-gray-600'
                    }`}>
                      {transaction.transaction_status}
                    </p>
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Data Status */}
      <div className="text-center py-4 border-t border-gray-200">
        <p className="text-xs text-gray-500">
          Last updated: <span className="font-medium">{new Date().toLocaleString()}</span>
        </p>
        <p className="text-xs text-gray-400 mt-1">Statistics refreshed from live database</p>
      </div>
    </div>
  );
}