import React, { useState, useEffect } from 'react';
import { 
  Database, 
  Download, 
  Upload, 
  BarChart3, 
  FileText, 
  Shield, 
  Clock, 
  HardDrive,
  Activity,
  TrendingUp,
  Users,
  Building,
  Calendar,
  RefreshCw
} from 'lucide-react';
import DatabaseBackup from './DatabaseBackup';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../ui/card';
import { Button } from '../ui/button';
import { Badge } from '../ui/badge';
import { toast } from 'sonner';

interface DatabaseStats {
  totalTables: number;
  totalRecords: number;
  databaseSize: string;
  lastBackup: string;
  activeUsers: number;
  totalProperties: number;
  storageUsage: number;
  lastUpdated: string;
}

export function DataManagement() {
  const [stats, setStats] = useState<DatabaseStats>({
    totalTables: 0,
    totalRecords: 0,
    databaseSize: 'Loading...',
    lastBackup: 'Loading...',
    activeUsers: 0,
    totalProperties: 0,
    storageUsage: 0,
    lastUpdated: ''
  });
  const [loading, setLoading] = useState(true);
  const [refreshing, setRefreshing] = useState(false);

  const fetchDatabaseStats = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/admin/database/stats', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (!response.ok) {
        throw new Error('Failed to fetch database stats');
      }

      const result = await response.json();
      if (result.success) {
        setStats(result.data);
      }
    } catch (error) {
      console.error('Error fetching database stats:', error);
      toast.error('Failed to load database statistics');
    } finally {
      setLoading(false);
      setRefreshing(false);
    }
  };

  const handleRefresh = async () => {
    setRefreshing(true);
    await fetchDatabaseStats();
    toast.success('Database statistics refreshed');
  };

  useEffect(() => {
    fetchDatabaseStats();
  }, []);

  const quickActions = [
    {
      title: 'Export Users',
      description: 'Download user data as CSV',
      icon: Users,
      color: 'bg-blue-500',
      action: () => console.log('Export users')
    },
    {
      title: 'Export Properties',
      description: 'Download property listings',
      icon: Building,
      color: 'bg-green-500',
      action: () => console.log('Export properties')
    },
    {
      title: 'Analytics Report',
      description: 'Generate usage analytics',
      icon: BarChart3,
      color: 'bg-purple-500',
      action: () => console.log('Generate report')
    },
    {
      title: 'Audit Logs',
      description: 'Download system audit logs',
      icon: FileText,
      color: 'bg-orange-500',
      action: () => console.log('Export logs')
    }
  ];

  return (
    <div className="space-y-8">
      {/* Stats Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-blue-600 text-sm font-medium">Total Tables</p>
                <p className="text-3xl font-bold text-blue-900">
                  {loading ? '...' : stats.totalTables}
                </p>
              </div>
              <div className="bg-blue-500 p-3 rounded-full">
                <Database className="h-6 w-6 text-white" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-green-600 text-sm font-medium">Total Records</p>
                <p className="text-3xl font-bold text-green-900">
                  {loading ? '...' : stats.totalRecords.toLocaleString()}
                </p>
              </div>
              <div className="bg-green-500 p-3 rounded-full">
                <Activity className="h-6 w-6 text-white" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-purple-600 text-sm font-medium">Database Size</p>
                <p className="text-3xl font-bold text-purple-900">
                  {loading ? '...' : stats.databaseSize}
                </p>
              </div>
              <div className="bg-purple-500 p-3 rounded-full">
                <HardDrive className="h-6 w-6 text-white" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-orange-600 text-sm font-medium">Last Backup</p>
                <p className="text-lg font-bold text-orange-900">
                  {loading ? '...' : stats.lastBackup}
                </p>
                <Badge variant="outline" className="mt-1 text-xs border-orange-300 text-orange-700">
                  Automated
                </Badge>
              </div>
              <div className="bg-orange-500 p-3 rounded-full">
                <Clock className="h-6 w-6 text-white" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Additional Stats Row */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
        <Card className="bg-gradient-to-br from-indigo-50 to-indigo-100 border-indigo-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-indigo-600 text-sm font-medium">Active Users</p>
                <p className="text-2xl font-bold text-indigo-900">
                  {loading ? '...' : stats.activeUsers.toLocaleString()}
                </p>
                <p className="text-xs text-indigo-600 mt-1">Last 30 days</p>
              </div>
              <div className="bg-indigo-500 p-3 rounded-full">
                <Users className="h-6 w-6 text-white" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-teal-50 to-teal-100 border-teal-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-teal-600 text-sm font-medium">Total Properties</p>
                <p className="text-2xl font-bold text-teal-900">
                  {loading ? '...' : stats.totalProperties.toLocaleString()}
                </p>
                <p className="text-xs text-teal-600 mt-1">All listings</p>
              </div>
              <div className="bg-teal-500 p-3 rounded-full">
                <Building className="h-6 w-6 text-white" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-rose-50 to-rose-100 border-rose-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-rose-600 text-sm font-medium">Storage Usage</p>
                <p className="text-2xl font-bold text-rose-900">
                  {loading ? '...' : `${stats.storageUsage}%`}
                </p>
                <p className="text-xs text-rose-600 mt-1">Disk space</p>
              </div>
              <div className="bg-rose-500 p-3 rounded-full">
                <HardDrive className="h-6 w-6 text-white" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Last Updated Info */}
      {stats.lastUpdated && (
        <div className="text-center text-sm text-gray-500">
          Last updated: {new Date(stats.lastUpdated).toLocaleString()}
        </div>
      )}

      {/* Quick Actions */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="h-5 w-5" />
            Quick Actions
          </CardTitle>
          <CardDescription>
            Common data management tasks and exports
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {quickActions.map((action, index) => {
              const IconComponent = action.icon;
              return (
                <div
                  key={index}
                  className="group p-4 rounded-lg border border-gray-200 hover:border-gray-300 hover:shadow-md transition-all cursor-pointer"
                  onClick={action.action}
                >
                  <div className="flex items-center gap-3 mb-3">
                    <div className={`${action.color} p-2 rounded-lg group-hover:scale-110 transition-transform`}>
                      <IconComponent className="h-4 w-4 text-white" />
                    </div>
                    <h3 className="font-semibold text-gray-900">{action.title}</h3>
                  </div>
                  <p className="text-sm text-gray-600">{action.description}</p>
                </div>
              );
            })}
          </div>
        </CardContent>
      </Card>

      {/* System Health */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Shield className="h-5 w-5 text-green-600" />
              System Health
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
              <span className="text-sm font-medium">Database Status</span>
              <Badge className="bg-green-100 text-green-800 hover:bg-green-100">Healthy</Badge>
            </div>
            <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
              <span className="text-sm font-medium">Backup Status</span>
              <Badge className="bg-blue-100 text-blue-800 hover:bg-blue-100">Active</Badge>
            </div>
            <div className="flex items-center justify-between p-3 bg-yellow-50 rounded-lg">
              <span className="text-sm font-medium">Storage Usage</span>
              <Badge className={`${stats.storageUsage > 80 ? 'bg-red-100 text-red-800' : 'bg-yellow-100 text-yellow-800'} hover:bg-yellow-100`}>
                {stats.storageUsage}%
              </Badge>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Calendar className="h-5 w-5 text-blue-600" />
              Recent Activity
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-3">
            <div className="flex items-center gap-3 p-2">
              <div className="w-2 h-2 bg-green-500 rounded-full"></div>
              <span className="text-sm">Database backup completed</span>
              <span className="text-xs text-gray-500 ml-auto">2h ago</span>
            </div>
            <div className="flex items-center gap-3 p-2">
              <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
              <span className="text-sm">Statistics refreshed</span>
              <span className="text-xs text-gray-500 ml-auto">Just now</span>
            </div>
            <div className="flex items-center gap-3 p-2">
              <div className="w-2 h-2 bg-purple-500 rounded-full"></div>
              <span className="text-sm">System health check passed</span>
              <span className="text-xs text-gray-500 ml-auto">1h ago</span>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Database Backup Section */}
      <DatabaseBackup />
    </div>
  );
}
