import React, { useState } from 'react';
import { Download, Database, Loader2, CheckCircle, AlertCircle, Shield, Clock, FileArchive } from 'lucide-react';
import { Button } from '../ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';
import { toast } from 'sonner';


const DatabaseBackup: React.FC = () => {
  const [isDownloading, setIsDownloading] = useState(false);
  const [downloadStatus, setDownloadStatus] = useState<'idle' | 'success' | 'error'>('idle');

  const handleDownloadBackup = async () => {
    setIsDownloading(true);
    setDownloadStatus('idle');

    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/admin/backup/download', {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Accept': 'application/gzip'
        }
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const blob = await response.blob();
      
      // Create download link
      const url = window.URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = url;
      
      // Generate filename with timestamp
      const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
      link.download = `nal_backup_${timestamp}.sql.gz`;
      
      // Trigger download
      document.body.appendChild(link);
      link.click();
      
      // Cleanup
      document.body.removeChild(link);
      window.URL.revokeObjectURL(url);

      setDownloadStatus('success');
      toast.success('Database backup downloaded successfully!');
      
    } catch (error: any) {
      console.error('Backup download failed:', error);
      setDownloadStatus('error');
      
      const errorMessage = error.message || 'Failed to download database backup';
      toast.error(errorMessage);
      
    } finally {
      setIsDownloading(false);
    }
  };

  const backupFeatures = [
    {
      icon: Shield,
      title: 'Secure Backup',
      description: 'Encrypted and compressed for security'
    },
    {
      icon: Clock,
      title: 'Point-in-Time',
      description: 'Captures exact database state'
    },
    {
      icon: FileArchive,
      title: 'Compressed',
      description: 'Optimized file size for storage'
    }
  ];

  return (
    <Card className="w-full bg-gradient-to-br from-slate-50 to-slate-100 border-slate-200">
      <CardHeader className="bg-gradient-to-r from-slate-600 to-slate-700 text-white rounded-t-lg">
        <CardTitle className="flex items-center gap-3">
          <div className="bg-white/20 p-2 rounded-lg">
            <Database className="h-6 w-6" />
          </div>
          <div>
            <h3 className="text-xl font-bold">Database Backup</h3>
            <p className="text-slate-200 text-sm font-normal">
              Create and download complete database backups
            </p>
          </div>
        </CardTitle>
      </CardHeader>
      <CardContent className="p-6 space-y-6">
        {/* Backup Features */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          {backupFeatures.map((feature, index) => {
            const IconComponent = feature.icon;
            return (
              <div key={index} className="flex items-center gap-3 p-3 bg-white rounded-lg border border-slate-200">
                <div className="bg-slate-100 p-2 rounded-lg">
                  <IconComponent className="h-4 w-4 text-slate-600" />
                </div>
                <div>
                  <p className="font-medium text-sm text-slate-900">{feature.title}</p>
                  <p className="text-xs text-slate-600">{feature.description}</p>
                </div>
              </div>
            );
          })}
        </div>

        {/* Main Backup Action */}
        <div className="bg-white rounded-xl border border-slate-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <div className="space-y-2">
              <div className="flex items-center gap-2">
                <h4 className="text-lg font-semibold text-slate-900">Complete Database Backup</h4>
                <Badge className="bg-green-100 text-green-800 hover:bg-green-100">
                  Recommended
                </Badge>
              </div>
              <p className="text-slate-600">
                Includes all tables, data, relationships, and schema
              </p>
            </div>
            
            <Button
              onClick={handleDownloadBackup}
              disabled={isDownloading}
              size="lg"
              className="bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 gap-2 px-6"
            >
              {isDownloading ? (
                <>
                  <Loader2 className="h-5 w-5 animate-spin" />
                  Creating Backup...
                </>
              ) : (
                <>
                  <Download className="h-5 w-5" />
                  Download Backup
                </>
              )}
            </Button>
          </div>

          {/* Status Messages */}
          {downloadStatus === 'success' && (
            <div className="flex items-center gap-3 p-4 bg-green-50 border border-green-200 text-green-800 rounded-lg">
              <CheckCircle className="h-5 w-5" />
              <div>
                <p className="font-medium">Backup downloaded successfully!</p>
                <p className="text-sm text-green-700">Your database backup is ready for use.</p>
              </div>
            </div>
          )}

          {downloadStatus === 'error' && (
            <div className="flex items-center gap-3 p-4 bg-red-50 border border-red-200 text-red-800 rounded-lg">
              <AlertCircle className="h-5 w-5" />
              <div>
                <p className="font-medium">Failed to download backup</p>
                <p className="text-sm text-red-700">Please try again or contact support if the issue persists.</p>
              </div>
            </div>
          )}
        </div>

        {/* Backup Information */}
        <div className="bg-slate-50 rounded-lg p-4 border border-slate-200">
          <h5 className="font-semibold text-slate-900 mb-3 flex items-center gap-2">
            <Database className="h-4 w-4" />
            Backup Information
          </h5>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-slate-600">File Format:</span>
                <span className="font-medium text-slate-900">Compressed SQL (.sql.gz)</span>
              </div>
              <div className="flex justify-between">
                <span className="text-slate-600">Estimated Size:</span>
                <span className="font-medium text-slate-900">10-50 MB</span>
              </div>
            </div>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-slate-600">Tables Included:</span>
                <span className="font-medium text-slate-900">All (85+ tables)</span>
              </div>
              <div className="flex justify-between">
                <span className="text-slate-600">Use Cases:</span>
                <span className="font-medium text-slate-900">Restore, Migration</span>
              </div>
            </div>
          </div>
        </div>

        {/* Security Notice */}
        <div className="bg-amber-50 border border-amber-200 rounded-lg p-4">
          <div className="flex items-start gap-3">
            <Shield className="h-5 w-5 text-amber-600 mt-0.5" />
            <div>
              <p className="font-medium text-amber-900 mb-1">Security Notice</p>
              <p className="text-sm text-amber-800">
                Backup files contain sensitive data. Store securely and restrict access. 
                Consider encrypting files for additional security.
              </p>
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default DatabaseBackup;
