import React from 'react';

interface Property {
  id: number;
  title: string;
  location: string;
  price: string;
  imageUrl?: string;
}

interface DuplicateWarningModalProps {
  isOpen: boolean;
  duplicates: Property[];
  onConfirm: () => void;
  onCancel: () => void;
}

export const DuplicateWarningModal: React.FC<DuplicateWarningModalProps> = ({
  isOpen,
  duplicates,
  onConfirm,
  onCancel
}) => {
  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 z-[9999] bg-black/50 flex items-center justify-center p-4">
      <div className="bg-white rounded-lg shadow-xl max-w-md w-full p-6">
        <div className="flex items-center gap-3 mb-4">
          <div className="w-8 h-8 bg-red-100 rounded-full flex items-center justify-center">
            <svg className="w-5 h-5 text-red-600" fill="currentColor" viewBox="0 0 20 20">
              <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
            </svg>
          </div>
          <h3 className="text-lg font-semibold text-gray-900">Potential Duplicate Detected</h3>
        </div>
        
        <p className="text-gray-600 mb-4">
          We found {duplicates.length} similar listing{duplicates.length > 1 ? 's' : ''}. Please review:
        </p>
        
        <div className="space-y-3 mb-6 max-h-60 overflow-y-auto">
          {duplicates.map(duplicate => (
            <div key={duplicate.id} className="border rounded-lg p-3 bg-gray-50">
              <div className="flex gap-3">
                {duplicate.imageUrl && (
                  <img src={duplicate.imageUrl} alt="" className="w-16 h-16 rounded object-cover" />
                )}
                <div className="flex-1">
                  <h4 className="font-medium text-gray-900 text-sm">{duplicate.title}</h4>
                  <p className="text-xs text-gray-600">{duplicate.location}</p>
                  <p className="text-sm font-semibold text-blue-600">{duplicate.price}</p>
                </div>
              </div>
            </div>
          ))}
        </div>
        
        <div className="flex gap-3">
          <button
            onClick={onCancel}
            className="flex-1 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
          >
            Cancel
          </button>
          <button
            onClick={onConfirm}
            className="flex-1 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors"
          >
            Post Anyway
          </button>
        </div>
      </div>
    </div>
  );
};