import React, { useState, useEffect } from 'react';
import { Flag, Users, BarChart3, Plus, Toggle } from 'lucide-react';

export function FeatureFlags() {
  const [flags, setFlags] = useState([]);
  const [selectedFlag, setSelectedFlag] = useState(null);
  const [users, setUsers] = useState([]);
  const [analytics, setAnalytics] = useState(null);
  const [loading, setLoading] = useState(true);
  const [showCreateModal, setShowCreateModal] = useState(false);
  const [showAssignModal, setShowAssignModal] = useState(false);
  const [newFlag, setNewFlag] = useState({ name: '', description: '', category: 'ai' });
  const [selectedUsers, setSelectedUsers] = useState([]);
  const [userType, setUserType] = useState('BUYER');

  useEffect(() => {
    fetchFlags();
  }, []);

  const fetchFlags = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const res = await fetch('http://localhost:8090/api/admin/feature-flags/', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await res.json();
      setFlags(data.flags || []);
    } catch (error) {
      console.error('Error fetching flags:', error);
    } finally {
      setLoading(false);
    }
  };

  const toggleFlag = async (flagName, enabled) => {
    try {
      const token = localStorage.getItem('access_token');
      await fetch('http://localhost:8090/api/admin/feature-flags/toggle/', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ flag_name: flagName, enabled })
      });
      fetchFlags();
    } catch (error) {
      console.error('Error toggling flag:', error);
    }
  };

  const fetchAnalytics = async (flagName) => {
    try {
      const token = localStorage.getItem('access_token');
      const res = await fetch(`http://localhost:8090/api/admin/feature-flags/usage/?flag_name=${flagName}`, {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await res.json();
      setAnalytics(data);
    } catch (error) {
      console.error('Error fetching analytics:', error);
    }
  };

  const createFlag = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const flagData = {
        name: newFlag.name.trim().toLowerCase().replace(/\s+/g, '_'),
        description: newFlag.description,
        category: newFlag.category
      };
      
      await fetch('http://localhost:8090/api/admin/feature-flags/create/', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(flagData)
      });
      setShowCreateModal(false);
      setNewFlag({ name: '', description: '', category: 'ai' });
      fetchFlags();
    } catch (error) {
      console.error('Error creating flag:', error);
    }
  };

  const fetchUsers = async (type) => {
    try {
      const token = localStorage.getItem('access_token');
      const res = await fetch(`http://localhost:8090/api/admin/feature-flags/users/?user_type=${type}`, {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      const data = await res.json();
      setUsers(data.users || []);
      
      // Fetch already assigned users for this flag
      if (selectedFlag) {
        const analyticsRes = await fetch(`http://localhost:8090/api/admin/feature-flags/usage/?flag_name=${selectedFlag.name}`, {
          headers: { 'Authorization': `Bearer ${token}` }
        });
        const analyticsData = await analyticsRes.json();
        const assignedUserIds = analyticsData.usage_details?.map(u => u.user_id) || [];
        setSelectedUsers(assignedUserIds);
      }
    } catch (error) {
      console.error('Error fetching users:', error);
    }
  };

  const assignFlag = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const payload = {
        flag_name: selectedFlag.name,
        user_ids: selectedUsers,
        user_type: userType
      };
      
      const response = await fetch('http://localhost:8090/api/admin/feature-flags/assign/', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(payload)
      });
      
      const data = await response.json();
      
      if (response.ok) {
        alert(`Successfully assigned to ${selectedUsers.length} users!`);
        setShowAssignModal(false);
        setSelectedUsers([]);
        fetchAnalytics(selectedFlag.name);
      } else {
        alert('Failed to assign: ' + (data.message || 'Unknown error'));
      }
    } catch (error) {
      console.error('Error assigning flag:', error);
      alert('Error: ' + error.message);
    }
  };

  if (loading) return <div className="p-8">Loading...</div>;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-white border-b-4 border-blue-600 rounded-2xl shadow-xl p-8">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <div className="bg-blue-600 p-4 rounded-xl">
              <Flag className="w-8 h-8 text-white" />
            </div>
            <div>
              <h2 className="text-3xl font-bold text-gray-900">Feature Flags</h2>
              <p className="text-gray-600 text-lg">Manage AI features and themes</p>
            </div>
          </div>
          <button
            onClick={() => setShowCreateModal(true)}
            className="bg-gray-900 text-white px-6 py-3 rounded-lg font-semibold hover:bg-gray-800 flex items-center gap-2"
          >
            <Plus className="w-5 h-5" />
            Create Flag
          </button>
        </div>
      </div>

      {/* Flags Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        {flags.map((flag) => (
          <div
            key={flag.name}
            className="bg-white border-2 border-gray-200 rounded-xl p-6 hover:border-blue-500 hover:shadow-lg transition-all"
          >
            <div className="flex items-start justify-between mb-4">
              <div className="flex-1">
                <div className="flex items-center gap-2 mb-2">
                  <span className={`px-2 py-1 text-xs font-bold rounded ${
                    flag.category === 'ai' ? 'bg-purple-100 text-purple-700' : 'bg-blue-100 text-blue-700'
                  }`}>
                    {flag.category.toUpperCase()}
                  </span>
                </div>
                <h4 className="font-bold text-gray-900 text-lg mb-2">{flag.name.replace(/_/g, ' ').toUpperCase()}</h4>
                <p className="text-sm text-gray-600">{flag.description}</p>
              </div>
            </div>
            
            <div className="space-y-2 mt-4">
              <button
                onClick={() => toggleFlag(flag.name, !flag.is_enabled)}
                className={`w-full px-4 py-2 rounded-lg font-bold transition-colors text-white ${
                  flag.is_enabled
                    ? 'bg-red-600 hover:bg-red-700'
                    : 'bg-green-500 hover:bg-green-600'
                }`}
              >
                {flag.is_enabled ? 'Disable' : 'Enable'}
              </button>
              
              <div className="flex gap-2">
                <button
                  onClick={() => {
                    setSelectedFlag(flag);
                    fetchUsers('BUYER');
                    setShowAssignModal(true);
                  }}
                  className="flex-1 px-4 py-2 bg-purple-600 text-white rounded-lg font-semibold hover:bg-purple-700 transition-colors flex items-center justify-center gap-1"
                >
                  <Users className="w-4 h-4" />
                  Assign
                </button>
                <button
                  onClick={() => {
                    setSelectedFlag(flag);
                    fetchAnalytics(flag.name);
                  }}
                  className="flex-1 px-4 py-2 bg-gray-900 text-white rounded-lg font-semibold hover:bg-gray-800 transition-colors flex items-center justify-center gap-1"
                >
                  <BarChart3 className="w-4 h-4" />
                  Analytics
                </button>
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Analytics Modal */}
      {selectedFlag && analytics && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-2xl p-8 max-w-4xl w-full max-h-[80vh] overflow-y-auto">
            <div className="flex items-center justify-between mb-6">
              <h3 className="text-2xl font-bold text-gray-900">{selectedFlag.name} - Analytics</h3>
              <button
                onClick={() => { setSelectedFlag(null); setAnalytics(null); }}
                className="text-gray-500 hover:text-gray-700 text-2xl"
              >
                ×
              </button>
            </div>

            <div className="grid grid-cols-3 gap-4 mb-6">
              <div className="bg-blue-50 p-4 rounded-lg">
                <div className="text-sm text-gray-600">Total Assigned</div>
                <div className="text-3xl font-bold text-blue-600">{analytics.total_users_assigned}</div>
              </div>
              <div className="bg-green-50 p-4 rounded-lg">
                <div className="text-sm text-gray-600">Users Active</div>
                <div className="text-3xl font-bold text-green-600">{analytics.total_users_used}</div>
              </div>
              <div className="bg-purple-50 p-4 rounded-lg">
                <div className="text-sm text-gray-600">Total Usage</div>
                <div className="text-3xl font-bold text-purple-600">{analytics.total_usage_count}</div>
              </div>
            </div>

            <div className="space-y-2">
              <h4 className="font-bold text-gray-900 mb-4">User Activity</h4>
              {analytics.usage_details?.map((user) => (
                <div key={user.user_id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                  <div>
                    <div className="font-semibold text-gray-900">{user.email}</div>
                    <div className="text-sm text-gray-600">
                      {user.usage_count > 0 ? `Used ${user.usage_count} times` : 'Not used yet'}
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="text-sm text-gray-600">
                      {user.last_used ? new Date(user.last_used).toLocaleDateString() : 'Never'}
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      {/* Create Flag Modal */}
      {showCreateModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-2xl p-8 max-w-md w-full">
            <h3 className="text-2xl font-bold text-gray-900 mb-6">Create Feature Flag</h3>
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Flag Name</label>
                <input
                  type="text"
                  value={newFlag.name}
                  onChange={(e) => setNewFlag({...newFlag, name: e.target.value})}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  placeholder="e.g., ai_new_feature"
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Description</label>
                <textarea
                  value={newFlag.description}
                  onChange={(e) => setNewFlag({...newFlag, description: e.target.value})}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                  rows={3}
                  placeholder="Describe the feature..."
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Category</label>
                <select
                  value={newFlag.category}
                  onChange={(e) => setNewFlag({...newFlag, category: e.target.value})}
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                >
                  <option value="ai">AI</option>
                  <option value="theme">Theme</option>
                  <option value="font">Font</option>
                  <option value="feature">Feature</option>
                </select>
              </div>
            </div>
            <div className="flex gap-3 mt-6">
              <button
                onClick={createFlag}
                className="flex-1 bg-gray-900 text-white px-6 py-3 rounded-lg font-semibold hover:bg-gray-800"
              >
                Create
              </button>
              <button
                onClick={() => setShowCreateModal(false)}
                className="flex-1 bg-gray-300 text-gray-700 px-6 py-3 rounded-lg font-semibold hover:bg-gray-400"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Assign Users Modal */}
      {showAssignModal && selectedFlag && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-2xl p-8 max-w-3xl w-full max-h-[90vh] overflow-y-auto">
            <div className="flex items-center justify-between mb-6">
              <div>
                <h3 className="text-2xl font-bold text-gray-900">Assign Feature Flag</h3>
                <p className="text-gray-600 mt-1">{selectedFlag.name.replace(/_/g, ' ')}</p>
              </div>
              <button
                onClick={() => { setShowAssignModal(false); setSelectedUsers([]); }}
                className="text-gray-500 hover:text-gray-700 text-2xl"
              >
                ×
              </button>
            </div>
            
            <div className="mb-6">
              <label className="block text-sm font-semibold text-gray-900 mb-3">Select User Type</label>
              <div className="flex gap-2">
                {['BUYER', 'SELLER', 'AGENT'].map((type) => (
                  <button
                    key={type}
                    onClick={() => { setUserType(type); fetchUsers(type); setSelectedUsers([]); }}
                    className={`px-6 py-2 rounded-lg font-semibold transition-colors ${
                      userType === type
                        ? 'bg-purple-600 text-white'
                        : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                    }`}
                  >
                    {type}
                  </button>
                ))}
              </div>
            </div>

            <div className="mb-4">
              <div className="flex items-center justify-between mb-3">
                <label className="text-sm font-semibold text-gray-900">Select Users ({users.length} available)</label>
                <button
                  onClick={() => {
                    if (selectedUsers.length === users.length) {
                      setSelectedUsers([]);
                    } else {
                      setSelectedUsers(users.map(u => u.user_id));
                    }
                  }}
                  className="text-sm text-purple-600 hover:text-purple-800 font-semibold"
                >
                  {selectedUsers.length === users.length ? 'Deselect All' : 'Select All'}
                </button>
              </div>
              
              <div className="space-y-2 max-h-60 overflow-y-scroll border border-gray-200 rounded-lg p-3">
                {users.length === 0 ? (
                  <div className="text-center py-8 text-gray-500">No users found for {userType}</div>
                ) : (
                  users.map((user) => (
                    <label key={user.user_id} className="flex items-center p-4 bg-gray-50 rounded-lg hover:bg-purple-50 cursor-pointer transition-colors border-2 border-transparent hover:border-purple-200">
                      <input
                        type="checkbox"
                        checked={selectedUsers.includes(user.user_id)}
                        onChange={(e) => {
                          if (e.target.checked) {
                            setSelectedUsers([...selectedUsers, user.user_id]);
                          } else {
                            setSelectedUsers(selectedUsers.filter(id => id !== user.user_id));
                          }
                        }}
                        className="mr-4 w-5 h-5 text-purple-600"
                      />
                      <div className="flex-1">
                        <div className="flex items-center gap-2">
                          <span className="font-semibold text-gray-900">{user.email}</span>
                          {user.is_verified && (
                            <span className="text-xs bg-green-100 text-green-700 px-2 py-0.5 rounded-full">Verified</span>
                          )}
                        </div>
                      </div>
                    </label>
                  ))
                )}
              </div>
            </div>
            
            <div className="flex gap-3 mt-6">
              <button
                onClick={assignFlag}
                disabled={selectedUsers.length === 0}
                className={`flex-1 px-6 py-3 rounded-lg font-semibold transition-colors ${
                  selectedUsers.length === 0
                    ? 'bg-gray-300 text-gray-500 cursor-not-allowed'
                    : 'bg-purple-600 text-white hover:bg-purple-700'
                }`}
              >
                Assign to {selectedUsers.length} User{selectedUsers.length !== 1 ? 's' : ''}
              </button>
              <button
                onClick={() => { setShowAssignModal(false); setSelectedUsers([]); }}
                className="px-6 py-3 bg-gray-200 text-gray-700 rounded-lg font-semibold hover:bg-gray-300"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
