import React, { useState } from 'react';
import RevenueChart from '../../finance/RevenueChart';
import { Download, Filter, Calendar } from 'lucide-react';

// Mock Data
const mockRevenueData = [
    { date: 'Mon', revenue: 4000, refunds: 240 },
    { date: 'Tue', revenue: 3000, refunds: 139 },
    { date: 'Wed', revenue: 2000, refunds: 980 },
    { date: 'Thu', revenue: 2780, refunds: 390 },
    { date: 'Fri', revenue: 1890, refunds: 480 },
    { date: 'Sat', revenue: 2390, refunds: 380 },
    { date: 'Sun', revenue: 3490, refunds: 430 },
];

const mockTransactions = [
    { id: 'TXN-001', user: 'John Doe', type: 'CREDIT', amount: 120.00, status: 'COMPLETED', date: '2024-02-08' },
    { id: 'TXN-002', user: 'Jane Smith', type: 'DEBIT', amount: 50.00, status: 'REFUNDED', date: '2024-02-07' },
    { id: 'TXN-003', user: 'Alice Brown', type: 'CREDIT', amount: 1200.00, status: 'COMPLETED', date: '2024-02-06' },
    { id: 'TXN-004', user: 'Bob Wilson', type: 'CREDIT', amount: 35.00, status: 'FAILED', date: '2024-02-06' },
];

const LedgerPage: React.FC = () => {
    const [filter, setFilter] = useState('all');

    return (
        <div className="p-6 space-y-6 bg-gray-50 min-h-screen">
            <div className="flex justify-between items-center mb-6">
                <div>
                    <h1 className="text-2xl font-bold text-gray-900">Revenue Ledger</h1>
                    <p className="text-sm text-gray-500">Track all financial movements in the system.</p>
                </div>
                <div className="flex gap-3">
                    <button className="flex items-center gap-2 px-4 py-2 bg-white border border-gray-200 rounded-lg text-gray-700 hover:bg-gray-50 text-sm font-medium transition-colors">
                        <Calendar className="w-4 h-4" /> This Week
                    </button>
                    <button className="flex items-center gap-2 px-4 py-2 bg-[#0056D2] hover:bg-[#0041A3] text-white rounded-lg text-sm font-medium transition-colors shadow-sm">
                        <Download className="w-4 h-4" /> Export CSV
                    </button>
                </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div className="lg:col-span-2">
                    <RevenueChart data={mockRevenueData} />
                </div>
                <div className="space-y-6">
                    <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-100">
                        <h3 className="text-gray-500 text-sm font-medium mb-1">Total Revenue</h3>
                        <p className="text-3xl font-bold text-[#0056D2]">$45,231.89</p>
                        <div className="mt-4 flex items-center text-sm text-green-600 bg-green-50 w-fit px-2 py-1 rounded">
                            +20.1% from last month
                        </div>
                    </div>
                    <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-100">
                        <h3 className="text-gray-500 text-sm font-medium mb-1">Total Refunds</h3>
                        <p className="text-3xl font-bold text-gray-900">$1,205.00</p>
                        <div className="mt-4 flex items-center text-sm text-gray-500">
                            2.6% refund rate
                        </div>
                    </div>
                </div>
            </div>

            <div className="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                <div className="px-6 py-4 border-b border-gray-100 flex justify-between items-center">
                    <h3 className="font-semibold text-gray-900">Recent Transactions</h3>
                    <button className="flex items-center gap-2 text-sm text-gray-500 hover:text-[#0056D2]">
                        <Filter className="w-4 h-4" /> Filter
                    </button>
                </div>
                <table className="w-full text-left text-sm">
                    <thead className="bg-gray-50 text-gray-500 uppercase font-medium">
                        <tr>
                            <th className="px-6 py-3">Transaction ID</th>
                            <th className="px-6 py-3">User</th>
                            <th className="px-6 py-3">Date</th>
                            <th className="px-6 py-3">Type</th>
                            <th className="px-6 py-3 text-right">Amount</th>
                            <th className="px-6 py-3 text-center">Status</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-100">
                        {mockTransactions.map((txn) => (
                            <tr key={txn.id} className="hover:bg-gray-50 transition-colors">
                                <td className="px-6 py-4 font-mono text-gray-600">{txn.id}</td>
                                <td className="px-6 py-4 font-medium text-gray-900">{txn.user}</td>
                                <td className="px-6 py-4 text-gray-500">{txn.date}</td>
                                <td className="px-6 py-4">
                                    <span className={`px-2 py-1 rounded text-xs font-medium ${txn.type === 'CREDIT' ? 'bg-green-100 text-[#00BFA6]' : 'bg-red-100 text-red-600'}`}>
                                        {txn.type}
                                    </span>
                                </td>
                                <td className={`px-6 py-4 text-right font-medium ${txn.type === 'CREDIT' ? 'text-gray-900' : 'text-red-600'}`}>
                                    {txn.type === 'DEBIT' ? '-' : ''}${txn.amount.toFixed(2)}
                                </td>
                                <td className="px-6 py-4 text-center">
                                    <span className={`px-2 py-1 rounded-full text-xs font-semibold
                                ${txn.status === 'COMPLETED' ? 'bg-green-50 text-[#00BFA6]' :
                                            txn.status === 'REFUNDED' ? 'bg-gray-100 text-gray-600' : 'bg-red-50 text-red-600'}`
                                    }>
                                        {txn.status}
                                    </span>
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>
        </div>
    );
};

export default LedgerPage;
