import React, { useState } from 'react';
import { IndianRupee, Check, X, Search, Filter } from 'lucide-react';

// Mock Data
const mockRefunds = [
    { id: 'RF-101', user: 'John Doe', amount: 50.00, reason: 'Duplicate Charge', status: 'PENDING', date: '2024-02-08' },
    { id: 'RF-102', user: 'Jane Smith', amount: 120.00, reason: 'Accidental Purchase', status: 'PENDING', date: '2024-02-07' },
    { id: 'RF-103', user: 'Bob Wilson', amount: 35.00, reason: 'Service Issue', status: 'APPROVED', date: '2024-02-06' },
];

const RefundsPage: React.FC = () => {
    const [selectedTab, setSelectedTab] = useState('pending');

    return (
        <div className="p-6 space-y-6 bg-gray-50 min-h-screen">
            <div className="flex justify-between items-center mb-6">
                <div>
                    <h1 className="text-2xl font-bold text-gray-900">Refund Requests</h1>
                    <p className="text-sm text-gray-500">Manage and process user refund requests.</p>
                </div>
            </div>

            <div className="flex gap-4 mb-6 border-b border-gray-200">
                <button
                    onClick={() => setSelectedTab('pending')}
                    className={`pb-3 px-4 text-sm font-medium border-b-2 transition-colors ${selectedTab === 'pending' ? 'border-[#0056D2] text-[#0056D2]' : 'border-transparent text-gray-500 hover:text-gray-700'}`}
                >
                    Pending Review
                </button>
                <button
                    onClick={() => setSelectedTab('processed')}
                    className={`pb-3 px-4 text-sm font-medium border-b-2 transition-colors ${selectedTab === 'processed' ? 'border-[#0056D2] text-[#0056D2]' : 'border-transparent text-gray-500 hover:text-gray-700'}`}
                >
                    Processed History
                </button>
            </div>

            <div className="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                <div className="px-6 py-4 border-b border-gray-100 flex gap-4 items-center bg-gray-50">
                    <div className="relative flex-1 max-w-sm">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400" />
                        <input
                            type="text"
                            placeholder="Search by ticket ID or user..."
                            className="w-full pl-10 pr-4 py-2 rounded-lg border border-gray-200 focus:outline-none focus:ring-2 focus:ring-[#0056D2]/20 text-sm"
                        />
                    </div>
                    <button className="flex items-center gap-2 text-sm text-gray-500 hover:text-[#0056D2] ml-auto">
                        <Filter className="w-4 h-4" /> Filter
                    </button>
                </div>

                <div className="overflow-x-auto">
                    <table className="w-full text-left text-sm">
                        <thead className="bg-gray-50 text-gray-500 uppercase font-medium">
                            <tr>
                                <th className="px-6 py-3">Ticket ID</th>
                                <th className="px-6 py-3">User</th>
                                <th className="px-6 py-3">Date</th>
                                <th className="px-6 py-3">Reason</th>
                                <th className="px-6 py-3 text-right">Amount</th>
                                <th className="px-6 py-3 text-center">Status</th>
                                <th className="px-6 py-3 text-right">Actions</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-gray-100">
                            {mockRefunds.filter(r => selectedTab === 'pending' ? r.status === 'PENDING' : r.status !== 'PENDING').map((refund) => (
                                <tr key={refund.id} className="hover:bg-gray-50 group">
                                    <td className="px-6 py-4 font-mono text-gray-600">{refund.id}</td>
                                    <td className="px-6 py-4 font-medium text-gray-900">{refund.user}</td>
                                    <td className="px-6 py-4 text-gray-500">{refund.date}</td>
                                    <td className="px-6 py-4 text-gray-700">{refund.reason}</td>
                                    <td className="px-6 py-4 text-right font-medium text-gray-900">
                                        ${refund.amount.toFixed(2)}
                                    </td>
                                    <td className="px-6 py-4 text-center">
                                        <span className={`px-2 py-1 rounded-full text-xs font-semibold
                                            ${refund.status === 'APPROVED' ? 'bg-green-50 text-[#00BFA6]' :
                                                refund.status === 'PENDING' ? 'bg-yellow-50 text-yellow-600' : 'bg-red-50 text-red-600'}`
                                        }>
                                            {refund.status}
                                        </span>
                                    </td>
                                    <td className="px-6 py-4 text-right flex justify-end gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                        {refund.status === 'PENDING' && (
                                            <>
                                                <button className="p-1.5 text-white bg-[#00BFA6] hover:bg-[#00A38D] rounded shadow-sm" title="Approve">
                                                    <Check className="w-4 h-4" />
                                                </button>
                                                <button className="p-1.5 text-white bg-red-500 hover:bg-red-600 rounded shadow-sm" title="Reject">
                                                    <X className="w-4 h-4" />
                                                </button>
                                            </>
                                        )}
                                    </td>
                                </tr>
                            ))}
                            {mockRefunds.filter(r => selectedTab === 'pending' ? r.status === 'PENDING' : r.status !== 'PENDING').length === 0 && (
                                <tr>
                                    <td colSpan={7} className="px-6 py-12 text-center text-gray-400 bg-white">
                                        <Check className="w-12 h-12 mx-auto mb-3 text-gray-200" />
                                        <p>No refund requests pending. Good job!</p>
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    );
};

export default RefundsPage;
