import React, { useState } from "react";
import {
  IndianRupee, TrendingUp, AlertCircle, CheckCircle, Clock,
  Download, Filter, Search, FileText, CreditCard, Users,
  RefreshCw, Shield, Calculator, Receipt, AlertTriangle, Activity
} from "lucide-react";
import { AdminPropertyPurchasePayments } from "./AdminPropertyPurchasePayments";
import { SecurePaymentGateway } from "./SecurePaymentGateway";
import { MultiChannelPaymentAdmin } from "./MultiChannelPaymentAdmin";

export function FinancialsPayments() {
  const [activeTab, setActiveTab] = useState("overview");
  const [searchTerm, setSearchTerm] = useState("");
  const [dateFilter, setDateFilter] = useState("all");

  const tabs = [
    { id: "overview", label: "Overview", icon: TrendingUp },
    { id: "multi_channel", label: "Multi-Channel", icon: Activity },
    { id: "gateway", label: "Payment Gateway", icon: Shield },
    { id: "token", label: "Token Payments", icon: IndianRupee },
    { id: "rental", label: "Rental Payments", icon: Clock },
    { id: "transactions", label: "Transactions", icon: FileText },
    { id: "payouts", label: "Seller Payouts", icon: Users },
    { id: "commission", label: "Commissions", icon: Calculator },
    { id: "refunds", label: "Refunds", icon: RefreshCw },
    { id: "reconciliation", label: "Reconciliation", icon: CheckCircle },
    { id: "reports", label: "Reports", icon: Download },
    { id: "tax", label: "Tax/GST", icon: Receipt },
    { id: "disputes", label: "Disputes", icon: AlertTriangle }
  ];

  // Mock data
  const stats = {
    totalRevenue: "₹12.5L",
    pendingPayments: "₹2.3L",
    commission: "₹1.8L",
    refunds: "₹45K",
    taxCollected: "₹2.1L"
  };

  const renderOverview = () => (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
        <StatCard title="Total Revenue" value={stats.totalRevenue} change="+18%" positive icon={IndianRupee} />
        <StatCard title="Pending Payments" value={stats.pendingPayments} count="12 pending" icon={Clock} />
        <StatCard title="Commission Earned" value={stats.commission} change="+5%" positive icon={Calculator} />
        <StatCard title="Refunds" value={stats.refunds} count="8 requests" icon={RefreshCw} />
        <StatCard title="Tax Collected" value={stats.taxCollected} change="+12%" positive icon={Receipt} />
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div className="bg-white rounded-lg shadow p-6">
          <h3 className="text-lg font-semibold mb-4">Recent Transactions</h3>
          <TransactionTable limit={5} />
        </div>
        <div className="bg-white rounded-lg shadow p-6">
          <h3 className="text-lg font-semibold mb-4">Pending Actions</h3>
          <PendingActions />
        </div>
      </div>
    </div>
  );

  const renderTokenPayments = () => (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h2 className="text-xl font-bold">Token Payments Monitoring</h2>
        <button className="btn-primary">Export Data</button>
      </div>
      <FilterBar searchTerm={searchTerm} setSearchTerm={setSearchTerm} dateFilter={dateFilter} setDateFilter={setDateFilter} />
      <DataTable
        headers={["Token ID", "Property", "Buyer", "Amount", "Status", "Date", "Actions"]}
        data={mockTokenPayments}
        type="token"
      />
    </div>
  );

  const renderRentalPayments = () => (
    <div className="space-y-4">
      <h2 className="text-xl font-bold">Rental Payments Tracking</h2>
      <FilterBar searchTerm={searchTerm} setSearchTerm={setSearchTerm} dateFilter={dateFilter} setDateFilter={setDateFilter} />
      <DataTable
        headers={["Rental ID", "Property", "Tenant", "Amount", "Due Date", "Status", "Actions"]}
        data={mockRentalPayments}
        type="rental"
      />
    </div>
  );

  const renderPurchasePayments = () => (
    <AdminPropertyPurchasePayments />
  );

  const renderTransactions = () => (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h2 className="text-xl font-bold">All Transactions</h2>
        <button className="btn-primary flex items-center gap-2">
          <Download size={16} /> Export
        </button>
      </div>
      <FilterBar searchTerm={searchTerm} setSearchTerm={setSearchTerm} dateFilter={dateFilter} setDateFilter={setDateFilter} />
      <TransactionTable />
    </div>
  );

  const renderPayouts = () => (
    <div className="space-y-4">
      <h2 className="text-xl font-bold">Seller Payouts Management</h2>
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
        <StatCard title="Pending Payouts" value="₹8.5L" count="23 sellers" icon={Clock} />
        <StatCard title="Processed Today" value="₹3.2L" count="12 payouts" icon={CheckCircle} />
        <StatCard title="Total This Month" value="₹45.8L" count="156 payouts" icon={IndianRupee} />
      </div>
      <DataTable
        headers={["Payout ID", "Seller", "Amount", "Property", "Status", "Scheduled Date", "Actions"]}
        data={mockPayouts}
        type="payout"
      />
    </div>
  );

  const renderCommission = () => (
    <div className="space-y-4">
      <h2 className="text-xl font-bold">Commission Calculation & Tracking</h2>
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <StatCard title="Total Commission" value="₹1.8L" change="+5%" positive icon={Calculator} />
        <StatCard title="Agent Commission" value="₹1.2L" count="45 agents" icon={Users} />
        <StatCard title="Platform Fee" value="₹0.6L" icon={IndianRupee} />
        <StatCard title="Pending" value="₹0.3L" count="8 pending" icon={Clock} />
      </div>
      <DataTable
        headers={["Transaction ID", "Property", "Type", "Base Amount", "Rate", "Commission", "Agent"]}
        data={mockCommissions}
        type="commission"
      />
    </div>
  );

  const renderRefunds = () => (
    <div className="space-y-4">
      <h2 className="text-xl font-bold">Refund Processing</h2>
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
        <StatCard title="Pending Refunds" value="₹45K" count="8 requests" icon={Clock} />
        <StatCard title="Processed Today" value="₹12K" count="3 refunds" icon={CheckCircle} />
        <StatCard title="Total This Month" value="₹1.2L" count="28 refunds" icon={RefreshCw} />
      </div>
      <DataTable
        headers={["Refund ID", "User", "Amount", "Reason", "Request Date", "Status", "Actions"]}
        data={mockRefunds}
        type="refund"
      />
    </div>
  );

  const renderReconciliation = () => (
    <div className="space-y-4">
      <h2 className="text-xl font-bold">Payment Gateway Reconciliation</h2>
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <StatCard title="Matched" value="₹10.2L" count="245 txns" icon={CheckCircle} />
        <StatCard title="Mismatched" value="₹15K" count="3 txns" icon={AlertCircle} />
        <StatCard title="Pending" value="₹8K" count="2 txns" icon={Clock} />
        <StatCard title="Failed" value="₹5K" count="1 txn" icon={AlertTriangle} />
      </div>
      <DataTable
        headers={["Gateway", "Transaction ID", "Amount", "Platform Amount", "Status", "Date", "Actions"]}
        data={mockReconciliation}
        type="reconciliation"
      />
    </div>
  );

  const renderReports = () => (
    <div className="space-y-6">
      <h2 className="text-xl font-bold">Payment Reports & Export</h2>
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <ReportCard title="Daily Report" description="Today's transactions and revenue" />
        <ReportCard title="Weekly Report" description="Last 7 days summary" />
        <ReportCard title="Monthly Report" description="Current month financial overview" />
        <ReportCard title="Quarterly Report" description="Q4 2024 comprehensive report" />
        <ReportCard title="Annual Report" description="Year 2024 financial summary" />
        <ReportCard title="Custom Report" description="Build your own custom report" />
      </div>
    </div>
  );

  const renderTax = () => (
    <div className="space-y-4">
      <h2 className="text-xl font-bold">GST/Tax Compliance</h2>
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <StatCard title="GST Collected" value="₹2.1L" change="+8%" positive icon={Receipt} />
        <StatCard title="TDS Deducted" value="₹45K" icon={Calculator} />
        <StatCard title="Tax Liability" value="₹1.8L" icon={AlertCircle} />
        <StatCard title="Compliance Score" value="98%" icon={Shield} />
      </div>
      <DataTable
        headers={["Invoice ID", "Transaction", "GST Amount", "TDS", "Total Tax", "Status", "Actions"]}
        data={mockTaxData}
        type="tax"
      />
    </div>
  );

  const renderDisputes = () => (
    <div className="space-y-4">
      <h2 className="text-xl font-bold">Financial Dispute Handling</h2>
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
        <StatCard title="Open Disputes" value="5" icon={AlertTriangle} />
        <StatCard title="In Progress" value="3" icon={Clock} />
        <StatCard title="Resolved Today" value="2" icon={CheckCircle} />
        <StatCard title="Avg Resolution Time" value="2.5 days" icon={TrendingUp} />
      </div>
      <DataTable
        headers={["Dispute ID", "User", "Type", "Amount", "Status", "Priority", "Created", "Actions"]}
        data={mockDisputes}
        type="dispute"
      />
    </div>
  );

  const renderPaymentGateway = () => (
    <SecurePaymentGateway />
  );

  const renderMultiChannel = () => (
    <MultiChannelPaymentAdmin />
  );

  const renderContent = () => {
    switch (activeTab) {
      case "overview": return renderOverview();
      case "multi_channel": return renderMultiChannel();
      case "gateway": return renderPaymentGateway();
      case "token": return renderTokenPayments();
      case "rental": return renderRentalPayments();
      case "transactions": return renderTransactions();
      case "payouts": return renderPayouts();
      case "commission": return renderCommission();
      case "refunds": return renderRefunds();
      case "reconciliation": return renderReconciliation();
      case "reports": return renderReports();
      case "tax": return renderTax();
      case "disputes": return renderDisputes();
      default: return renderOverview();
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-2xl font-bold text-gray-900">Financials & Payments</h1>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow overflow-x-auto">
        <div className="flex border-b">
          {tabs.map((tab) => {
            const Icon = tab.icon;
            return (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id)}
                className={`flex items-center gap-2 px-4 py-3 border-b-2 transition-colors whitespace-nowrap ${activeTab === tab.id
                  ? "border-blue-600 text-blue-600 bg-blue-50"
                  : "border-transparent text-gray-600 hover:text-gray-900 hover:bg-gray-50"
                  }`}
              >
                <Icon size={16} />
                <span className="text-sm font-medium">{tab.label}</span>
              </button>
            );
          })}
        </div>
      </div>

      {/* Content */}
      <div>{renderContent()}</div>
    </div>
  );
}

// Helper Components
function StatCard({ title, value, change, positive, count, icon: Icon }: any) {
  return (
    <div className="bg-white p-6 rounded-lg shadow">
      <div className="flex items-center justify-between mb-2">
        <h3 className="text-sm text-gray-600">{title}</h3>
        {Icon && <Icon size={20} className="text-gray-400" />}
      </div>
      <p className="text-2xl font-bold text-gray-900">{value}</p>
      {change && (
        <p className={`text-xs ${positive ? "text-green-600" : "text-red-600"}`}>
          {change} from last month
        </p>
      )}
      {count && <p className="text-xs text-gray-600">{count}</p>}
    </div>
  );
}

function FilterBar({ searchTerm, setSearchTerm, dateFilter, setDateFilter }: any) {
  return (
    <div className="bg-white p-4 rounded-lg shadow flex gap-4 items-center">
      <div className="flex-1 relative">
        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" size={16} />
        <input
          type="text"
          placeholder="Search..."
          value={searchTerm}
          onChange={(e) => setSearchTerm(e.target.value)}
          className="w-full pl-10 pr-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500"
        />
      </div>
      <select
        value={dateFilter}
        onChange={(e) => setDateFilter(e.target.value)}
        className="px-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500"
      >
        <option value="all">All Time</option>
        <option value="today">Today</option>
        <option value="week">This Week</option>
        <option value="month">This Month</option>
        <option value="quarter">This Quarter</option>
      </select>
    </div>
  );
}

function DataTable({ headers, data, type }: any) {
  return (
    <div className="bg-white rounded-lg shadow overflow-x-auto">
      <table className="w-full">
        <thead className="bg-gray-50 border-b">
          <tr>
            {headers.map((header: string, i: number) => (
              <th key={i} className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                {header}
              </th>
            ))}
          </tr>
        </thead>
        <tbody className="divide-y divide-gray-200">
          {data.map((row: any, i: number) => (
            <tr key={i} className="hover:bg-gray-50">
              {Object.values(row).map((cell: any, j: number) => (
                <td key={j} className="px-6 py-4 text-sm text-gray-900 whitespace-nowrap">
                  {typeof cell === 'object' ? cell : cell}
                </td>
              ))}
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
}

function TransactionTable({ limit }: any) {
  const data = mockTransactions.slice(0, limit || mockTransactions.length);
  return (
    <div className="space-y-2">
      {data.map((txn: any, i: number) => (
        <div key={i} className="flex justify-between items-center p-3 border rounded-lg hover:bg-gray-50">
          <div>
            <p className="font-medium text-sm">{txn.id}</p>
            <p className="text-xs text-gray-600">{txn.user}</p>
          </div>
          <div className="text-right">
            <p className="font-semibold">{txn.amount}</p>
            <StatusBadge status={txn.status} />
          </div>
        </div>
      ))}
    </div>
  );
}

function PendingActions() {
  const actions = [
    { type: "Refund Request", count: 8, priority: "high" },
    { type: "Payout Approval", count: 23, priority: "medium" },
    { type: "Dispute Resolution", count: 5, priority: "high" },
    { type: "Tax Filing", count: 1, priority: "low" }
  ];

  return (
    <div className="space-y-3">
      {actions.map((action, i) => (
        <div key={i} className="flex justify-between items-center p-3 border rounded-lg">
          <div>
            <p className="font-medium text-sm">{action.type}</p>
            <p className="text-xs text-gray-600">{action.count} pending</p>
          </div>
          <span className={`px-2 py-1 rounded text-xs font-medium ${action.priority === 'high' ? 'bg-red-100 text-red-800' :
            action.priority === 'medium' ? 'bg-yellow-100 text-yellow-800' :
              'bg-green-100 text-green-800'
            }`}>
            {action.priority}
          </span>
        </div>
      ))}
    </div>
  );
}

function ReportCard({ title, description }: any) {
  return (
    <div className="bg-white p-6 rounded-lg shadow hover:shadow-lg transition-shadow">
      <h3 className="font-semibold mb-2">{title}</h3>
      <p className="text-sm text-gray-600 mb-4">{description}</p>
      <button className="btn-primary w-full flex items-center justify-center gap-2">
        <Download size={16} /> Generate Report
      </button>
    </div>
  );
}

function StatusBadge({ status }: any) {
  const colors: any = {
    success: "bg-green-100 text-green-800",
    pending: "bg-yellow-100 text-yellow-800",
    failed: "bg-red-100 text-red-800",
    processing: "bg-blue-100 text-blue-800"
  };
  return (
    <span className={`px-2 py-1 rounded text-xs font-medium ${colors[status] || colors.pending}`}>
      {status}
    </span>
  );
}

// Mock Data
const mockTransactions = [
  { id: "#TXN001", user: "John Doe", amount: "₹25,000", status: "success", date: "2024-12-05" },
  { id: "#TXN002", user: "Jane Smith", amount: "₹15,000", status: "pending", date: "2024-12-04" },
  { id: "#TXN003", user: "Bob Wilson", amount: "₹50,000", status: "success", date: "2024-12-03" },
  { id: "#TXN004", user: "Alice Brown", amount: "₹8,000", status: "failed", date: "2024-12-02" },
  { id: "#TXN005", user: "Charlie Davis", amount: "₹32,000", status: "success", date: "2024-12-01" }
];

const mockTokenPayments = [
  { id: "TKN001", property: "Luxury Villa, Goa", buyer: "John Doe", amount: "₹5L", status: <StatusBadge status="success" />, date: "2024-12-05", actions: <button className="text-blue-600 text-sm">View</button> },
  { id: "TKN002", property: "Beach House, Mumbai", buyer: "Jane Smith", amount: "₹3L", status: <StatusBadge status="pending" />, date: "2024-12-04", actions: <button className="text-blue-600 text-sm">Verify</button> }
];

const mockRentalPayments = [
  { id: "RNT001", property: "Apartment 2BHK", tenant: "Bob Wilson", amount: "₹25K", due: "2024-12-10", status: <StatusBadge status="pending" />, actions: <button className="text-blue-600 text-sm">Remind</button> },
  { id: "RNT002", property: "Studio Flat", tenant: "Alice Brown", amount: "₹18K", due: "2024-12-05", status: <StatusBadge status="success" />, actions: <button className="text-blue-600 text-sm">View</button> }
];

const mockPurchasePayments = [
  { id: "PUR001", property: "Villa, Bangalore", buyer: "Charlie Davis", total: "₹2.5Cr", paid: "₹1.5Cr", pending: "₹1Cr", status: <StatusBadge status="processing" /> },
  { id: "PUR002", property: "Penthouse, Delhi", buyer: "David Lee", total: "₹3Cr", paid: "₹3Cr", pending: "₹0", status: <StatusBadge status="success" /> }
];

const mockPayouts = [
  { id: "PAY001", seller: "Raj Kumar", amount: "₹45L", property: "Villa, Goa", status: <StatusBadge status="pending" />, date: "2024-12-10", actions: <button className="btn-primary text-xs">Approve</button> },
  { id: "PAY002", seller: "Priya Sharma", amount: "₹32L", property: "Apartment, Mumbai", status: <StatusBadge status="success" />, date: "2024-12-05", actions: <button className="text-blue-600 text-sm">View</button> }
];

const mockCommissions = [
  { id: "COM001", property: "Villa, Goa", type: "Sale", base: "₹2.5Cr", rate: "2%", commission: "₹5L", agent: "Agent A" },
  { id: "COM002", property: "Apartment, Delhi", type: "Rental", base: "₹25K", rate: "10%", commission: "₹2.5K", agent: "Agent B" }
];

const mockRefunds = [
  { id: "REF001", user: "John Doe", amount: "₹5K", reason: "Booking Cancelled", date: "2024-12-04", status: <StatusBadge status="pending" />, actions: <button className="btn-primary text-xs">Process</button> },
  { id: "REF002", user: "Jane Smith", amount: "₹3K", reason: "Duplicate Payment", date: "2024-12-03", status: <StatusBadge status="success" />, actions: <button className="text-blue-600 text-sm">View</button> }
];

const mockReconciliation = [
  { gateway: "Razorpay", txnId: "rzp_001", amount: "₹25K", platformAmount: "₹25K", status: <StatusBadge status="success" />, date: "2024-12-05", actions: <button className="text-green-600 text-sm">Matched</button> },
  { gateway: "Paytm", txnId: "ptm_002", amount: "₹15K", platformAmount: "₹15.5K", status: <span className="text-red-600 text-sm">Mismatch</span>, date: "2024-12-04", actions: <button className="text-blue-600 text-sm">Resolve</button> }
];

const mockTaxData = [
  { invoice: "INV001", transaction: "TXN001", gst: "₹4.5K", tds: "₹1K", total: "₹5.5K", status: <StatusBadge status="success" />, actions: <button className="text-blue-600 text-sm">Download</button> },
  { invoice: "INV002", transaction: "TXN002", gst: "₹2.7K", tds: "₹0.5K", total: "₹3.2K", status: <StatusBadge status="pending" />, actions: <button className="text-blue-600 text-sm">Generate</button> }
];

const mockDisputes = [
  { id: "DIS001", user: "John Doe", type: "Payment Failure", amount: "₹25K", status: <StatusBadge status="pending" />, priority: <span className="text-red-600 font-medium">High</span>, created: "2024-12-04", actions: <button className="btn-primary text-xs">Resolve</button> },
  { id: "DIS002", user: "Jane Smith", type: "Wrong Amount", amount: "₹5K", status: <span className="text-blue-600 text-sm">In Progress</span>, priority: <span className="text-yellow-600 font-medium">Medium</span>, created: "2024-12-03", actions: <button className="text-blue-600 text-sm">View</button> }
];
