import React, { useState, useEffect } from "react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../ui/tabs";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "../ui/card";
import { Building2, Construction, MapPin, TrendingUp, BarChart3, Activity } from "lucide-react";
import { MetroProjects } from "./hyperlocal/MetroProjects";
import { InfrastructureProjects } from "./hyperlocal/InfrastructureProjects";
import { ZoningChanges } from "./hyperlocal/ZoningChanges";

interface HyperlocalStats {
  total_metro_projects: number;
  total_infrastructure_projects: number;
  total_zoning_changes: number;
  total_impact_areas: number;
}

export function HyperlocalEconomic() {
  const [activeTab, setActiveTab] = useState("metro");
  const [isGoogleMapsLoaded, setIsGoogleMapsLoaded] = useState(false);
  const [stats, setStats] = useState<HyperlocalStats>({
    total_metro_projects: 0,
    total_infrastructure_projects: 0,
    total_zoning_changes: 0,
    total_impact_areas: 0
  });
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const checkGoogleMaps = () => {
      if (window.google?.maps) {
        setIsGoogleMapsLoaded(true);
        return;
      }
      setTimeout(checkGoogleMaps, 100);
    };
    checkGoogleMaps();
  }, []);

  useEffect(() => {
    const fetchStats = async () => {
      try {
        const token = localStorage.getItem('access_token');
        const response = await fetch('http://localhost:8090/api/admin/dashboard/stats', {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        });
        
        if (response.ok) {
          const data = await response.json();
          if (data.hyperlocal_stats) {
            setStats(data.hyperlocal_stats);
          }
        }
      } catch (error) {
        console.error('Failed to fetch hyperlocal stats:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchStats();
  }, []);

  const statsCards = [
    {
      title: "Active Metro Projects",
      value: loading ? "..." : stats.total_metro_projects.toString(),
      change: "+0 this month",
      icon: Building2,
      color: "text-blue-600"
    },
    {
      title: "Infrastructure Projects",
      value: loading ? "..." : stats.total_infrastructure_projects.toString(),
      change: "+0 this month",
      icon: Construction,
      color: "text-orange-600"
    },
    {
      title: "Zoning Changes",
      value: loading ? "..." : stats.total_zoning_changes.toString(),
      change: "+0 this month",
      icon: MapPin,
      color: "text-green-600"
    },
    {
      title: "Total Impact Areas",
      value: loading ? "..." : stats.total_impact_areas.toString(),
      change: "+0 this month",
      icon: Activity,
      color: "text-purple-600"
    }
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center gap-3">
        <TrendingUp className="h-8 w-8 text-blue-600" />
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Hyperlocal Economic Indicators</h1>
          <p className="text-gray-600">Monitor infrastructure developments and zoning changes affecting property values</p>
        </div>
      </div>

      {/* Stats Overview */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {statsCards.map((stat, index) => {
          const IconComponent = stat.icon;
          return (
            <Card key={index}>
              <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                <CardTitle className="text-sm font-medium text-gray-600">
                  {stat.title}
                </CardTitle>
                <IconComponent className={`h-4 w-4 ${stat.color}`} />
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold">{stat.value}</div>
                <p className="text-xs text-gray-500 mt-1">
                  {stat.change}
                </p>
              </CardContent>
            </Card>
          );
        })}
      </div>

      {/* Main Content Tabs */}
      <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="metro" className="flex items-center gap-2">
            <Building2 className="h-4 w-4" />
            Metro Projects
          </TabsTrigger>
          <TabsTrigger value="infrastructure" className="flex items-center gap-2">
            <Construction className="h-4 w-4" />
            Infrastructure Projects
          </TabsTrigger>
          <TabsTrigger value="zoning" className="flex items-center gap-2">
            <MapPin className="h-4 w-4" />
            Zoning Changes
          </TabsTrigger>
        </TabsList>

        <TabsContent value="metro" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Building2 className="h-5 w-5 text-blue-600" />
                Metro Projects Management
              </CardTitle>
              <CardDescription>
                Track metro line developments and their impact on surrounding property values
              </CardDescription>
            </CardHeader>
            <CardContent>
              <MetroProjects />
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="infrastructure" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Construction className="h-5 w-5 text-orange-600" />
                Infrastructure Projects Management
              </CardTitle>
              <CardDescription>
                Monitor infrastructure developments including roads, bridges, airports, and commercial projects
              </CardDescription>
            </CardHeader>
            <CardContent>
              <InfrastructureProjects />
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="zoning" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <MapPin className="h-5 w-5 text-green-600" />
                Zoning Changes Management
              </CardTitle>
              <CardDescription>
                Track zoning modifications and regulatory changes affecting property development
              </CardDescription>
            </CardHeader>
            <CardContent>
              <ZoningChanges />
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}