import React, { useState } from 'react';
import { ChevronDown, User, ShoppingBag, Briefcase } from 'lucide-react';
import { simulateRole } from '../../api/impersonation';

interface ImpersonateButtonProps {
  className?: string;
}

export const ImpersonateButton: React.FC<ImpersonateButtonProps> = ({ className }) => {
  const [isOpen, setIsOpen] = useState(false);
  const [isLoading, setIsLoading] = useState(false);

  const handleSimulate = async (role: 'buyer' | 'seller' | 'agent') => {
    setIsLoading(true);
    try {
      const response = await simulateRole(role);

      const currentToken = localStorage.getItem('access_token');
      if (currentToken) {
        localStorage.setItem('admin_restore_token', currentToken);
      }

      localStorage.setItem('access_token', response.access_token);
      localStorage.setItem('refresh_token', response.refresh_token);

      const dashboardMap = {
        buyer: '/user/dashboard',
        seller: '/seller/dashboard',
        agent: '/agent/dashboard'
      };

      window.location.href = dashboardMap[role];
    } catch (error: any) {
      console.error('Failed to simulate role:', error);
      alert(error.message || 'Failed to switch role');
      setIsLoading(false);
    }
  };

  return (
    <div className="relative">
      <button
        onClick={() => setIsOpen(!isOpen)}
        disabled={isLoading}
        className={`flex items-center gap-2 px-4 py-2 bg-[#0056D2] hover:bg-[#0056D2]/90 text-white rounded-lg font-medium text-sm transition-all shadow-md hover:shadow-lg disabled:opacity-50 ${className}`}
      >
        Impersonate <ChevronDown className="h-4 w-4" />
      </button>

      {isOpen && (
        <>
          <div className="fixed inset-0 z-40" onClick={() => setIsOpen(false)} />
          <div className="absolute left-0 right-0 mt-2 w-full bg-white rounded-lg shadow-2xl border border-gray-200 z-50 overflow-hidden">
            <div className="py-2">
              <div className="px-4 py-2 text-xs font-semibold text-gray-500 uppercase tracking-wider border-b border-gray-100">
                Switch Role
              </div>
              <button
                onClick={() => { handleSimulate('buyer'); setIsOpen(false); }}
                disabled={isLoading}
                className="w-full flex items-center gap-3 px-4 py-3 text-sm text-gray-700 hover:bg-blue-50 hover:text-[#0056D2] transition-colors disabled:opacity-50 group"
              >
                <div className="p-1.5 rounded-md bg-blue-50 group-hover:bg-[#0056D2] transition-colors">
                  <ShoppingBag className="h-4 w-4 text-[#0056D2] group-hover:text-white" />
                </div>
                <span className="font-medium">Buyer</span>
              </button>
              <button
                onClick={() => { handleSimulate('seller'); setIsOpen(false); }}
                disabled={isLoading}
                className="w-full flex items-center gap-3 px-4 py-3 text-sm text-gray-700 hover:bg-blue-50 hover:text-[#0056D2] transition-colors disabled:opacity-50 group"
              >
                <div className="p-1.5 rounded-md bg-blue-50 group-hover:bg-[#0056D2] transition-colors">
                  <Briefcase className="h-4 w-4 text-[#0056D2] group-hover:text-white" />
                </div>
                <span className="font-medium">Seller</span>
              </button>
              <button
                onClick={() => { handleSimulate('agent'); setIsOpen(false); }}
                disabled={isLoading}
                className="w-full flex items-center gap-3 px-4 py-3 text-sm text-gray-700 hover:bg-blue-50 hover:text-[#0056D2] transition-colors disabled:opacity-50 group"
              >
                <div className="p-1.5 rounded-md bg-blue-50 group-hover:bg-[#0056D2] transition-colors">
                  <User className="h-4 w-4 text-[#0056D2] group-hover:text-white" />
                </div>
                <span className="font-medium">Agent</span>
              </button>
            </div>
          </div>
        </>
      )}
    </div>
  );
};
