import React, { useState, useEffect } from "react";
import {
  MessageSquare,
  Building,
  X,
  Mail,
  Phone,
  Check,
  CheckCheck,
  Calendar,
  Search,
  Filter,
  Archive,
  ArchiveRestore,
  Eye,
  Edit3,
} from "lucide-react";
import {
  getAllBookings,
  updateBookingStatus,
  VisitBooking,
} from "../../api/visitBookings";

interface Lead {
  id: string;
  name: string;
  email: string;
  phone: string;
  buyer_name?: string;
  buyer_email?: string;
  buyer_phone?: string;
  property_id: number;
  property_title?: string;
  property_type?: string;
  property_location?: string;
  message?: string;
  budget?: string;
  preferred_contact: string;
  lead_type: string;
  status: string;
  score?: number;
  ai_category?: string;
  agent_name?: string;
  agent_id?: string;
  agent_email?: string;
  agent_phone?: string;
  seller_name?: string;
  seller_id?: string;
  seller_email?: string;
  seller_phone?: string;
  owner_id?: string;
  user_role?: string;
  prospect_at?: string;
  contacted_at?: string;
  created_at: string;
  updated_at: string;
}

const API_BASE = import.meta.env.VITE_API_BASE || "http://localhost:8090/api";

export const InquiryManagement: React.FC = () => {
  const [activeTab, setActiveTab] = useState<"leads" | "itinerary">("leads");
  const [leads, setLeads] = useState<Lead[]>([]);
  const [visitBookings, setVisitBookings] = useState<VisitBooking[]>([]);
  const [selectedLead, setSelectedLead] = useState<Lead | null>(null);
  const [selectedBooking, setSelectedBooking] = useState<VisitBooking | null>(
    null
  );
  const [loading, setLoading] = useState(true);
  const [updatingStatus, setUpdatingStatus] = useState(false);
  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState("all");
  const [currentPage, setCurrentPage] = useState(1);
  const leadsPerPage = 10;

  useEffect(() => {
    if (activeTab === "leads") {
      fetchLeads();
    } else if (activeTab === "itinerary") {
      fetchVisitBookings();
    }
  }, [activeTab]);

  useEffect(() => {
    fetchLeads();
    // Auto-refresh every 30 seconds to get latest status updates
    const interval = setInterval(fetchLeads, 30000);
    return () => clearInterval(interval);
  }, []);

  const fetchLeads = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem("access_token");

      // Try the admin-specific endpoint first
      let response = await fetch(`${API_BASE}/leads-management/all`, {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
      });

      // If admin endpoint fails, try the general all leads endpoint
      if (!response.ok) {
        console.log("Admin endpoint failed, trying general endpoint...");
        response = await fetch(`${API_BASE}/leads-management/all`, {
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "application/json",
          },
        });
      }

      if (response.ok) {
        const data = await response.json();
        let leadsData = Array.isArray(data) ? data : data.leads || [];

        // Enhance leads with agent/seller information
        leadsData = await Promise.all(
          leadsData.map(async (lead) => {
            try {
              // Fetch property details to determine if it's seller or agent property
              const propertyResponse = await fetch(
                `${API_BASE}/properties/${lead.property_id}`,
                {
                  headers: {
                    Authorization: `Bearer ${token}`,
                    "Content-Type": "application/json",
                  },
                }
              );

              if (propertyResponse.ok) {
                const propertyData = await propertyResponse.json();
                console.log(`Property ${lead.property_id} data:`, propertyData);
                
                // Check for listed_by_id (used for both sellers and agents)
                const listedById = propertyData.listed_by_id || propertyData.listedById;
                const listedByType = propertyData.listed_by_type || propertyData.listedByType;
                
                if (listedById) {
                  console.log(`Property ${lead.property_id} has listed_by_id:`, listedById, `type:`, listedByType);
                  // Fetch user details
                  const userResponse = await fetch(
                    `${API_BASE}/profile/${listedById}`,
                    {
                      headers: {
                        Authorization: `Bearer ${token}`,
                        "Content-Type": "application/json",
                      },
                    }
                  );
                  
                  if (userResponse.ok) {
                    const userData = await userResponse.json();
                    const fullName = `${userData.first_name || ""} ${userData.last_name || ""}`.trim();
                    
                    // Use listed_by_type from property to determine role
                    if (listedByType === 'agent') {
                      lead.agent_name = fullName || "Agent";
                      lead.agent_email = userData.email;
                      lead.agent_phone = userData.phone_number;
                      lead.agent_id = listedById;
                      console.log(`Agent assigned to lead ${lead.id}:`, lead.agent_name);
                    } else {
                      lead.seller_name = fullName || "Property Owner";
                      lead.seller_email = userData.email;
                      lead.seller_phone = userData.phone_number;
                      lead.seller_id = listedById;
                      console.log(`Seller assigned to lead ${lead.id}:`, lead.seller_name);
                    }
                  }
                } else {
                  console.log(`Property ${lead.property_id} has no listed_by_id - setting default`);
                  lead.seller_name = "Property Owner";
                }
              } else {
                console.error(`Failed to fetch property ${lead.property_id}:`, propertyResponse.status);
                lead.seller_name = "Property Owner";
              }

              return lead;
            } catch (error) {
              console.error(
                "Error fetching property/user details for lead:",
                lead.id,
                error
              );
              return lead;
            }
          })
        );

        console.log("Admin fetched leads:", leadsData.length);
        if (leadsData.length > 0) {
          console.log("Sample lead data:", leadsData[0]);
        }
        setLeads(leadsData);
      } else {
        console.error("Failed to fetch leads:", response.status);
        setLeads([]);
      }
    } catch (error) {
      console.error("Error fetching leads:", error);
      setLeads([]);
    } finally {
      setLoading(false);
    }
  };

  const fetchVisitBookings = async () => {
    try {
      setLoading(true);
      console.log("🔍 Admin fetching all visit bookings...");
      const bookings = await getAllBookings();
      console.log("📊 Admin received bookings:", bookings);
      setVisitBookings(bookings);
    } catch (error) {
      console.error("❌ Admin error fetching visit bookings:", error);
      setVisitBookings([]);
    } finally {
      setLoading(false);
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const calculateLeadStatus = (lead: Lead) => {
    // Priority Order: Archived > Contacted > Prospect > Lead
    if (lead.status === "archived" || lead.status === "ARCHIVED") {
      return {
        status: "Archived",
        engagement: "Closed"
      };
    }
    
    if (lead.status === "contacted" || lead.status === "CONTACTED" || lead.contacted_at) {
      const role = lead.seller_name ? "Seller" : lead.agent_name ? "Agent" : "Representative";
      return {
        status: "Contacted",
        engagement: `${role} Engaged`
      };
    }
    
    if (lead.status === "prospect" || lead.status === "PROSPECT" || lead.prospect_at) {
      return {
        status: "Prospect",
        engagement: "Buyer Interested"
      };
    }
    
    return {
      status: "Lead",
      engagement: "Initial Inquiry"
    };
  };

  const refreshLeadData = async (leadId: string) => {
    try {
      const token = localStorage.getItem("access_token");
      const response = await fetch(`${API_BASE}/leads-management/leads/${leadId}`, {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
      });
      
      if (response.ok) {
        const updatedLead = await response.json();
        // Update the lead in the list
        setLeads(prev => prev.map(lead => lead.id === leadId ? updatedLead : lead));
        // Update selected lead if it's the one being viewed
        if (selectedLead && selectedLead.id === leadId) {
          setSelectedLead(updatedLead);
        }
        return updatedLead;
      }
    } catch (error) {
      console.error("Error refreshing lead data:", error);
    }
    return null;
  };

  const updateLeadStatus = async (leadId: string, status: string) => {
    try {
      setUpdatingStatus(true);
      const token = localStorage.getItem("access_token");
      const response = await fetch(
        `${API_BASE}/leads-management/leads/${leadId}/status?status=${status}&user_id=admin`,
        {
          method: "PUT",
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "application/json",
          },
        }
      );

      if (response.ok) {
        // Update local state
        setLeads((prev) =>
          prev.map((lead) => (lead.id === leadId ? { ...lead, status } : lead))
        );
        // Update selected lead if it's the one being modified
        if (selectedLead && selectedLead.id === leadId) {
          setSelectedLead({ ...selectedLead, status });
        }
        // Refresh leads to get latest data
        await fetchLeads();
      }
    } catch (error) {
      console.error("Error updating lead status:", error);
    } finally {
      setUpdatingStatus(false);
    }
  };

  const filteredLeads = leads.filter((lead) => {
    const matchesSearch =
      searchTerm === "" ||
      lead.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      lead.phone.includes(searchTerm) ||
      lead.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
      lead.property_title?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      lead.agent_name?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      lead.seller_name?.toLowerCase().includes(searchTerm.toLowerCase());

    const matchesStatus =
      statusFilter === "all" || lead.status === statusFilter;

    return matchesSearch && matchesStatus;
  });

  const totalPages = Math.ceil(filteredLeads.length / leadsPerPage);
  const startIndex = (currentPage - 1) * leadsPerPage;
  const paginatedLeads = filteredLeads.slice(
    startIndex,
    startIndex + leadsPerPage
  );

  const leadStatusCounts = {
    all: leads.length,
    new: leads.filter((l) => l.status === "new").length,
    prospect: leads.filter((l) => l.status === "prospect").length,
    contacted: leads.filter((l) => l.status === "contacted").length,
    archived: leads.filter((l) => l.status === "archived").length,
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Inquiry Management</h1>
      </div>

      {/* Toggle Buttons */}
      <div className="flex space-x-4">
        <button
          onClick={() => setActiveTab("leads")}
          className={`flex items-center px-4 py-2 rounded-lg font-medium ${
            activeTab === "leads"
              ? "bg-blue-500 text-white"
              : "bg-gray-100 text-gray-700 hover:bg-gray-200"
          }`}
        >
          <MessageSquare className="w-4 h-4 mr-2" />
          Leads & Enquiries ({leads.length})
        </button>
        <button
          onClick={() => setActiveTab("itinerary")}
          className={`flex items-center px-4 py-2 rounded-lg font-medium ${
            activeTab === "itinerary"
              ? "bg-blue-500 text-white"
              : "bg-gray-100 text-gray-700 hover:bg-gray-200"
          }`}
        >
          <Calendar className="w-4 h-4 mr-2" />
          Itinerary Enquiries ({visitBookings.length})
        </button>
      </div>

      {/* Content */}
      {loading ? (
        <div className="flex justify-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      ) : activeTab === "leads" ? (
        <div className="bg-white rounded-lg shadow">
          {/* Search and Filters */}
          <div className="p-4 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row gap-4">
              <div className="flex-1">
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                  <input
                    type="text"
                    placeholder="Search by name, phone, email, property, agent..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10 pr-4 py-2 w-full border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  />
                </div>
              </div>
              <div className="flex gap-2">
                <select
                  value={statusFilter}
                  onChange={(e) => setStatusFilter(e.target.value)}
                  className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                >
                  <option value="all">
                    All Status ({leadStatusCounts.all})
                  </option>
                  <option value="new">Lead ({leadStatusCounts.new})</option>
                  <option value="prospect">
                    Prospect ({leadStatusCounts.prospect})
                  </option>
                  <option value="contacted">
                    Contacted ({leadStatusCounts.contacted})
                  </option>
                  <option value="archived">
                    Archived ({leadStatusCounts.archived})
                  </option>
                </select>
              </div>
            </div>
          </div>
          {/* Leads Table */}
          {paginatedLeads.length === 0 ? (
            <div className="text-center py-12">
              <MessageSquare className="mx-auto h-12 w-12 text-gray-400" />
              <h3 className="mt-2 text-sm font-medium text-gray-900">
                No leads found
              </h3>
              <p className="mt-1 text-sm text-gray-500">
                No leads match your current filters.
              </p>
            </div>
          ) : (
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Buyer
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Contact
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Property
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Agent/Seller
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Date
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {paginatedLeads.map((lead) => (
                    <tr key={lead.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">
                          {lead.name || lead.buyer_name || "No Name"}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">
                          {lead.email || lead.buyer_email || "No Email"}
                        </div>
                        <div className="text-sm text-gray-500">
                          {lead.phone || lead.buyer_phone || "No Phone"}
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="text-sm text-gray-900 max-w-xs truncate">
                          {lead.property_title ||
                            `Property #${lead.property_id}`}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">
                          {lead.seller_name
                            ? `Owner: ${lead.seller_name}`
                            : lead.agent_name
                            ? `Agent: ${lead.agent_name}`
                            : "Property Owner"}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        {(() => {
                          const { status } = calculateLeadStatus(lead);
                          return (
                            <span
                              className={`px-2 py-1 text-xs font-medium rounded-full ${
                                status === "Lead"
                                  ? "bg-blue-100 text-blue-800"
                                  : status === "Prospect"
                                  ? "bg-yellow-100 text-yellow-800"
                                  : status === "Contacted"
                                  ? "bg-green-100 text-green-800"
                                  : status === "Archived"
                                  ? "bg-gray-100 text-gray-800"
                                  : "bg-gray-100 text-gray-800"
                              }`}
                            >
                              {status}
                            </span>
                          );
                        })()}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {formatDate(lead.created_at)}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex gap-2">
                          <button
                            onClick={async () => {
                              await refreshLeadData(lead.id);
                              setSelectedLead(lead);
                            }}
                            className="text-blue-600 hover:text-blue-900 text-sm font-medium"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                          {lead.status !== "archived" ? (
                            <button
                              onClick={() =>
                                updateLeadStatus(lead.id, "archived")
                              }
                              className="text-gray-600 hover:text-gray-900 text-sm font-medium"
                            >
                              <Archive className="w-4 h-4" />
                            </button>
                          ) : (
                            <button
                              onClick={() => updateLeadStatus(lead.id, "new")}
                              className="text-green-600 hover:text-green-900 text-sm font-medium"
                            >
                              <ArchiveRestore className="w-4 h-4" />
                            </button>
                          )}
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </div>
      ) : (
        <div className="bg-white rounded-lg shadow">
          {visitBookings.length === 0 ? (
            <div className="text-center py-12">
              <Calendar className="mx-auto h-12 w-12 text-gray-400" />
              <h3 className="mt-2 text-sm font-medium text-gray-900">
                No visit bookings
              </h3>
              <p className="mt-1 text-sm text-gray-500">
                No visit bookings have been made yet.
              </p>
            </div>
          ) : (
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Visitor
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Property
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Visit Date
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {visitBookings.map((booking) => (
                    <tr key={booking.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">
                          {booking.full_name}
                        </div>
                        <div className="text-sm text-gray-500">
                          {booking.mobile_number}
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="text-sm text-gray-900">
                          Property ID: {booking.property_id}
                        </div>
                        <div className="text-sm text-gray-500">
                          {booking.purpose_of_visit}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">
                          {new Date(booking.visit_date).toLocaleDateString()}
                        </div>
                        <div className="text-sm text-gray-500">
                          {booking.visit_start_time}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span
                          className={`px-2 py-1 text-xs font-medium rounded-full ${
                            booking.status === "confirmed"
                              ? "bg-green-100 text-green-800"
                              : booking.status === "scheduled"
                              ? "bg-blue-100 text-blue-800"
                              : booking.status === "completed"
                              ? "bg-gray-100 text-gray-800"
                              : "bg-yellow-100 text-yellow-800"
                          }`}
                        >
                          {booking.status}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <button
                          onClick={() => setSelectedBooking(booking)}
                          className="text-blue-600 hover:text-blue-900 text-sm font-medium"
                        >
                          View
                        </button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </div>
      )}

      {/* Lead Details Modal */}
      {selectedLead && (
        <div className="fixed inset-0 bg-black/50 z-50 flex items-center justify-center p-4">
          <div className="bg-white rounded-lg w-full max-w-2xl max-h-[90vh] overflow-hidden">
            {/* Header */}
            <div className="flex justify-between items-center p-4 border-b bg-white">
              <h3 className="text-lg font-medium text-gray-900">
                Lead Details
              </h3>
              <button
                onClick={() => setSelectedLead(null)}
                className="px-4 py-2 bg-red-500 text-white rounded hover:bg-red-600"
              >
                Close
              </button>
            </div>

            {/* Content */}
            <div
              className="overflow-y-auto p-4"
              style={{ maxHeight: "calc(90vh - 80px)" }}
            >
              <div className="space-y-6">
                {/* Status Management & Engagement */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-3">
                    Lead Status & Engagement
                  </label>
                  <div className="bg-gray-50 p-4 rounded-lg">
                    <div className="grid grid-cols-2 gap-4 mb-4">
                      {(() => {
                        const { status, engagement } = calculateLeadStatus(selectedLead);
                        return (
                          <>
                            <div>
                              <span className="text-sm font-medium text-gray-700">
                                Current Status:
                              </span>
                              <span
                                className={`ml-2 px-2 py-1 text-xs font-medium rounded-full ${
                                  status === "Lead"
                                    ? "bg-blue-100 text-blue-800"
                                    : status === "Prospect"
                                    ? "bg-yellow-100 text-yellow-800"
                                    : status === "Contacted"
                                    ? "bg-green-100 text-green-800"
                                    : status === "Archived"
                                    ? "bg-gray-100 text-gray-800"
                                    : "bg-gray-100 text-gray-800"
                                }`}
                              >
                                {status}
                              </span>
                            </div>
                            <div>
                              <span className="text-sm font-medium text-gray-700">
                                Engagement Level:
                              </span>
                              <span className="ml-2 text-sm text-gray-900">
                                {engagement}
                              </span>
                            </div>
                          </>
                        );
                      })()}
                    </div>

                    {/* Engagement Details */}
                    {(() => {
                      const { status } = calculateLeadStatus(selectedLead);
                      return (
                        <div className="space-y-2">
                          <div className="flex items-center gap-2">
                            <span className="w-3 h-3 rounded-full bg-green-500"></span>
                            <span className="text-sm text-gray-700">
                              Buyer submitted inquiry
                            </span>
                            <span className="text-xs text-gray-500">
                              ({formatDate(selectedLead.created_at)})
                            </span>
                          </div>

                          {(status === "Prospect" || status === "Contacted") && (
                            <div className="flex items-center gap-2">
                              <span className="w-3 h-3 rounded-full bg-yellow-500"></span>
                              <span className="text-sm text-gray-700">
                                Buyer initiated contact (clicked Call/WhatsApp)
                              </span>
                              {selectedLead.prospect_at ? (
                                <span className="text-xs text-gray-500">
                                  ({formatDate(selectedLead.prospect_at)})
                                </span>
                              ) : (
                                <span className="text-xs text-gray-500">
                                  (Date not recorded)
                                </span>
                              )}
                            </div>
                          )}

                          {status === "Contacted" && (
                            <div className="flex items-center gap-2">
                              <span className="w-3 h-3 rounded-full bg-blue-500"></span>
                              <span className="text-sm text-gray-700">
                                {selectedLead.seller_name ? 'Seller' : selectedLead.agent_name ? 'Agent' : 'Representative'} contacted buyer
                              </span>
                              {selectedLead.contacted_at ? (
                                <span className="text-xs text-gray-500">
                                  ({formatDate(selectedLead.contacted_at)})
                                </span>
                              ) : (
                                <span className="text-xs text-gray-500">
                                  (Date not recorded)
                                </span>
                              )}
                            </div>
                          )}

                          {status === "Archived" && (
                            <div className="flex items-center gap-2">
                              <span className="w-3 h-3 rounded-full bg-gray-500"></span>
                              <span className="text-sm text-gray-700">
                                Lead archived by {selectedLead.archived_by_role || 'admin'}
                              </span>
                              <span className="text-xs text-gray-500">
                                ({formatDate(selectedLead.updated_at)})
                              </span>
                            </div>
                          )}
                        </div>
                      );
                    })()}

                    {/* Admin Actions */}
                    <div className="mt-4 pt-3 border-t border-gray-200">
                      <span className="text-sm font-medium text-gray-700 mb-2 block">
                        Admin Actions:
                      </span>
                      <div className="flex flex-wrap gap-2">
                        {["new", "prospect", "contacted", "archived"].map(
                          (status) => (
                            <button
                              key={status}
                              onClick={() =>
                                updateLeadStatus(selectedLead.id, status)
                              }
                              disabled={
                                updatingStatus || selectedLead.status === status
                              }
                              className={`px-3 py-1 text-xs font-medium rounded ${
                                selectedLead.status === status
                                  ? "bg-gray-300 text-gray-500 cursor-not-allowed"
                                  : "bg-blue-100 text-blue-700 hover:bg-blue-200"
                              } ${
                                updatingStatus
                                  ? "opacity-50 cursor-not-allowed"
                                  : ""
                              }`}
                            >
                              Set as{" "}
                              {status === "new"
                                ? "Lead"
                                : status === "prospect"
                                ? "Prospect"
                                : status === "contacted"
                                ? "Contacted"
                                : "Archived"}
                            </button>
                          )
                        )}
                      </div>
                    </div>
                  </div>
                </div>

                {/* 1️⃣ Buyer Information */}
                <div className="bg-blue-50 p-4 rounded-lg">
                  <h4 className="font-semibold text-gray-900 mb-3">
                    👤 Buyer Information
                  </h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Full Name
                      </label>
                      <p className="text-sm text-gray-900">
                        {selectedLead.name ||
                          selectedLead.buyer_name ||
                          "No Name"}
                      </p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Phone Number
                      </label>
                      <p className="text-sm text-gray-900">
                        {selectedLead.phone ||
                          selectedLead.buyer_phone ||
                          "No Phone"}
                      </p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Email Address
                      </label>
                      <p className="text-sm text-gray-900">
                        {selectedLead.email ||
                          selectedLead.buyer_email ||
                          "No Email"}
                      </p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Preferred Contact
                      </label>
                      <p className="text-sm text-gray-900 capitalize">
                        {selectedLead.preferred_contact}
                      </p>
                    </div>
                  </div>
                  {selectedLead.message && (
                    <div className="mt-4">
                      <label className="block text-sm font-medium text-gray-700">
                        Buyer Message / Enquiry
                      </label>
                      <p className="text-sm text-gray-900 bg-white p-3 rounded border">
                        {selectedLead.message}
                      </p>
                    </div>
                  )}
                </div>

                {/* 2️⃣ Property Information */}
                <div className="bg-green-50 p-4 rounded-lg">
                  <h4 className="font-semibold text-gray-900 mb-3">
                    🏠 Property Information
                  </h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Property Title
                      </label>
                      <p className="text-sm text-gray-900">
                        {selectedLead.property_title ||
                          `Property #${selectedLead.property_id}`}
                      </p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Property ID
                      </label>
                      <p className="text-sm text-gray-900">
                        {selectedLead.property_id}
                      </p>
                    </div>
                    {selectedLead.property_type && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700">
                          Property Type
                        </label>
                        <p className="text-sm text-gray-900">
                          {selectedLead.property_type}
                        </p>
                      </div>
                    )}
                    {selectedLead.property_location && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700">
                          Location
                        </label>
                        <p className="text-sm text-gray-900">
                          {selectedLead.property_location}
                        </p>
                      </div>
                    )}
                  </div>
                </div>

                {/* 3️⃣ Agent / Seller Information */}
                <div className="bg-yellow-50 p-4 rounded-lg">
                  <h4 className="font-semibold text-gray-900 mb-3">
                    👨‍💼 Agent / Seller Information
                  </h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Assigned To
                      </label>
                      <p className="text-sm text-gray-900">
                        {selectedLead.agent_name ||
                          selectedLead.seller_name ||
                          "Unassigned"}
                      </p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Role
                      </label>
                      <p className="text-sm text-gray-900">
                        {selectedLead.agent_name
                          ? "Agent"
                          : selectedLead.seller_name
                          ? "Owner"
                          : "N/A"}
                      </p>
                    </div>
                    {(selectedLead.agent_email ||
                      selectedLead.seller_email) && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700">
                          Email
                        </label>
                        <p className="text-sm text-gray-900">
                          {selectedLead.agent_email ||
                            selectedLead.seller_email}
                        </p>
                      </div>
                    )}
                    {(selectedLead.agent_phone ||
                      selectedLead.seller_phone) && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700">
                          Phone
                        </label>
                        <p className="text-sm text-gray-900">
                          {selectedLead.agent_phone ||
                            selectedLead.seller_phone}
                        </p>
                      </div>
                    )}
                    {(selectedLead.agent_id ||
                      selectedLead.seller_id ||
                      selectedLead.owner_id) && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700">
                          User ID
                        </label>
                        <p className="text-sm text-gray-900 font-mono text-xs">
                          {selectedLead.agent_id ||
                            selectedLead.seller_id ||
                            selectedLead.owner_id}
                        </p>
                      </div>
                    )}
                  </div>
                </div>

                {/* 4️⃣ Lead Status & Stage */}
                <div className="bg-purple-50 p-4 rounded-lg">
                  <h4 className="font-semibold text-gray-900 mb-3">
                    📊 Lead Status & Timeline
                  </h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Current Status
                      </label>
                      <span
                        className={`inline-block px-2 py-1 text-xs font-medium rounded-full ${
                          selectedLead.status === "new"
                            ? "bg-blue-100 text-blue-800"
                            : selectedLead.status === "prospect"
                            ? "bg-yellow-100 text-yellow-800"
                            : selectedLead.status === "contacted"
                            ? "bg-green-100 text-green-800"
                            : selectedLead.status === "archived"
                            ? "bg-gray-100 text-gray-800"
                            : "bg-gray-100 text-gray-800"
                        }`}
                      >
                        {selectedLead.status === "new"
                          ? "Lead"
                          : selectedLead.status === "prospect"
                          ? "Prospect"
                          : selectedLead.status === "contacted"
                          ? "Contacted"
                          : selectedLead.status === "archived"
                          ? "Archived"
                          : selectedLead.status}
                      </span>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Created Date
                      </label>
                      <p className="text-sm text-gray-900">
                        {formatDate(selectedLead.created_at)}
                      </p>
                    </div>
                    {selectedLead.prospect_at && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700">
                          Became Prospect
                        </label>
                        <p className="text-sm text-gray-900">
                          {formatDate(selectedLead.prospect_at)}
                        </p>
                      </div>
                    )}
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Last Updated
                      </label>
                      <p className="text-sm text-gray-900">
                        {formatDate(selectedLead.updated_at)}
                      </p>
                    </div>
                  </div>
                </div>

                {/* 6️⃣ Budget & Preferences */}
                <div className="bg-orange-50 p-4 rounded-lg">
                  <h4 className="font-semibold text-gray-900 mb-3">
                    💰 Budget & Preferences
                  </h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Budget
                      </label>
                      <p className="text-sm text-gray-900">
                        {selectedLead.budget || "Not specified"}
                      </p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700">
                        Lead Type
                      </label>
                      <p className="text-sm text-gray-900 capitalize">
                        {selectedLead.lead_type}
                      </p>
                    </div>
                    {selectedLead.score && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700">
                          Lead Score
                        </label>
                        <p className="text-sm text-gray-900">
                          {selectedLead.score}/100
                        </p>
                      </div>
                    )}
                    {selectedLead.ai_category && (
                      <div>
                        <label className="block text-sm font-medium text-gray-700">
                          AI Category
                        </label>
                        <p className="text-sm text-gray-900 capitalize">
                          {selectedLead.ai_category}
                        </p>
                      </div>
                    )}
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
      {selectedBooking && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto shadow-xl">
            <div className="p-6">
              <div className="flex justify-between items-start mb-4">
                <h3 className="text-lg font-medium text-gray-900">
                  Visit Booking Details
                </h3>
                <button
                  onClick={() => setSelectedBooking(null)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <X className="w-5 h-5" />
                </button>
              </div>

              <div className="space-y-6">
                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <h4 className="font-semibold text-gray-900 mb-3">
                      Visitor Information
                    </h4>
                    <div className="space-y-2">
                      <p>
                        <span className="font-medium">Name:</span>{" "}
                        {selectedBooking.full_name}
                      </p>
                      <p>
                        <span className="font-medium">Mobile:</span>{" "}
                        {selectedBooking.mobile_number}
                      </p>
                      <p>
                        <span className="font-medium">Email:</span>{" "}
                        {selectedBooking.email || "Not provided"}
                      </p>
                      <p>
                        <span className="font-medium">Visitors:</span>{" "}
                        {selectedBooking.number_of_visitors}
                      </p>
                    </div>
                  </div>
                  <div>
                    <h4 className="font-semibold text-gray-900 mb-3">
                      Visit Details
                    </h4>
                    <div className="space-y-2">
                      <p>
                        <span className="font-medium">Date:</span>{" "}
                        {new Date(
                          selectedBooking.visit_date
                        ).toLocaleDateString()}
                      </p>
                      <p>
                        <span className="font-medium">Time:</span>{" "}
                        {selectedBooking.visit_start_time}
                      </p>
                      <p>
                        <span className="font-medium">Duration:</span>{" "}
                        {selectedBooking.visit_duration} minutes
                      </p>
                      <p>
                        <span className="font-medium">Purpose:</span>{" "}
                        {selectedBooking.purpose_of_visit}
                      </p>
                    </div>
                  </div>
                </div>

                <div>
                  <h4 className="font-semibold text-gray-900 mb-3">
                    Status Management
                  </h4>
                  <div className="flex gap-2">
                    {[
                      "scheduled",
                      "confirmed",
                      "in-progress",
                      "completed",
                      "cancelled",
                    ].map((status) => (
                      <button
                        key={status}
                        onClick={async () => {
                          try {
                            await updateBookingStatus(
                              selectedBooking.id!,
                              status
                            );
                            setSelectedBooking({
                              ...selectedBooking,
                              status: status as any,
                            });
                            setVisitBookings((prev) =>
                              prev.map((b) =>
                                b.id === selectedBooking.id
                                  ? { ...b, status: status as any }
                                  : b
                              )
                            );
                          } catch (error) {
                            console.error("Error updating status:", error);
                          }
                        }}
                        className={`px-3 py-1 text-xs font-medium rounded-full ${
                          selectedBooking.status === status
                            ? "bg-blue-500 text-white"
                            : "bg-gray-100 text-gray-700 hover:bg-gray-200"
                        }`}
                      >
                        {status}
                      </button>
                    ))}
                  </div>
                </div>

                {selectedBooking.custom_requests && (
                  <div>
                    <h4 className="font-semibold text-gray-900 mb-2">
                      Special Requests
                    </h4>
                    <p className="text-gray-700 bg-gray-50 p-3 rounded">
                      {selectedBooking.custom_requests}
                    </p>
                  </div>
                )}
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Visit Booking Details Modal */}
    </div>
  );
};
