import React, { useState, useEffect } from 'react';
import {
  TrendingUp, IndianRupee, Activity, AlertTriangle, CheckCircle,
  Clock, Shield, Settings, Download, RefreshCw, Filter, Search
} from 'lucide-react';

interface ChannelStats {
  channel: string;
  total_transactions: number;
  successful: number;
  failed: number;
  total_amount_cents: number;
  success_rate: number;
}

interface Transaction {
  id: string;
  user_id: string;
  amount_cents: number;
  channel: string;
  gateway: string;
  status: string;
  created_at: string;
  confirmed_at?: string;
}

export function MultiChannelPaymentAdmin() {
  const [channelStats, setChannelStats] = useState<ChannelStats[]>([]);
  const [recentTransactions, setRecentTransactions] = useState<Transaction[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedChannel, setSelectedChannel] = useState<string>('all');
  const [dateRange, setDateRange] = useState<string>('today');

  useEffect(() => {
    fetchDashboardData();
  }, [selectedChannel, dateRange]);

  const fetchDashboardData = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('token');

      // Fetch channel statistics
      const statsResponse = await fetch(
        `/api/payments/admin/multi-channel/overview?channel=${selectedChannel}&range=${dateRange}`,
        { headers: { 'Authorization': `Bearer ${token}` } }
      );
      const statsData = await statsResponse.json();
      setChannelStats(statsData.channel_stats || []);

      // Fetch recent transactions
      const txnResponse = await fetch(
        `/api/payments/admin/multi-channel/transactions?page_size=20`,
        { headers: { 'Authorization': `Bearer ${token}` } }
      );
      const txnData = await txnResponse.json();
      setRecentTransactions(txnData.transactions || []);
    } catch (error) {
      console.error('Failed to fetch dashboard data:', error);
    } finally {
      setLoading(false);
    }
  };

  const getChannelIcon = (channel: string) => {
    const icons: Record<string, any> = {
      UPI: '📱',
      CARD: '💳',
      WALLET: '👛',
      NETBANKING: '🏦',
      BANK_TRANSFER: '💰'
    };
    return icons[channel] || '💵';
  };

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      confirmed: 'bg-green-100 text-green-700',
      settled: 'bg-blue-100 text-blue-700',
      pending_auth: 'bg-yellow-100 text-yellow-700',
      failed: 'bg-red-100 text-red-700',
      refunded: 'bg-purple-100 text-purple-700'
    };
    return colors[status] || 'bg-gray-100 text-gray-700';
  };

  const calculateTotals = () => {
    const total = channelStats.reduce((acc, stat) => ({
      transactions: acc.transactions + stat.total_transactions,
      amount: acc.amount + stat.total_amount_cents,
      successful: acc.successful + stat.successful,
      failed: acc.failed + stat.failed
    }), { transactions: 0, amount: 0, successful: 0, failed: 0 });

    const successRate = total.transactions > 0
      ? ((total.successful / total.transactions) * 100).toFixed(1)
      : '0.0';

    return { ...total, successRate };
  };

  const totals = calculateTotals();

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold text-gray-900 flex items-center gap-2">
            <Activity className="w-7 h-7 text-blue-600" />
            Multi-Channel Payment Management
          </h2>
          <p className="text-gray-600 mt-1">Monitor and manage all payment channels</p>
        </div>
        <div className="flex gap-2">
          <button className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 flex items-center gap-2">
            <Download className="w-4 h-4" />
            Export
          </button>
          <button
            onClick={fetchDashboardData}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 flex items-center gap-2"
          >
            <RefreshCw className="w-4 h-4" />
            Refresh
          </button>
        </div>
      </div>

      {/* Filters */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-4">
        <div className="flex gap-4">
          <div className="flex-1">
            <label className="block text-sm font-medium text-gray-700 mb-2">Channel</label>
            <select
              value={selectedChannel}
              onChange={(e) => setSelectedChannel(e.target.value)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg"
            >
              <option value="all">All Channels</option>
              <option value="UPI">UPI</option>
              <option value="CARD">Cards</option>
              <option value="WALLET">Wallets</option>
              <option value="NETBANKING">Net Banking</option>
              <option value="BANK_TRANSFER">Bank Transfer</option>
            </select>
          </div>
          <div className="flex-1">
            <label className="block text-sm font-medium text-gray-700 mb-2">Date Range</label>
            <select
              value={dateRange}
              onChange={(e) => setDateRange(e.target.value)}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg"
            >
              <option value="today">Today</option>
              <option value="week">This Week</option>
              <option value="month">This Month</option>
              <option value="quarter">This Quarter</option>
            </select>
          </div>
        </div>
      </div>

      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Total Transactions</p>
              <p className="text-3xl font-bold text-gray-900 mt-2">{totals.transactions}</p>
            </div>
            <TrendingUp className="w-8 h-8 text-blue-600" />
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Total Revenue</p>
              <p className="text-3xl font-bold text-green-600 mt-2">
                ₹{(totals.amount / 100).toLocaleString('en-IN')}
              </p>
            </div>
            <IndianRupee className="w-8 h-8 text-green-600" />
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Success Rate</p>
              <p className="text-3xl font-bold text-blue-600 mt-2">{totals.successRate}%</p>
            </div>
            <CheckCircle className="w-8 h-8 text-blue-600" />
          </div>
        </div>

        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm text-gray-600">Failed Payments</p>
              <p className="text-3xl font-bold text-red-600 mt-2">{totals.failed}</p>
            </div>
            <AlertTriangle className="w-8 h-8 text-red-600" />
          </div>
        </div>
      </div>

      {/* Channel Performance */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-lg font-semibold mb-4">Channel Performance</h3>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50 border-b">
              <tr>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Channel</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Total Txns</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Successful</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Failed</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Success Rate</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Total Amount</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-gray-200">
              {channelStats.map((stat) => (
                <tr key={stat.channel} className="hover:bg-gray-50">
                  <td className="px-4 py-3">
                    <div className="flex items-center gap-2">
                      <span className="text-2xl">{getChannelIcon(stat.channel)}</span>
                      <span className="font-medium text-gray-900">{stat.channel}</span>
                    </div>
                  </td>
                  <td className="px-4 py-3 text-gray-600">{stat.total_transactions}</td>
                  <td className="px-4 py-3 text-green-600 font-medium">{stat.successful}</td>
                  <td className="px-4 py-3 text-red-600 font-medium">{stat.failed}</td>
                  <td className="px-4 py-3">
                    <span className={`px-2 py-1 rounded-full text-xs font-medium ${stat.success_rate >= 95 ? 'bg-green-100 text-green-700' :
                        stat.success_rate >= 85 ? 'bg-yellow-100 text-yellow-700' :
                          'bg-red-100 text-red-700'
                      }`}>
                      {stat.success_rate.toFixed(1)}%
                    </span>
                  </td>
                  <td className="px-4 py-3 font-semibold text-gray-900">
                    ₹{(stat.total_amount_cents / 100).toLocaleString('en-IN')}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Recent Transactions */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-lg font-semibold mb-4">Recent Transactions</h3>
        <div className="space-y-3">
          {recentTransactions.map((txn) => (
            <div key={txn.id} className="flex items-center justify-between p-4 border border-gray-200 rounded-lg hover:bg-gray-50">
              <div className="flex items-center gap-4">
                <div className="text-2xl">{getChannelIcon(txn.channel)}</div>
                <div>
                  <p className="font-medium text-gray-900">₹{(txn.amount_cents / 100).toLocaleString('en-IN')}</p>
                  <p className="text-sm text-gray-500">
                    {txn.channel} • {txn.gateway} • {new Date(txn.created_at).toLocaleString('en-IN')}
                  </p>
                </div>
              </div>
              <span className={`px-3 py-1 rounded-full text-xs font-medium ${getStatusColor(txn.status)}`}>
                {txn.status}
              </span>
            </div>
          ))}
        </div>
      </div>

      {/* Gateway Health */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Shield className="w-5 h-5 text-green-600" />
          Gateway Health Status
        </h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          {['Razorpay', 'Stripe', 'Cashfree'].map((gateway) => (
            <div key={gateway} className="p-4 border border-gray-200 rounded-lg">
              <div className="flex items-center justify-between mb-2">
                <span className="font-medium text-gray-900">{gateway}</span>
                <span className="px-2 py-1 bg-green-100 text-green-700 rounded-full text-xs font-medium">
                  Active
                </span>
              </div>
              <div className="space-y-1 text-sm text-gray-600">
                <p>Uptime: 99.9%</p>
                <p>Avg Response: 1.2s</p>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}
