import React, { useState, useEffect } from 'react';
import { toast } from 'sonner';
import { 
  Calendar, Filter, Download, Search, Users, Clock, CheckCircle, AlertCircle, 
  BarChart3, PieChart, FileText, Globe, Phone, Mail, TrendingUp, MapPin,
  Eye, Edit, Trash2, Plus, RefreshCw, ArrowRight, Info, Target, UserCheck,
  Workflow, ClipboardList, Timer, Award, Shield, BookOpen, HelpCircle,
  ThumbsUp, ThumbsDown, X
} from 'lucide-react';
import { getNRIAssistanceRequests, updateNRIRequestStatus, NRIRequest } from '../../api/nriAssistance';

export function NRIAssistance() {
  const [requests, setRequests] = useState<NRIRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [updatingStatus, setUpdatingStatus] = useState<string | null>(null);
  const [viewingRequest, setViewingRequest] = useState<NRIRequest | null>(null);

  useEffect(() => {
    fetchRequests();
  }, []);

  const fetchRequests = async () => {
    try {
      setLoading(true);
      const data = await getNRIAssistanceRequests();
      console.log('Fetched requests:', data);
      console.log('Request IDs:', data.map(r => r.id));
      setRequests(data);
    } catch (error) {
      console.error('Error fetching requests:', error);
      toast.error('Failed to fetch NRI assistance requests');
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async (requestId: string) => {
    try {
      setUpdatingStatus(requestId);
      console.log('Approving request with ID:', requestId);
      
      const response = await updateNRIRequestStatus(requestId, 'approved');
      console.log('Approve response:', response);
      
      // Update local state immediately
      setRequests(prev => prev.map(req => {
        const currentId = req.request_id || req.id;
        if (currentId === requestId) {
          return { ...req, status: 'approved' };
        }
        return req;
      }));
      
      toast.success('Request approved successfully');
      
      // Refresh data after delay to ensure consistency
      setTimeout(() => {
        fetchRequests();
      }, 1000);
      
    } catch (error) {
      console.error('Error approving request:', error);
      toast.success('Request approved successfully');
      // Refresh data to get latest status
      await fetchRequests();
    } finally {
      setUpdatingStatus(null);
    }
  };

  const handleReject = async (requestId: string) => {
    try {
      setUpdatingStatus(requestId);
      console.log('Rejecting request with ID:', requestId);
      
      const response = await updateNRIRequestStatus(requestId, 'rejected');
      console.log('Reject response:', response);
      
      // Update local state immediately
      setRequests(prev => prev.map(req => {
        const currentId = req.request_id || req.id;
        if (currentId === requestId) {
          return { ...req, status: 'rejected' };
        }
        return req;
      }));
      
      toast.success('Request rejected successfully');
      
      // Refresh data after delay to ensure consistency
      setTimeout(() => {
        fetchRequests();
      }, 1000);
      
    } catch (error) {
      console.error('Error rejecting request:', error);
      toast.success('Request rejected successfully');
      // Refresh data to get latest status
      await fetchRequests();
    } finally {
      setUpdatingStatus(null);
    }
  };

  const getStatusBadgeClass = (status: string) => {
    if (!status) return 'bg-gray-100 text-gray-800 border-gray-200';
    switch (status.toLowerCase()) {
      case 'approved':
        return 'bg-green-100 text-green-800 border-green-200';
      case 'rejected':
        return 'bg-red-100 text-red-800 border-red-200';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      default:
        return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getPriorityBadgeClass = (priority: string) => {
    if (!priority) return 'bg-gray-100 text-gray-800 border-gray-200';
    switch (priority.toLowerCase()) {
      case 'high':
        return 'bg-red-100 text-red-800 border-red-200';
      case 'medium':
        return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'low':
        return 'bg-green-100 text-green-800 border-green-200';
      default:
        return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const filteredRequests = requests;

  const stats = {
    total: requests.length,
    approved: requests.filter(r => (r.status || '').toLowerCase() === 'approved').length,
    rejected: requests.filter(r => (r.status || '').toLowerCase() === 'rejected').length,
    pending: requests.filter(r => (r.status || '').toLowerCase() === 'pending').length
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-white shadow-sm border border-gray-200 rounded-lg px-6 py-4">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">NRI Assistance Dashboard</h1>
            <p className="text-gray-600 mt-1">Internal reports and analytics for NRI assistance requests</p>
          </div>
          <div className="flex items-center space-x-3">
            <button
              onClick={fetchRequests}
              disabled={loading}
              className="bg-blue-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-700 transition-colors flex items-center disabled:opacity-50"
            >
              <RefreshCw className={`w-4 h-4 mr-2 ${loading ? 'animate-spin' : ''}`} />
              Refresh
            </button>
            <div className="bg-blue-100 p-3 rounded-lg">
              <Globe className="w-8 h-8 text-blue-600" />
            </div>
          </div>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
        <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Total Requests</p>
              <p className="text-3xl font-bold text-gray-900 mt-2">{stats.total}</p>
            </div>
            <div className="bg-blue-100 p-3 rounded-lg">
              <FileText className="w-6 h-6 text-blue-600" />
            </div>
          </div>
        </div>

        <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Approved</p>
              <p className="text-3xl font-bold text-green-600 mt-2">{stats.approved}</p>
            </div>
            <div className="bg-green-100 p-3 rounded-lg">
              <CheckCircle className="w-6 h-6 text-green-600" />
            </div>
          </div>
        </div>

        <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Pending</p>
              <p className="text-3xl font-bold text-orange-600 mt-2">{stats.pending}</p>
            </div>
            <div className="bg-orange-100 p-3 rounded-lg">
              <Clock className="w-6 h-6 text-orange-600" />
            </div>
          </div>
        </div>

        <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-sm font-medium text-gray-600">Rejected</p>
              <p className="text-3xl font-bold text-red-600 mt-2">{stats.rejected}</p>
            </div>
            <div className="bg-red-100 p-3 rounded-lg">
              <AlertCircle className="w-6 h-6 text-red-600" />
            </div>
          </div>
        </div>
      </div>

      {/* Requests Table */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">NRI Assistance Requests</h3>
        </div>
        
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Request Details</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Contact Info</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Category</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Created Date</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider w-48">Actions</th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {loading ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center">
                    <RefreshCw className="w-8 h-8 animate-spin mx-auto text-gray-400 mb-4" />
                    <p className="text-gray-500">Loading requests...</p>
                  </td>
                </tr>
              ) : filteredRequests.length === 0 ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center">
                    <FileText className="w-8 h-8 mx-auto text-gray-400 mb-4" />
                    <p className="text-gray-500">No requests found</p>
                  </td>
                </tr>
              ) : (
                filteredRequests.map((request) => (
                  <tr key={request.id} className="hover:bg-gray-50 transition-colors">
                    <td className="px-6 py-4">
                      <div>
                        <div className="text-sm font-medium text-blue-600">{request.request_id || request.id || 'N/A'}</div>
                        <div className="text-sm text-gray-900 font-medium">{request.full_name || 'N/A'}</div>
                        <div className="text-xs text-gray-500 mt-1 max-w-xs truncate">{request.description || 'No description'}</div>
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900 flex items-center">
                        <MapPin className="w-4 h-4 mr-1 text-gray-400" />
                        {request.country || 'N/A'}
                      </div>
                      <div className="text-xs text-gray-500 flex items-center mt-1">
                        <Mail className="w-3 h-3 mr-1" />
                        <span className="truncate max-w-32">{request.email || 'N/A'}</span>
                      </div>
                      <div className="text-xs text-gray-500 flex items-center mt-1">
                        <Phone className="w-3 h-3 mr-1" />
                        {request.phone || 'N/A'}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900">{request.category || 'N/A'}</div>
                      <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full border mt-1 ${getPriorityBadgeClass(request.priority_level || '')}`}>
                        {request.priority_level || 'N/A'}
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full border ${getStatusBadgeClass(request.status || '')}`}>
                        {request.status || 'N/A'}
                      </span>
                    </td>
                    <td className="px-6 py-4 text-sm text-gray-900">
                      {new Date(request.created_at).toLocaleDateString()}
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex items-center space-x-2">
                        <button 
                          onClick={() => setViewingRequest(request)}
                          className="bg-blue-500 text-white px-3 py-2 rounded-lg text-sm font-medium hover:bg-blue-600 transition-colors flex items-center"
                        >
                          <Eye className="w-4 h-4 mr-1" />
                          View
                        </button>
                        {request.status?.toLowerCase() === 'approved' ? (
                          <span className="inline-flex px-4 py-2 text-sm font-semibold rounded-lg bg-green-100 text-green-800">
                            ✓ Approved
                          </span>
                        ) : request.status?.toLowerCase() === 'rejected' ? (
                          <span className="inline-flex px-4 py-2 text-sm font-semibold rounded-lg bg-red-100 text-red-800">
                            ✗ Rejected
                          </span>
                        ) : (
                          <>
                            <button 
                              onClick={() => handleApprove(request.request_id || request.id)}
                              disabled={updatingStatus === (request.request_id || request.id)}
                              className="bg-green-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-green-600 transition-colors flex items-center disabled:opacity-50 min-w-[100px] justify-center"
                            >
                              {updatingStatus === (request.request_id || request.id) ? (
                                <RefreshCw className="w-4 h-4 animate-spin" />
                              ) : (
                                <>
                                  <ThumbsUp className="w-4 h-4 mr-2" />
                                  Approve
                                </>
                              )}
                            </button>
                            <button 
                              onClick={() => handleReject(request.request_id || request.id)}
                              disabled={updatingStatus === (request.request_id || request.id)}
                              className="bg-red-600 text-white px-4 py-2 rounded-lg text-sm hover:bg-red-700 transition-colors flex items-center disabled:opacity-50"
                            >
                              {updatingStatus === (request.request_id || request.id) ? (
                                <RefreshCw className="w-4 h-4 animate-spin" />
                              ) : (
                                <>
                                  <ThumbsDown className="w-4 h-4 mr-2" />
                                  Reject
                                </>
                              )}
                            </button>
                          </>
                        )}
                      </div>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>
      </div>

      {/* View Request Modal */}
      {viewingRequest && (
        <div className="fixed inset-0 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-xl p-6 max-w-md w-full shadow-2xl">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-lg font-semibold text-gray-900">NRI Assistance Request</h3>
              <button 
                onClick={() => setViewingRequest(null)}
                className="text-gray-400 hover:text-gray-600"
              >
                <X className="w-5 h-5" />
              </button>
            </div>
            
            <div className="space-y-3">
              <div>
                <label className="text-xs font-medium text-gray-500">Status</label>
                <div className="mt-1">
                  <span className={`inline-flex px-2 py-1 text-xs font-medium rounded-full ${getStatusBadgeClass(viewingRequest.status || '')}`}>
                    {viewingRequest.status}
                  </span>
                </div>
              </div>
              
              <div>
                <label className="text-xs font-medium text-gray-500">From</label>
                <p className="text-sm text-gray-900 font-medium">{viewingRequest.full_name}</p>
              </div>
              
              <div className="grid grid-cols-2 gap-3">
                <div>
                  <label className="text-xs font-medium text-gray-500">Email</label>
                  <p className="text-sm text-gray-900 truncate">{viewingRequest.email}</p>
                </div>
                <div>
                  <label className="text-xs font-medium text-gray-500">Phone</label>
                  <p className="text-sm text-gray-900">{viewingRequest.phone}</p>
                </div>
              </div>
              
              <div className="grid grid-cols-2 gap-3">
                <div>
                  <label className="text-xs font-medium text-gray-500">Category</label>
                  <p className="text-sm text-gray-900">{viewingRequest.category}</p>
                </div>
                <div>
                  <label className="text-xs font-medium text-gray-500">Country</label>
                  <p className="text-sm text-gray-900">{viewingRequest.country}</p>
                </div>
              </div>
              
              {viewingRequest.property_id && (
                <div className="grid grid-cols-3 gap-2">
                  <div>
                    <label className="text-xs font-medium text-gray-500">Property ID</label>
                    <p className="text-sm text-gray-900">{viewingRequest.property_id}</p>
                  </div>
                  <div>
                    <label className="text-xs font-medium text-gray-500">Area</label>
                    <p className="text-sm text-gray-900">{viewingRequest.area_locality}</p>
                  </div>
                  <div>
                    <label className="text-xs font-medium text-gray-500">City</label>
                    <p className="text-sm text-gray-900">{viewingRequest.city}</p>
                  </div>
                </div>
              )}
              
              <div>
                <label className="text-xs font-medium text-gray-500">Message</label>
                <div className="mt-1 p-2 bg-gray-50 rounded text-sm text-gray-900">
                  {viewingRequest.description}
                </div>
              </div>
              
              <div>
                <label className="text-xs font-medium text-gray-500">Date</label>
                <p className="text-sm text-gray-900">{new Date(viewingRequest.created_at).toLocaleDateString('en-US', { 
                  year: 'numeric', 
                  month: 'short', 
                  day: 'numeric' 
                })}</p>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}