import React, { useState, useEffect } from "react";
import { Download, FileText, Calendar, Loader2, Users, Building, Gavel, MessageSquare, Shield, TrendingUp, BarChart3, BookOpen, ShieldCheck, Globe, Trash2 } from "lucide-react";
import { toast } from 'sonner';
import { propertyGenerateAdminAPI, AdminPropertyDocument } from '../../api/propertyGenerate';

interface TimePeriodModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSelect: (period: string) => void;
}

function TimePeriodModal({ isOpen, onClose, onSelect }: TimePeriodModalProps) {
  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 flex items-center justify-center z-50">
      <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4 shadow-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Select Time Period</h3>
        <p className="text-gray-600 mb-6">Choose the time period for your admin property activity report:</p>

        <div className="space-y-3">
          <button
            onClick={() => onSelect('30_days')}
            className="w-full text-left p-4 border border-gray-200 rounded-lg hover:bg-blue-50 hover:border-blue-300 transition-colors"
          >
            <div className="font-medium text-gray-900">Last 30 Days</div>
            <div className="text-sm text-gray-600">Recent admin activity and current data</div>
          </button>

          <button
            onClick={() => onSelect('60_days')}
            className="w-full text-left p-4 border border-gray-200 rounded-lg hover:bg-blue-50 hover:border-blue-300 transition-colors"
          >
            <div className="font-medium text-gray-900">Last 60 Days</div>
            <div className="text-sm text-gray-600">Extended admin activity overview</div>
          </button>

          <button
            onClick={() => onSelect('1_year')}
            className="w-full text-left p-4 border border-gray-200 rounded-lg hover:bg-blue-50 hover:border-blue-300 transition-colors"
          >
            <div className="font-medium text-gray-900">Last Year</div>
            <div className="text-sm text-gray-600">Comprehensive annual admin activity report</div>
          </button>
        </div>

        <div className="flex justify-end mt-6">
          <button
            onClick={onClose}
            className="px-4 py-2 text-gray-600 hover:text-gray-800 transition-colors"
          >
            Cancel
          </button>
        </div>
      </div>
    </div>
  );
}

export function PropertyGeneratorAdmin() {
  console.log('PropertyGeneratorAdmin component rendered');
  const [documents, setDocuments] = useState<AdminPropertyDocument[]>([]);
  const [loading, setLoading] = useState(false);
  const [generating, setGenerating] = useState(false);
  const [deletingId, setDeletingId] = useState<string | null>(null);
  const [showTimeModal, setShowTimeModal] = useState(false);

  const fetchDocuments = async () => {
    try {
      setLoading(true);
      const response = await propertyGenerateAdminAPI.getAdminDocuments();
      setDocuments(response);
    } catch (error) {
      console.error('Error fetching admin documents:', error);
      toast.error('Failed to load admin documents');
    } finally {
      setLoading(false);
    }
  };

  const generateDocument = () => {
    setShowTimeModal(true);
  };

  const handleTimePeriodSelect = async (timePeriod: string) => {
    setShowTimeModal(false);
    try {
      setGenerating(true);
      const response = await propertyGenerateAdminAPI.generateAdminDocument(timePeriod);
      toast.success('Admin property document generated successfully!');
      fetchDocuments(); // Refresh the list
    } catch (error) {
      console.error('Error generating admin document:', error);
      toast.error('Failed to generate admin document');
    } finally {
      setGenerating(false);
    }
  };

  const deleteDocument = async (documentId: string) => {
    console.log('Delete button clicked for admin document:', documentId);

    const confirmed = window.confirm('Are you sure you want to delete this admin report? This action cannot be undone.');
    if (!confirmed) {
      console.log('User cancelled delete operation');
      return;
    }

    console.log('User confirmed delete, proceeding...');

    // Set deleting state to show loading
    setDeletingId(documentId);

    try {
      console.log('Calling API to delete admin document:', documentId);

      // Call the delete API
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/property-generate-admin/${documentId}`, {
        method: 'DELETE',
      });

      if (!response.ok) {
        throw new Error('Failed to delete document');
      }

      console.log('Admin document deleted from backend successfully');

      // Remove from local state
      setDocuments(prevDocuments => prevDocuments.filter(doc => doc.id !== documentId));
      console.log('Admin document removed from local state');

      toast.success('Admin document deleted successfully!');
    } catch (error) {
      console.error('Error deleting admin document:', error);
      const errorMessage = error instanceof Error ? error.message : 'Unknown error';
      console.error('Error details:', errorMessage);
      toast.error(`Failed to delete admin document: ${errorMessage}`);
    } finally {
      // Clear deleting state
      setDeletingId(null);
    }
  };

  const downloadDocument = async (documentId: string) => {
    try {
      // For admin documents, the API returns a direct PDF blob
      // Admin endpoints are excluded from authentication, so no Authorization header needed
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/property-generate-admin/${documentId}/download`, {
        method: 'GET',
      });

      if (!response.ok) {
        throw new Error('Failed to download document');
      }

      // Get the PDF blob directly from the response
      const blob = await response.blob();

      // Create download link
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;

      // Extract filename from Content-Disposition header or use default
      const contentDisposition = response.headers.get('Content-Disposition');
      let filename = `admin_property_report_${new Date().toISOString().split('T')[0]}.pdf`;
      if (contentDisposition) {
        const filenameMatch = contentDisposition.match(/filename="(.+)"/);
        if (filenameMatch) {
          filename = filenameMatch[1];
        }
      }

      a.download = filename;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);

      toast.success('Admin PDF downloaded successfully!');
    } catch (error) {
      console.error('Error downloading admin document:', error);
      toast.error('Failed to download admin document');
    }
  };

  useEffect(() => {
    fetchDocuments().catch((error) => {
      console.error('Failed to fetch documents on mount:', error);
      // Don't show error toast on mount, just log it
    });
  }, []);

  return (
    <div className="space-y-6">
      <TimePeriodModal
        isOpen={showTimeModal}
        onClose={() => setShowTimeModal(false)}
        onSelect={handleTimePeriodSelect}
      />

      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Property Generator</h2>
          <p className="text-gray-600">Generate comprehensive reports about all admin dashboard features</p>
        </div>
      </div>

      {/* Single Blue Generate Button */}
      <div className="flex justify-center py-8">
        <button
          onClick={generateDocument}
          disabled={generating}
          style={{
            backgroundColor: '#1e40af',
            borderColor: '#1d4ed8',
            color: 'white',
            padding: '32px 80px',
            borderRadius: '12px',
            fontSize: '30px',
            fontWeight: 'bold',
            borderWidth: '4px',
            boxShadow: '0 25px 50px -12px rgba(0, 0, 0, 0.25)',
            transition: 'all 0.3s ease',
            display: 'flex',
            alignItems: 'center',
            gap: '12px'
          }}
          onMouseEnter={(e) => {
            e.currentTarget.style.backgroundColor = '#1d4ed8';
            e.currentTarget.style.transform = 'scale(1.05)';
          }}
          onMouseLeave={(e) => {
            e.currentTarget.style.backgroundColor = '#1e40af';
            e.currentTarget.style.transform = 'scale(1)';
          }}
        >
          {generating ? (
            <Loader2 className="w-10 h-10 animate-spin" />
          ) : (
            <FileText className="w-10 h-10" />
          )}
          {generating ? 'Generating Report...' : 'Generate Report'}
        </button>
      </div>

      {/* Features Overview */}
      <div className="bg-white rounded-lg shadow-sm p-6 border border-gray-200">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Report Includes</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          <div className="flex items-center gap-3 p-3 bg-blue-50 rounded-lg">
            <Users className="w-5 h-5 text-blue-600" />
            <span className="text-sm font-medium">User Management</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-green-50 rounded-lg">
            <Building className="w-5 h-5 text-green-600" />
            <span className="text-sm font-medium">Property Management</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-purple-50 rounded-lg">
            <Gavel className="w-5 h-5 text-purple-600" />
            <span className="text-sm font-medium">Auction Management</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-orange-50 rounded-lg">
            <MessageSquare className="w-5 h-5 text-orange-600" />
            <span className="text-sm font-medium">Inquiry Management</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-red-50 rounded-lg">
            <Calendar className="w-5 h-5 text-red-600" />
            <span className="text-sm font-medium">Visit Schedules</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-indigo-50 rounded-lg">
            <FileText className="w-5 h-5 text-indigo-600" />
            <span className="text-sm font-medium">Document Verification</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-yellow-50 rounded-lg">
            <Shield className="w-5 h-5 text-yellow-600" />
            <span className="text-sm font-medium">Financials & Payments</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-pink-50 rounded-lg">
            <TrendingUp className="w-5 h-5 text-pink-600" />
            <span className="text-sm font-medium">Campaigns & Promotions</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-teal-50 rounded-lg">
            <BarChart3 className="w-5 h-5 text-teal-600" />
            <span className="text-sm font-medium">Reports & Analytics</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-cyan-50 rounded-lg">
            <BookOpen className="w-5 h-5 text-cyan-600" />
            <span className="text-sm font-medium">Content & CMS</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-gray-50 rounded-lg">
            <ShieldCheck className="w-5 h-5 text-gray-600" />
            <span className="text-sm font-medium">Security & Audit</span>
          </div>
          <div className="flex items-center gap-3 p-3 bg-lime-50 rounded-lg">
            <Globe className="w-5 h-5 text-lime-600" />
            <span className="text-sm font-medium">Property Trends</span>
          </div>
        </div>
      </div>

      {/* Generated Documents */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Generated Reports</h3>
          <p className="text-gray-600 text-sm mt-1">Download previously generated admin property reports</p>
        </div>

        <div className="p-6">
          {loading ? (
            <div className="text-center py-8">
              <Loader2 className="w-8 h-8 animate-spin mx-auto mb-4 text-blue-600" />
              <p className="text-gray-600">Loading reports...</p>
            </div>
          ) : documents.length > 0 ? (
            <div className="space-y-4">
              {documents.map((doc) => (
                <div key={doc.id} className="flex items-center justify-between p-4 border border-gray-200 rounded-lg">
                  <div className="flex items-center gap-4">
                    <div className="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                      <FileText className="w-5 h-5 text-blue-600" />
                    </div>
                    <div>
                      <h4 className="font-medium text-gray-900">
                        Admin Property Report - {new Date(doc.generated_at).toLocaleDateString()}
                      </h4>
                      <p className="text-sm text-gray-600">
                        Generated on {new Date(doc.generated_at).toLocaleString()}
                      </p>
                    </div>
                  </div>
                  <div className="flex items-center gap-2">
                    <button
                      onClick={() => downloadDocument(doc.id)}
                      className="flex items-center gap-2 px-4 py-2 bg-blue-100 text-blue-700 rounded-lg hover:bg-blue-200 transition-colors"
                    >
                      <Download className="w-4 h-4" />
                      Download
                    </button>
                    <button
                      onClick={async (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        console.log('Delete button clicked for admin document:', doc.id);
                        try {
                          await deleteDocument(doc.id);
                        } catch (error) {
                          console.error('Delete button error:', error);
                        }
                      }}
                      className={`flex items-center gap-2 px-4 py-2 rounded-lg transition-colors ${
                        deletingId === doc.id
                          ? 'bg-gray-100 text-gray-500 cursor-not-allowed'
                          : 'bg-red-100 text-red-700 hover:bg-red-200'
                      }`}
                      type="button"
                      disabled={deletingId === doc.id}
                    >
                      {deletingId === doc.id ? (
                        <Loader2 className="w-4 h-4 animate-spin" />
                      ) : (
                        <Trash2 className="w-4 h-4" />
                      )}
                      {deletingId === doc.id ? 'Deleting...' : 'Delete'}
                    </button>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center py-8">
              <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <h4 className="text-lg font-medium text-gray-900 mb-2">No reports generated yet</h4>
              <p className="text-gray-600 mb-4">Click "Generate Report" to create your first admin property activity report.</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}