import React from 'react';
import { Badge } from '../ui/badge';
import { Button } from '../ui/button';
import { Eye } from 'lucide-react';

interface PropertyLegalTabProps {
  legalHealth: any;
  legalDocuments: any[];
  loadingLegal: boolean;
}

export function PropertyLegalTab({ legalHealth, legalDocuments, loadingLegal }: PropertyLegalTabProps) {
  return (
    <div className="space-y-6">
      {/* Legal Disclaimer */}
      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <p className="text-sm text-yellow-800">
          ⚠️ <strong>Disclaimer:</strong> Legal documents are uploaded by property owners or agents for reference only. 
          The platform does not verify or certify legal ownership or title.
        </p>
      </div>
      
      {loadingLegal ? (
        <div className="flex justify-center py-8">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      ) : (
        <>
          {/* Legal Health Summary */}
          {legalHealth && (
            <div className="bg-gray-50 rounded-lg p-4">
              <h3 className="text-lg font-medium mb-3">Legal Health Summary</h3>
              <div className="grid grid-cols-4 gap-4">
                <div>
                  <p className="text-xs text-gray-600 mb-1">Legal Status</p>
                  <Badge className={`${
                    legalHealth.legal_status === 'COMPLETE' ? 'bg-green-100 text-green-800' :
                    legalHealth.legal_status === 'INCOMPLETE' ? 'bg-yellow-100 text-yellow-800' :
                    'bg-gray-100 text-gray-800'
                  }`}>
                    {legalHealth.legal_status}
                  </Badge>
                </div>
                <div>
                  <p className="text-xs text-gray-600 mb-1">Documents</p>
                  <p className="text-sm font-medium">{legalHealth.uploaded_documents} / {legalHealth.required_documents}</p>
                </div>
                <div>
                  <p className="text-xs text-gray-600 mb-1">Completion</p>
                  <p className="text-sm font-medium">{legalHealth.completion_percentage}%</p>
                </div>
                <div>
                  <p className="text-xs text-gray-600 mb-1">Risk Level</p>
                  <Badge className={`${
                    legalHealth.risk_level === 'LOW' ? 'bg-green-100 text-green-800' :
                    legalHealth.risk_level === 'MEDIUM' ? 'bg-yellow-100 text-yellow-800' :
                    'bg-red-100 text-red-800'
                  }`}>
                    {legalHealth.risk_level}
                  </Badge>
                </div>
              </div>
              {legalHealth.missing_documents?.length > 0 && (
                <div className="mt-4">
                  <p className="text-xs text-gray-600 mb-2">Missing Documents:</p>
                  <div className="flex flex-wrap gap-2">
                    {legalHealth.missing_documents.map((doc: string) => (
                      <Badge key={doc} className="bg-red-50 text-red-700 text-xs">{doc}</Badge>
                    ))}
                  </div>
                </div>
              )}
            </div>
          )}
          
          {/* Document List */}
          <div>
            <h3 className="text-lg font-medium mb-3">Uploaded Documents</h3>
            {legalDocuments.length === 0 ? (
              <p className="text-sm text-gray-500 text-center py-8">No documents uploaded yet</p>
            ) : (
              <div className="border rounded-lg overflow-hidden">
                <table className="w-full text-sm">
                  <thead className="bg-gray-50 border-b">
                    <tr>
                      <th className="px-4 py-2 text-left font-medium text-gray-600">Document Type</th>
                      <th className="px-4 py-2 text-left font-medium text-gray-600">Uploaded By</th>
                      <th className="px-4 py-2 text-left font-medium text-gray-600">Date</th>
                      <th className="px-4 py-2 text-left font-medium text-gray-600">Action</th>
                    </tr>
                  </thead>
                  <tbody className="divide-y">
                    {legalDocuments.map((doc: any) => (
                      <tr key={doc.id} className="hover:bg-gray-50">
                        <td className="px-4 py-3">{doc.display_name}</td>
                        <td className="px-4 py-3 text-gray-600">{doc.created_by || 'Seller'}</td>
                        <td className="px-4 py-3 text-gray-600">
                          {doc.created_at ? new Date(doc.created_at).toLocaleDateString() : 'N/A'}
                        </td>
                        <td className="px-4 py-3">
                          <Button
                            variant="ghost"
                            size="sm"
                            className="text-blue-600 hover:bg-blue-50 text-xs"
                            onClick={() => window.open(`http://localhost:8090/api/property-documents/${doc.id}/download`, '_blank')}
                          >
                            <Eye className="w-3 h-3 mr-1" /> View
                          </Button>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}
          </div>
        </>
      )}
    </div>
  );
}
