import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../ui/card";
import { Button } from "../ui/button";
import { Badge } from "../ui/badge";
import { Input } from "../ui/input";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../ui/tabs";
import { 
  Search, Filter, Download, Plus, Eye, Edit, Ban, Check, X,
  ChevronLeft, ChevronRight, Home, Building, MapPin, Clock, CheckCircle, XCircle
} from "lucide-react";
import { getAdminPropertiesList, AdminPropertyResponse, AdminPaginatedPropertiesResponse, getAdminPropertyStats } from "../../api/admin";
import { getPendingProperties, approveProperty, rejectProperty, PendingProperty, PendingPropertiesResponse } from "../../api/adminProperties";
import { toast } from 'sonner';
import { PropertyLegalTab } from './PropertyLegalTab';


interface PropertyManagementProps {
  initialTab?: string;
}

export function PropertyManagement({ initialTab }: PropertyManagementProps) {
  const [activeTab, setActiveTab] = useState(initialTab || 'all');
  const [propertiesData, setPropertiesData] = useState<AdminPaginatedPropertiesResponse | null>(null);
  const [pendingData, setPendingData] = useState<PendingPropertiesResponse | null>(null);
  const [loading, setLoading] = useState(true);
  const [propertyStats, setPropertyStats] = useState<any>(null);
  const [searchQuery, setSearchQuery] = useState('');
  const [listingTypeFilter, setListingTypeFilter] = useState('');
  const [propertyTypeFilter, setPropertyTypeFilter] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(20);
  const [actionLoading, setActionLoading] = useState<string | null>(null);
  const [showApproveModal, setShowApproveModal] = useState<string | null>(null);
  const [showRejectModal, setShowRejectModal] = useState<string | null>(null);
  const [showDetailsModal, setShowDetailsModal] = useState<AdminPropertyResponse | null>(null);
  const [detailsTab, setDetailsTab] = useState<'overview' | 'legal'>('overview');
  const [legalHealth, setLegalHealth] = useState<any>(null);
  const [legalDocuments, setLegalDocuments] = useState<any[]>([]);
  const [loadingLegal, setLoadingLegal] = useState(false);
  const [approveMessage, setApproveMessage] = useState('');
  const [rejectReason, setRejectReason] = useState('');

  // Test function to check admin email detection
  const testAdminEmail = () => {
    console.log('=== ADMIN EMAIL TEST ===');
    console.log('localStorage keys:', Object.keys(localStorage));
    console.log('login_email:', localStorage.getItem('login_email'));
    console.log('adminEmail:', localStorage.getItem('adminEmail'));
    console.log('user_email:', localStorage.getItem('user_email'));
    
    const user = localStorage.getItem('user');
    if (user) {
      try {
        const parsedUser = JSON.parse(user);
        console.log('Parsed user object:', parsedUser);
        console.log('User email from user object:', parsedUser.email);
      } catch (e) {
        console.log('Failed to parse user object');
      }
    }
    console.log('=== END ADMIN EMAIL TEST ===');
  };

  // Run test on component mount
  useEffect(() => {
    testAdminEmail();
  }, []);

  const fetchProperties = async () => {
    try {
      setLoading(true);
      const params = {
        page: currentPage,
        limit: pageSize,
        ...(searchQuery && { search: searchQuery }),
        ...(listingTypeFilter && { listing_type_filter: listingTypeFilter }),
        ...(propertyTypeFilter && { property_type_filter: propertyTypeFilter }),
        ...(statusFilter && { status_filter: statusFilter })
      };
      
      const [data, stats] = await Promise.all([
        getAdminPropertiesList(params),
        getAdminPropertyStats()
      ]);
      setPropertiesData(data);
      setPropertyStats(stats);
    } catch (error) {
      console.error('Error fetching properties:', error);
      toast.error('Failed to fetch properties');
    } finally {
      setLoading(false);
    }
  };

  const fetchPendingProperties = async () => {
    try {
      setLoading(true);
      const data = await getPendingProperties({ page: currentPage, limit: pageSize });
      setPendingData(data);
    } catch (error) {
      console.error('Error fetching pending properties:', error);
      toast.error('Failed to fetch pending properties');
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async (propertyId: string) => {
    try {
      setActionLoading(propertyId);
      const adminEmail = localStorage.getItem('login_email') || localStorage.getItem('adminEmail') || 'admin@nal.com';
      console.log('=== APPROVAL DEBUG ===');
      console.log('Approving property:', propertyId);
      console.log('Admin email for approval:', adminEmail);
      console.log('Approval data:', { approved_by: adminEmail });
      
      const response = await approveProperty(propertyId, { approved_by: adminEmail });
      console.log('Approval response:', response);
      console.log('=== APPROVAL DEBUG END ===');
      
      toast.success('Property has been approved successfully');
      setShowApproveModal(null);
      fetchPendingProperties();
    } catch (error) {
      console.error('Approval error:', error);
      toast.error('Failed to approve property');
    } finally {
      setActionLoading(null);
    }
  };

  const fetchLegalData = async (propertyId: string) => {
    setLoadingLegal(true);
    try {
      const API_BASE = 'http://localhost:8090/api';
      
      const [healthRes, docsRes] = await Promise.all([
        fetch(`${API_BASE}/legal-health/property/${propertyId}/legal-health`),
        fetch(`${API_BASE}/legal-health/property/${propertyId}/documents`)
      ]);
      
      if (healthRes.ok) setLegalHealth(await healthRes.json());
      if (docsRes.ok) {
        const data = await docsRes.json();
        setLegalDocuments(data.documents || []);
      }
    } catch (error) {
      console.error('Error fetching legal data:', error);
    } finally {
      setLoadingLegal(false);
    }
  };

  const handleReject = async (propertyId: string) => {
    if (!rejectReason.trim()) {
      toast.error('Please provide a rejection reason');
      return;
    }
    try {
      setActionLoading(propertyId);
      const adminEmail = localStorage.getItem('login_email') || localStorage.getItem('adminEmail') || 'admin@nal.com';
      console.log('=== REJECTION DEBUG ===');
      console.log('Rejecting property:', propertyId);
      console.log('Admin email for rejection:', adminEmail);
      console.log('Rejection data:', { rejection_reason: rejectReason, rejected_by: adminEmail });
      
      const response = await rejectProperty(propertyId, { rejection_reason: rejectReason, rejected_by: adminEmail });
      console.log('Rejection response:', response);
      console.log('=== REJECTION DEBUG END ===');
      
      toast.warning('Property has been rejected successfully');
      setShowRejectModal(null);
      setRejectReason('');
      fetchPendingProperties();
    } catch (error) {
      console.error('Rejection error:', error);
      toast.error('Failed to reject property');
    } finally {
      setActionLoading(null);
    }
  };

  useEffect(() => {
    // Set initial tab if provided
    if (initialTab) {
      setActiveTab(initialTab);
    }
  }, [initialTab]);

  useEffect(() => {
    // Always fetch pending count for badge display
    fetchPendingProperties();
    
    if (activeTab === 'all' || activeTab === 'approved') {
      fetchProperties();
    }
  }, [currentPage, pageSize, activeTab]);

  // Fetch pending count on component mount
  useEffect(() => {
    fetchPendingProperties();
  }, []);

  const handleTabChange = (tab: string) => {
    setActiveTab(tab);
    setCurrentPage(1);
    setSearchQuery('');
    setListingTypeFilter('');
    setPropertyTypeFilter('');
    setStatusFilter('');
  };

  const handleSearch = () => {
    setCurrentPage(1);
    fetchProperties();
  };

  const handleFilterChange = () => {
    setCurrentPage(1);
    fetchProperties();
  };

  const clearFilters = () => {
    setSearchQuery('');
    setListingTypeFilter('');
    setPropertyTypeFilter('');
    setStatusFilter('');
    setCurrentPage(1);
    fetchProperties();
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  const getStatusBadge = (status: string) => {
    const colors = {
      active: 'bg-green-100 text-green-800',
      inactive: 'bg-gray-100 text-gray-800',
      sold: 'bg-purple-100 text-purple-800',
      rented: 'bg-blue-100 text-blue-800',
      under_review: 'bg-yellow-100 text-yellow-800'
    };
    return (
      <Badge className={`${colors[status as keyof typeof colors] || colors.active} text-xs`}>
        {status.replace('_', ' ')}
      </Badge>
    );
  };

  const getListingTypeBadge = (type: string) => {
    const colors = {
      sell: 'bg-green-100 text-green-800',
      rent: 'bg-blue-100 text-blue-800',
      'urgent-sale': 'bg-red-100 text-red-800',
      bidding: 'bg-purple-100 text-purple-800',
      lease: 'bg-orange-100 text-orange-800',
      'pg-coliving': 'bg-teal-100 text-teal-800'
    };
    return (
      <Badge className={`${colors[type as keyof typeof colors] || colors.sell} text-xs`}>
        {type.replace('-', ' ')}
      </Badge>
    );
  };

  const formatPrice = (salePrice: number, monthlyRent: number, listingType: string) => {
    const formatAmount = (amount: number) => {
      if (amount >= 10000000) {
        return `₹${(amount / 10000000).toFixed(2)} Cr`;
      } else if (amount >= 100000) {
        return `₹${(amount / 100000).toFixed(2)} L`;
      } else {
        return `₹${amount.toLocaleString()}`;
      }
    };

    if (listingType === 'rent' && monthlyRent) {
      return `${formatAmount(monthlyRent)}/month`;
    }
    if (salePrice) {
      return formatAmount(salePrice);
    }
    return 'Price not set';
  };

  if (loading && !propertiesData) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  const renderPendingTable = () => (
    <div className="overflow-x-auto">
      <table className="w-full">
        <thead className="bg-gray-50 border-b">
          <tr>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Property</th>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Type</th>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Location</th>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Price</th>
            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Actions</th>
          </tr>
        </thead>
        <tbody className="bg-white divide-y divide-gray-200">
          {loading ? (
            Array.from({ length: 5 }).map((_, i) => (
              <tr key={i} className="animate-pulse">
                <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-32"></div></td>
                <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-20"></div></td>
                <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-24"></div></td>
                <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-20"></div></td>
                <td className="px-4 py-3"><div className="h-8 bg-gray-200 rounded w-32"></div></td>
              </tr>
            ))
          ) : pendingData?.properties.map((property) => (
            <tr key={property.id} className="hover:bg-gray-50">
              <td className="px-4 py-3">
                <div className="flex items-center space-x-3">
                  <div className="w-7 h-7 bg-yellow-600 rounded-full flex items-center justify-center text-white text-xs font-medium">
                    <Clock className="w-3 h-3" />
                  </div>
                  <div>
                    <button
                      onClick={() => window.open(`/property/${property.id}`, '_blank')}
                      className="text-sm font-medium text-gray-900 hover:text-blue-600 transition-colors text-left"
                    >
                      {property.title}
                    </button>
                    <div className="text-xs text-gray-500">ID: {String(property.id).slice(0, 8)}</div>
                  </div>
                </div>
              </td>
              <td className="px-4 py-3">
                <div className="space-y-1">
                  {getListingTypeBadge(property.listing_type)}
                  <div className="text-xs text-gray-500">{property.property_type}</div>
                </div>
              </td>
              <td className="px-4 py-3">
                <div className="text-sm text-gray-900">{property.city}</div>
              </td>
              <td className="px-4 py-3">
                <div className="text-sm font-medium text-gray-900">
                  {formatPrice(property.sale_price, property.monthly_rent, property.listing_type)}
                </div>
              </td>
              <td className="px-4 py-3">
                <div className="flex items-center space-x-2">
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => window.open(`/property/${property.id}`, '_blank')}
                    className="text-white bg-green-500 hover:bg-green-600 hover:!text-white px-2 py-1 text-xs transition-colors !important"
                    title="View Property"
                  >
                    View
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => setShowApproveModal(property.id)}
                    className="text-white bg-blue-500 hover:bg-blue-600 px-2 py-1 text-xs transition-colors !important"
                    title="Accept Property"
                  >
                    Accept
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => setShowRejectModal(property.id)}
                    className="text-white bg-red-500 hover:bg-red-600 px-2 py-1 text-xs transition-colors !important"
                    title="Reject Property"
                  >
                    Reject
                  </Button>
                </div>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );

  return (
    <div className="p-6 space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900">Property Management</h1>
          <p className="text-sm text-gray-600 mt-1">Manage and monitor platform properties</p>
        </div>
      </div>

      {/* Enhanced Tabs */}
      <div className="bg-white rounded-xl border border-gray-200 p-1 shadow-sm">
        <div className="grid w-full grid-cols-2 gap-1">
          {['all', 'pending'].map((tab) => {
            const isActive = activeTab === tab;
            const icons = {
              all: <Home className="w-4 h-4" />,
              pending: <Clock className="w-4 h-4" />
            };
            const labels = {
              all: 'All Properties',
              pending: 'Pending Approval'
            };
            
            return (
              <button
                key={tab}
                onClick={() => handleTabChange(tab)}
                className={`flex items-center justify-center gap-3 px-8 py-4 rounded-lg text-sm font-semibold transition-all duration-200 relative ${
                  isActive 
                    ? 'bg-blue-500 text-white shadow-lg' 
                    : 'text-gray-600 bg-transparent hover:bg-gray-50'
                }`}
              >
                {icons[tab as keyof typeof icons]}
                <span>{labels[tab as keyof typeof labels]}</span>
                {tab === 'pending' && pendingData && pendingData.total_count > 0 && (
                  <div className="ml-2 px-2 py-1 text-xs rounded-full font-bold bg-orange-500 text-white min-w-[20px] flex items-center justify-center">
                    {pendingData.total_count > 99 ? '99+' : pendingData.total_count}
                  </div>
                )}
              </button>
            );
          })}
        </div>
      </div>

      <Tabs value={activeTab} onValueChange={handleTabChange}>

        <TabsContent value="all" className="space-y-6 mt-6">
          {/* Stats */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <Card className="border border-gray-200 shadow-sm">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="flex-1">
                    <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Total Properties</p>
                    <p className="text-2xl font-semibold text-gray-900 mt-1">{propertiesData?.total_count || 0}</p>
                  </div>
                  <div className="p-2 bg-[#2B256D]/10 rounded-lg">
                    <Home className="w-5 h-5 text-[#2B256D]" />
                  </div>
                </div>
              </CardContent>
            </Card>
            
            <Card className="border border-gray-200 shadow-sm">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="flex-1">
                    <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Active</p>
                    <p className="text-2xl font-semibold text-gray-900 mt-1">
                      {propertyStats?.active_properties || 0}
                    </p>
                  </div>
                  <div className="p-2 bg-green-100 rounded-lg">
                    <Building className="w-5 h-5 text-green-600" />
                  </div>
                </div>
              </CardContent>
            </Card>
            
            <Card className="border border-gray-200 shadow-sm">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="flex-1">
                    <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Total Views</p>
                    <p className="text-2xl font-semibold text-gray-900 mt-1">
                      {propertiesData?.properties.reduce((acc, p) => acc + p.views, 0) || 0}
                    </p>
                  </div>
                  <div className="p-2 bg-blue-100 rounded-lg">
                    <Eye className="w-5 h-5 text-blue-600" />
                  </div>
                </div>
              </CardContent>
            </Card>
            
            <Card className="border border-gray-200 shadow-sm">
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="flex-1">
                    <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Total Inquiries</p>
                    <p className="text-2xl font-semibold text-gray-900 mt-1">
                      {propertiesData?.properties.reduce((acc, p) => acc + p.inquiries, 0) || 0}
                    </p>
                  </div>
                  <div className="p-2 bg-purple-100 rounded-lg">
                    <MapPin className="w-5 h-5 text-purple-600" />
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Table */}
          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-0">


              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50 border-b">
                    <tr>
                      <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Property</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Type</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Location</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Price</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Status</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Performance</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Created</th>
                      <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Actions</th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {loading ? (
                      Array.from({ length: 5 }).map((_, i) => (
                        <tr key={i} className="animate-pulse">
                          <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-32"></div></td>
                          <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-20"></div></td>
                          <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-24"></div></td>
                          <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-20"></div></td>
                          <td className="px-4 py-3"><div className="h-6 bg-gray-200 rounded w-16"></div></td>
                          <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-16"></div></td>
                          <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-20"></div></td>
                          <td className="px-4 py-3"><div className="h-8 bg-gray-200 rounded w-24"></div></td>
                        </tr>
                      ))
                    ) : propertiesData?.properties.map((property) => (
                      <tr key={property.id} className="hover:bg-gray-50">
                        <td className="px-4 py-3">
                          <div className="flex items-center space-x-3">
                            <div className="w-7 h-7 bg-[#2B256D] rounded-full flex items-center justify-center text-white text-xs font-medium">
                              <Home className="w-3 h-3" />
                            </div>
                            <div>
                              <div className="text-sm font-medium text-gray-900">
                                {property.title}
                              </div>
                              <div className="text-xs text-gray-500">ID: {String(property.id).slice(0, 8)}</div>
                            </div>
                          </div>
                        </td>
                        <td className="px-4 py-3">
                          <div className="space-y-1">
                            {getListingTypeBadge(property.listing_type)}
                            <div className="text-xs text-gray-500">{property.property_type}</div>
                          </div>
                        </td>
                        <td className="px-4 py-3">
                          <div className="text-sm text-gray-900">{property.city}</div>
                        </td>
                        <td className="px-4 py-3">
                          <div className="text-sm font-medium text-gray-900">
                            {formatPrice(property.sale_price, property.monthly_rent, property.listing_type)}
                          </div>
                        </td>
                        <td className="px-4 py-3">
                          {getStatusBadge(property.status)}
                        </td>
                        <td className="px-4 py-3">
                          <div className="text-sm text-gray-900">
                            <div>{property.views} views</div>
                            <div className="text-xs text-gray-500">{property.inquiries} inquiries</div>
                          </div>
                        </td>
                        <td className="px-4 py-3 text-xs text-gray-500">
                          {formatDate(property.created_at)}
                        </td>
                        <td className="px-4 py-3">
                          <div className="flex items-center space-x-1">
                            <Button 
                              variant="ghost" 
                              size="sm" 
                              className="text-blue-600 hover:bg-blue-50 p-1"
                              onClick={() => setShowDetailsModal(property)}
                              title="View Property Details"
                            >
                              <Eye className="w-3 h-3" />
                            </Button>
                            <Button variant="ghost" size="sm" className="text-gray-600 hover:bg-gray-50 p-1">
                              <Edit className="w-3 h-3" />
                            </Button>
                            <Button variant="ghost" size="sm" className="text-red-600 hover:bg-red-50 p-1">
                              <Ban className="w-3 h-3" />
                            </Button>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
              
              {/* Pagination */}
              {propertiesData && propertiesData.total_pages > 1 && (
                <div className="bg-gray-50 px-4 py-3 border-t">
                  <div className="flex items-center justify-between">
                    <div className="text-xs text-gray-700">
                      Showing {((currentPage - 1) * pageSize) + 1} to {Math.min(currentPage * pageSize, propertiesData.total_count)} of {propertiesData.total_count} properties
                    </div>
                    <div className="flex items-center space-x-1">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                        disabled={currentPage === 1}
                        className="text-xs p-2"
                      >
                        <ChevronLeft className="w-3 h-3" />
                      </Button>
                      
                      {Array.from({ length: Math.min(5, propertiesData.total_pages) }, (_, i) => {
                        const page = i + 1;
                        return (
                          <Button
                            key={page}
                            variant={currentPage === page ? "default" : "outline"}
                            size="sm"
                            onClick={() => setCurrentPage(page)}
                            className={currentPage === page ? "bg-[#2B256D] hover:bg-[#1f1a5a] text-xs p-2" : "text-xs p-2"}
                          >
                            {page}
                          </Button>
                        );
                      })}
                      
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setCurrentPage(prev => Math.min(propertiesData.total_pages, prev + 1))}
                        disabled={currentPage === propertiesData.total_pages}
                        className="text-xs p-2"
                      >
                        <ChevronRight className="w-3 h-3" />
                      </Button>
                    </div>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="pending" className="space-y-6 mt-6">
          <Card className="border border-gray-200 shadow-sm">
            <CardContent className="p-0">
              {renderPendingTable()}
              
              {/* Pagination for pending */}
              {pendingData && pendingData.total_pages > 1 && (
                <div className="bg-gray-50 px-4 py-3 border-t">
                  <div className="flex items-center justify-between">
                    <div className="text-xs text-gray-700">
                      Showing {((currentPage - 1) * pageSize) + 1} to {Math.min(currentPage * pageSize, pendingData.total_count)} of {pendingData.total_count} pending properties
                    </div>
                    <div className="flex items-center space-x-1">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                        disabled={currentPage === 1}
                        className="text-xs p-2"
                      >
                        <ChevronLeft className="w-3 h-3" />
                      </Button>
                      
                      {Array.from({ length: Math.min(5, pendingData.total_pages) }, (_, i) => {
                        const page = i + 1;
                        return (
                          <Button
                            key={page}
                            variant={currentPage === page ? "default" : "outline"}
                            size="sm"
                            onClick={() => setCurrentPage(page)}
                            className={currentPage === page ? "bg-[#2B256D] hover:bg-[#1f1a5a] text-xs p-2" : "text-xs p-2"}
                          >
                            {page}
                          </Button>
                        );
                      })}
                      
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => setCurrentPage(prev => Math.min(pendingData.total_pages, prev + 1))}
                        disabled={currentPage === pendingData.total_pages}
                        className="text-xs p-2"
                      >
                        <ChevronRight className="w-3 h-3" />
                      </Button>
                    </div>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>


      </Tabs>

      {/* Approve Modal */}
      {showApproveModal && (
        <div className="fixed inset-0 backdrop-blur-sm  bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4 shadow-xl">
            <h3 className="text-lg font-medium mb-4">Approve Property</h3>
            <p className="text-sm text-gray-600 mb-6">Are you sure you want to approve this property? This action cannot be undone.</p>
            <div className="flex justify-end space-x-3 mt-6">
              <Button
                onClick={() => {
                  setShowApproveModal(null);
                  setApproveMessage('');
                }}
                variant="outline"
                size="sm"
                className="text-xs"
              >
                Cancel
              </Button>
              <Button
                onClick={() => handleApprove(showApproveModal)}
                disabled={actionLoading === showApproveModal}
                size="sm"
                className="bg-blue-500 ml-auto hover:bg-green-700 text-xs"
              >
                {actionLoading === showApproveModal ? 'Approving...' : 'Confirm Approval'}
              </Button>
            </div>
          </div>
        </div>
      )}

      {/* Reject Modal */}
      {showRejectModal && (
        <div className="fixed inset-0 backdrop-blur-sm  bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4 shadow-xl">
            <h3 className="text-lg font-medium mb-4">Reject Property</h3>
            <p className="text-sm text-gray-600 mb-4">Please provide a reason for rejecting this property.</p>
            <textarea
              value={rejectReason}
              onChange={(e) => setRejectReason(e.target.value)}
              placeholder="Enter rejection reason..."
              className="w-full px-3 py-2 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D] resize-none"
              rows={3}
            />
            <div className="flex justify-end space-x-3 mt-6">
              <Button
                onClick={() => {
                  setShowRejectModal(null);
                  setRejectReason('');
                }}
                variant="outline"
                size="sm"
                className="text-xs"
              >
                Cancel
              </Button>
              <Button
                onClick={() => handleReject(showRejectModal)}
                disabled={actionLoading === showRejectModal || !rejectReason.trim()}
                size="sm"
                className="bg-red-600 hover:bg-red-700 text-xs ml-auto"
              >
                {actionLoading === showRejectModal ? 'Rejecting...' : 'Confirm Rejection'}
              </Button>
            </div>
          </div>
        </div>
      )}

      {/* Property Details Modal */}
      {showDetailsModal && (
  <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4 overflow-y-auto">
    <div className="bg-white rounded-lg w-full max-w-5xl my-8 flex flex-col">
      <div className="sticky top-0 bg-white border-b px-6 py-4 flex items-center justify-between z-10">
        <h2 className="text-xl font-semibold">Property Details</h2>
        <Button variant="ghost" size="sm" onClick={() => {
          setShowDetailsModal(null);
          setDetailsTab('overview');
          setLegalHealth(null);
          setLegalDocuments([]);
        }}>
          <X className="w-5 h-5" />
        </Button>
      </div>
      
      {/* Tabs */}
      <div className="border-b px-6 bg-white sticky top-[73px] z-10">
        <div className="flex gap-4">
          <button
            onClick={() => setDetailsTab('overview')}
            className={`px-4 py-3 text-sm font-medium border-b-2 transition-colors ${
              detailsTab === 'overview'
                ? 'border-blue-600 text-blue-600'
                : 'border-transparent text-gray-600 hover:text-gray-900'
            }`}
          >
            Overview
          </button>
          <button
            onClick={() => {
              setDetailsTab('legal');
              if (!legalHealth) fetchLegalData(String(showDetailsModal.id));
            }}
            className={`px-4 py-3 text-sm font-medium border-b-2 transition-colors ${
              detailsTab === 'legal'
                ? 'border-blue-600 text-blue-600'
                : 'border-transparent text-gray-600 hover:text-gray-900'
            }`}
          >
            Legal Documents
          </button>
        </div>
      </div>
      
      <div className="p-6 overflow-y-auto max-h-[calc(90vh-200px)]">
        {detailsTab === 'overview' ? (
          <div className="space-y-6">
            {/* Existing overview content */}
            <div>
              <h3 className="text-lg font-medium mb-3">Basic Information</h3>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div><span className="font-medium">Title:</span> {showDetailsModal.title}</div>
                <div><span className="font-medium">ID:</span> {showDetailsModal.id}</div>
                <div><span className="font-medium">Type:</span> {showDetailsModal.property_type}</div>
                <div><span className="font-medium">Listing:</span> {showDetailsModal.listing_type}</div>
                <div><span className="font-medium">Status:</span> {getStatusBadge(showDetailsModal.status)}</div>
                <div><span className="font-medium">Price:</span> {formatPrice(showDetailsModal.sale_price, showDetailsModal.monthly_rent, showDetailsModal.listing_type)}</div>
              </div>
            </div>
            <div>
              <h3 className="text-lg font-medium mb-3">Location</h3>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div><span className="font-medium">City:</span> {showDetailsModal.city}</div>
                <div><span className="font-medium">Created:</span> {formatDate(showDetailsModal.created_at)}</div>
              </div>
            </div>
            <div>
              <h3 className="text-lg font-medium mb-3">Performance Metrics</h3>
              <div className="grid grid-cols-3 gap-4 text-sm">
                <div><span className="font-medium">Views:</span> {showDetailsModal.views}</div>
                <div><span className="font-medium">Inquiries:</span> {showDetailsModal.inquiries}</div>
                <div><span className="font-medium">Created:</span> {formatDate(showDetailsModal.created_at)}</div>
              </div>
            </div>
          </div>
        ) : (
          <PropertyLegalTab 
            legalHealth={legalHealth}
            legalDocuments={legalDocuments}
            loadingLegal={loadingLegal}
          />
        )}
      </div>
      
      <div className="border-t px-6 py-4 flex justify-end space-x-3 bg-white sticky bottom-0">
        <Button variant="outline" size="sm" onClick={() => window.open(`/property/${showDetailsModal.id}`, '_blank')}>
          View Full Page
        </Button>
        <Button variant="outline" size="sm" onClick={() => {
          setShowDetailsModal(null);
          setDetailsTab('overview');
          setLegalHealth(null);
          setLegalDocuments([]);
        }}>
          Close
        </Button>
      </div>
    </div>
  </div>
)}
    </div>
  );
}