import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../ui/card";
import { Button } from "../ui/button";
import {
  RefreshCw,
  TrendingUp,
  TrendingDown,
  Building,
  MapPin,
  Eye,
  MessageSquare,
  Activity,
  Percent,
} from "lucide-react";
import {
  getAdminPropertyStats,
  getAdminPropertiesList,
  AdminPropertyStats,
  AdminPropertyResponse,
} from "../../api/admin";
import { getAdminAbsorptionRate } from "../../api/analytics";
import { toast } from "sonner";

export function PropertyTrends() {
  const [propertyStats, setPropertyStats] = useState<AdminPropertyStats | null>(
    null
  );
  const [properties, setProperties] = useState<AdminPropertyResponse[]>([]);
  const [loading, setLoading] = useState(true);
  const [absorptionData, setAbsorptionData] = useState<{
    absorption_rate: number;
    sold_properties: number;
    active_properties: number;
  } | null>(null);
  const [demandAnalysis, setDemandAnalysis] = useState<{
    highDemandAreas: Array<{
      city: string;
      properties: number;
      avgViews: number;
      avgInquiries: number;
    }>;
    risingDemandAreas: Array<{
      city: string;
      inquiryRate: number;
      viewRate: number;
    }>;
    topPerformingProperties: AdminPropertyResponse[];
  } | null>(null);

  const fetchPropertyStats = async () => {
    try {
      setLoading(true);

      // Fetch property stats
      const statsData = await getAdminPropertyStats();
      setPropertyStats(statsData);

      // Fetch absorption rate from backend
      try {
        const absorptionResponse = await getAdminAbsorptionRate('1y');
        console.log('Admin absorption rate response:', absorptionResponse);
        setAbsorptionData({
          absorption_rate: absorptionResponse.absorption_rate,
          sold_properties: absorptionResponse.sold_properties,
          active_properties: absorptionResponse.active_properties
        });
      } catch (absorptionError) {
        console.error('Failed to fetch absorption rate:', absorptionError);
        setAbsorptionData(null);
      }

      // Try to get properties list (optional)
      try {
        const propertiesData = await getAdminPropertiesList({ limit: 100 });
        setProperties(propertiesData.properties || []);

        // Analyze demand patterns if we have properties
        if (propertiesData.properties && propertiesData.properties.length > 0) {
          const analysis = analyzeDemandPatterns(propertiesData.properties);
          setDemandAnalysis(analysis);
        } else {
          setDemandAnalysis(null);
        }
      } catch (propertiesError) {
        console.warn(
          "Properties list API failed, continuing with stats only:",
          propertiesError
        );
        setProperties([]);
        setDemandAnalysis(null);
      }
    } catch (error) {
      console.error("Error fetching property stats:", error);
      toast.error("Failed to fetch property trends");
    } finally {
      setLoading(false);
    }
  };

  const analyzeDemandPatterns = (properties: AdminPropertyResponse[]) => {
    if (!properties || properties.length === 0) {
      return {
        highDemandAreas: [],
        risingDemandAreas: [],
        topPerformingProperties: [],
      };
    }

    // Group properties by city and calculate metrics
    const cityMetrics = properties.reduce((acc, property) => {
      const city = property.city || "Unknown";
      if (!acc[city]) {
        acc[city] = { properties: [], totalViews: 0, totalInquiries: 0 };
      }
      acc[city].properties.push(property);
      acc[city].totalViews += property.views || 0;
      acc[city].totalInquiries += property.inquiries || 0;
      return acc;
    }, {} as Record<string, { properties: AdminPropertyResponse[]; totalViews: number; totalInquiries: number }>);

    // Calculate high demand areas
    const highDemandAreas = Object.entries(cityMetrics)
      .map(([city, data]) => ({
        city,
        properties: data.properties.length,
        avgViews:
          data.properties.length > 0
            ? Math.round(data.totalViews / data.properties.length)
            : 0,
        avgInquiries:
          data.properties.length > 0
            ? Math.round(data.totalInquiries / data.properties.length)
            : 0,
      }))
      .sort(
        (a, b) => b.avgViews + b.avgInquiries - (a.avgViews + a.avgInquiries)
      )
      .slice(0, 5);

    // Calculate rising demand (areas with high inquiry-to-view ratio)
    const risingDemandAreas = Object.entries(cityMetrics)
      .map(([city, data]) => {
        const viewRate = data.totalViews;
        const inquiryRate = data.totalInquiries;
        return {
          city,
          inquiryRate,
          viewRate,
          conversionRate: viewRate > 0 ? (inquiryRate / viewRate) * 100 : 0,
        };
      })
      .filter((area) => area.viewRate > 10) // Only areas with significant views
      .sort((a, b) => b.conversionRate - a.conversionRate)
      .slice(0, 5);

    // Top performing properties
    const topPerformingProperties = properties
      .filter((p) => (p.views || 0) > 0)
      .sort(
        (a, b) =>
          (b.views || 0) +
          (b.inquiries || 0) * 5 -
          ((a.views || 0) + (a.inquiries || 0) * 5)
      )
      .slice(0, 5);

    return {
      highDemandAreas,
      risingDemandAreas,
      topPerformingProperties,
    };
  };

  useEffect(() => {
    fetchPropertyStats();
  }, []);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!propertyStats) {
    return (
      <div className="text-center py-12">
        <p className="text-gray-500 mb-4">Failed to load property trends</p>
        <Button
          onClick={fetchPropertyStats}
          className="bg-blue-600 hover:bg-blue-700"
        >
          <RefreshCw className="w-4 h-4 mr-2" />
          Retry
        </Button>
      </div>
    );
  }

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900">Property Trends</h1>
          <p className="text-sm text-gray-600 mt-1">Market insights and performance analytics</p>
        </div>
        <Button
          onClick={fetchPropertyStats}
          variant="outline"
          className="flex items-center gap-2"
        >
          <RefreshCw className="w-4 h-4" />
          Refresh
        </Button>
      </div>

      {/* Real-time Market Insights */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Active Properties</p>
                <p className="text-3xl font-semibold text-gray-900 mt-2">
                  {propertyStats.active_properties.toLocaleString()}
                </p>
                <p className="text-sm text-green-600 mt-1">+{propertyStats.new_properties_today || 0} today</p>
              </div>
              <div className="p-3 bg-green-100 rounded-lg">
                <TrendingUp className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Average Price</p>
                <p className="text-3xl font-semibold text-gray-900 mt-2">
                  {propertyStats.average_price && propertyStats.average_price > 0
                    ? `₹${(propertyStats.average_price / 10000000).toFixed(2)} Cr`
                    : "N/A"}
                </p>
                <p className="text-sm text-gray-600 mt-1">Market average</p>
              </div>
              <div className="p-3 bg-blue-100 rounded-lg">
                <Building className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Property Analytics */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card className="border border-gray-200 shadow-sm">
          <CardHeader className="pb-3">
            <CardTitle className="text-base font-medium">Property Type Distribution</CardTitle>
          </CardHeader>
          <CardContent className="pt-0">
            {Object.keys(propertyStats.properties_by_type || {}).length > 0 ? (
              <div className="space-y-3">
                {Object.entries(propertyStats.properties_by_type || {})
                  .sort(([, a], [, b]) => b - a)
                  .map(([type, count]) => {
                    const percentage =
                      propertyStats.total_properties > 0
                        ? (count / propertyStats.total_properties) * 100
                        : 0;
                    return (
                      <div
                        key={type}
                        className="flex items-center justify-between"
                      >
                        <span className="text-sm font-medium text-gray-700 capitalize">
                          {type.replace("_", " ")}
                        </span>
                        <div className="flex items-center gap-3">
                          <div className="w-24 bg-gray-200 rounded-full h-2">
                            <div
                              className="bg-[#2B256D] h-2 rounded-full transition-all"
                              style={{ width: `${percentage}%` }}
                            ></div>
                          </div>
                          <span className="text-sm font-semibold text-gray-900 w-8 text-right">
                            {count}
                          </span>
                        </div>
                      </div>
                    );
                  })}
              </div>
            ) : (
              <div className="text-center py-8 text-gray-500">
                <Building className="w-12 h-12 mx-auto mb-4 text-gray-300" />
                <p className="text-sm">No property type data available</p>
              </div>
            )}
          </CardContent>
        </Card>

        <Card className="border border-gray-200 shadow-sm">
          <CardHeader className="pb-3">
            <CardTitle className="text-base font-medium">Market Performance</CardTitle>
          </CardHeader>
          <CardContent className="pt-0">
            <div className="space-y-4">
              <div className="p-4 bg-green-50 rounded-lg border border-green-200">
                <div className="flex items-center gap-2 mb-2">
                  <TrendingUp className="w-4 h-4 text-green-600" />
                  <span className="text-sm font-medium text-green-700">Sold Properties</span>
                </div>
                <p className="text-2xl font-bold text-gray-900">
                  {propertyStats.sold_properties.toLocaleString()}
                </p>
                <p className="text-xs text-gray-600 mt-1">
                  {propertyStats.total_properties > 0
                    ? Math.round(
                        (propertyStats.sold_properties /
                          propertyStats.total_properties) *
                          100
                      )
                    : 0}
                  % of total
                </p>
              </div>
              <div className="p-4 bg-purple-50 rounded-lg border border-purple-200">
                <div className="flex items-center gap-2 mb-2">
                  <Percent className="w-4 h-4 text-purple-600" />
                  <span className="text-sm font-medium text-purple-700">Absorption Rate</span>
                </div>
                <p className="text-2xl font-bold text-gray-900">
                  {absorptionData?.absorption_rate?.toFixed(1) || "0.0"}%
                </p>
                <p className="text-xs text-gray-600 mt-1">Platform-wide rate</p>
              </div>
              <div className="p-4 bg-orange-50 rounded-lg border border-orange-200">
                <div className="flex items-center gap-2 mb-2">
                  <Building className="w-4 h-4 text-orange-600" />
                  <span className="text-sm font-medium text-orange-700">Rented Properties</span>
                </div>
                <p className="text-2xl font-bold text-gray-900">
                  {propertyStats.rented_properties.toLocaleString()}
                </p>
                <p className="text-xs text-gray-600 mt-1">
                  {propertyStats.total_properties > 0
                    ? Math.round(
                        (propertyStats.rented_properties /
                          propertyStats.total_properties) *
                          100
                      )
                    : 0}
                  % of total
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Demand Analysis */}
      {demandAnalysis && demandAnalysis.highDemandAreas.length > 0 && (
        <>
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            <Card className="border-0 shadow-md">
              <CardHeader className="pb-4">
                <CardTitle className="flex items-center gap-2">
                  <Activity className="w-5 h-5 text-red-600" />
                  High Demand Areas
                </CardTitle>
              </CardHeader>
              <CardContent className="pt-0">
                <div className="space-y-4">
                  {demandAnalysis.highDemandAreas.map((area, index) => (
                    <div
                      key={area.city}
                      className="flex items-center justify-between p-3 bg-red-50 rounded-lg"
                    >
                      <div className="flex items-center gap-3">
                        <div
                          className={`w-6 h-6 rounded-full flex items-center justify-center text-xs font-bold ${
                            index === 0
                              ? "bg-red-600 text-white"
                              : index === 1
                              ? "bg-red-500 text-white"
                              : "bg-red-400 text-white"
                          }`}
                        >
                          {index + 1}
                        </div>
                        <div>
                          <p className="font-medium text-gray-800">
                            {area.city}
                          </p>
                          <p className="text-sm text-gray-600">
                            {area.properties} properties
                          </p>
                        </div>
                      </div>
                      <div className="text-right">
                        <div className="flex items-center gap-2 text-sm">
                          <Eye className="w-4 h-4 text-blue-600" />
                          <span>{area.avgViews}</span>
                          <MessageSquare className="w-4 h-4 text-green-600" />
                          <span>{area.avgInquiries}</span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            <Card className="border-0 shadow-md">
              <CardHeader className="pb-4">
                <CardTitle className="flex items-center gap-2">
                  <TrendingUp className="w-5 h-5 text-green-600" />
                  Rising Interest Areas
                </CardTitle>
              </CardHeader>
              <CardContent className="pt-0">
                <div className="space-y-4">
                  {demandAnalysis.risingDemandAreas.map((area, index) => (
                    <div
                      key={area.city}
                      className="flex items-center justify-between p-3 bg-green-50 rounded-lg"
                    >
                      <div className="flex items-center gap-3">
                        <TrendingUp className="w-5 h-5 text-green-600" />
                        {/* <div>
                          <p className="font-medium text-gray-800">{area.city}</p>
                          <p className="text-sm text-gray-600">
                            {area.conversionRate.toFixed(1)}% inquiry rate
                          </p>
                        </div> */}
                      </div>
                      <div className="text-right">
                        <div className="flex items-center gap-2 text-sm">
                          <Eye className="w-4 h-4 text-blue-600" />
                          <span>{area.viewRate}</span>
                          <MessageSquare className="w-4 h-4 text-green-600" />
                          <span>{area.inquiryRate}</span>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>

          <Card className="border-0 shadow-md">
            <CardHeader className="pb-4">
              <CardTitle className="flex items-center gap-2">
                <Building className="w-5 h-5 text-purple-600" />
                Top Performing Properties
              </CardTitle>
            </CardHeader>
            <CardContent className="pt-0">
              <div className="space-y-4">
                {demandAnalysis.topPerformingProperties.map(
                  (property, index) => (
                    <div
                      key={property.id}
                      className="flex items-center justify-between p-4 border rounded-lg hover:bg-gray-50"
                    >
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-2">
                          <div
                            className={`w-6 h-6 rounded-full flex items-center justify-center text-xs font-bold ${
                              index === 0
                                ? "bg-purple-600 text-white"
                                : index === 1
                                ? "bg-purple-500 text-white"
                                : "bg-purple-400 text-white"
                            }`}
                          >
                            {index + 1}
                          </div>
                          <h4 className="font-medium text-gray-900 truncate">
                            {property.title}
                          </h4>
                        </div>
                        <div className="flex items-center gap-4 text-sm text-gray-600 ml-9">
                          <span className="capitalize">
                            {property.property_type}
                          </span>
                          <span>•</span>
                          <span>{property.city}</span>
                          <span>•</span>
                          <span>
                            {property.listing_type === "sell"
                              ? `₹${(property.sale_price / 100000).toFixed(1)}L`
                              : `₹${property.monthly_rent?.toLocaleString()}/month`}
                          </span>
                        </div>
                      </div>
                      <div className="flex items-center gap-4 text-sm">
                        <div className="flex items-center gap-1">
                          <Eye className="w-4 h-4 text-blue-600" />
                          <span className="font-medium">
                            {property.views || 0}
                          </span>
                        </div>
                        <div className="flex items-center gap-1">
                          <MessageSquare className="w-4 h-4 text-green-600" />
                          <span className="font-medium">
                            {property.inquiries || 0}
                          </span>
                        </div>
                      </div>
                    </div>
                  )
                )}
              </div>
            </CardContent>
          </Card>
        </>
      )}

      {/* Recent Activity Summary */}
      <Card className="border border-gray-200 shadow-sm">
        <CardHeader className="pb-3">
          <CardTitle className="text-base font-medium">Recent Property Activity</CardTitle>
        </CardHeader>
        <CardContent className="pt-0">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="p-4 bg-blue-50 rounded-lg border border-blue-200">
              <div className="flex items-center gap-2 mb-2">
                <TrendingUp className="w-4 h-4 text-blue-600" />
                <span className="text-sm font-medium text-blue-700">Today</span>
              </div>
              <p className="text-2xl font-bold text-gray-900">
                {propertyStats.new_properties_today || 0}
              </p>
              <p className="text-xs text-gray-600 mt-1">New Properties</p>
            </div>
            <div className="p-4 bg-green-50 rounded-lg border border-green-200">
              <div className="flex items-center gap-2 mb-2">
                <TrendingUp className="w-4 h-4 text-green-600" />
                <span className="text-sm font-medium text-green-700">This Week</span>
              </div>
              <p className="text-2xl font-bold text-gray-900">
                {propertyStats.new_properties_this_week || 0}
              </p>
              <p className="text-xs text-gray-600 mt-1">New Properties</p>
            </div>
            <div className="p-4 bg-purple-50 rounded-lg border border-purple-200">
              <div className="flex items-center gap-2 mb-2">
                <Building className="w-4 h-4 text-purple-600" />
                <span className="text-sm font-medium text-purple-700">This Month</span>
              </div>
              <p className="text-2xl font-bold text-gray-900">
                {propertyStats.new_properties_this_month || 0}
              </p>
              <p className="text-xs text-gray-600 mt-1">New Properties</p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
