import React, { useState } from 'react';
import { Upload, CheckCircle, XCircle, AlertTriangle, Download, RefreshCw } from 'lucide-react';

interface ReconciliationMismatch {
  id: string;
  payment_id: string;
  order_id: string;
  psp_amount: number;
  db_amount: number;
  difference: number;
  psp_status: string;
  db_status: string;
  transaction_date: string;
  gateway: string;
}

export function ReconciliationDashboard() {
  const [uploading, setUploading] = useState(false);
  const [mismatches, setMismatches] = useState<ReconciliationMismatch[]>([]);
  const [reconciliationStats, setReconciliationStats] = useState({
    total_transactions: 0,
    matched: 0,
    mismatched: 0,
    pending_review: 0
  });

  const handleFileUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    setUploading(true);
    const formData = new FormData();
    formData.append('file', file);

    try {
      const token = localStorage.getItem('token');
      const response = await fetch('/api/payments/admin/reconciliation/upload', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`
        },
        body: formData
      });

      if (response.ok) {
        const data = await response.json();
        setMismatches(data.mismatches || []);
        setReconciliationStats(data.stats || {});
        alert('Reconciliation completed successfully!');
      } else {
        alert('Failed to upload reconciliation file');
      }
    } catch (error) {
      console.error('Upload error:', error);
      alert('Failed to upload file');
    } finally {
      setUploading(false);
    }
  };

  const resolveMismatch = async (mismatchId: string, resolution: 'accept_psp' | 'accept_db' | 'manual') => {
    try {
      const token = localStorage.getItem('token');
      const response = await fetch('/api/payments/admin/reconciliation/resolve', {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          mismatch_id: mismatchId,
          resolution: resolution
        })
      });

      if (response.ok) {
        setMismatches(mismatches.filter(m => m.id !== mismatchId));
        alert('Mismatch resolved successfully');
      } else {
        alert('Failed to resolve mismatch');
      }
    } catch (error) {
      console.error('Resolution error:', error);
      alert('Failed to resolve mismatch');
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h2 className="text-2xl font-bold text-gray-900">Payment Reconciliation</h2>
        <p className="text-gray-600 mt-1">Upload PSP settlement files and resolve mismatches</p>
      </div>

      {/* Upload Section */}
      <div className="bg-white border-2 border-dashed border-gray-300 rounded-lg p-8">
        <div className="text-center">
          <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-semibold text-gray-900 mb-2">Upload Settlement File</h3>
          <p className="text-sm text-gray-600 mb-4">
            Upload CSV or Excel file from Razorpay, Cashfree, or Paytm
          </p>
          <label className="inline-block">
            <input
              type="file"
              accept=".csv,.xlsx,.xls"
              onChange={handleFileUpload}
              className="hidden"
              disabled={uploading}
            />
            <span className="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors cursor-pointer inline-flex items-center gap-2">
              {uploading ? (
                <>
                  <RefreshCw className="w-4 h-4 animate-spin" />
                  Processing...
                </>
              ) : (
                <>
                  <Upload className="w-4 h-4" />
                  Choose File
                </>
              )}
            </span>
          </label>
        </div>
      </div>

      {/* Stats Cards */}
      {reconciliationStats.total_transactions > 0 && (
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
            <p className="text-sm text-blue-700 font-medium">Total Transactions</p>
            <p className="text-3xl font-bold text-blue-900 mt-2">
              {reconciliationStats.total_transactions}
            </p>
          </div>
          <div className="bg-green-50 border border-green-200 rounded-lg p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-700 font-medium">Matched</p>
                <p className="text-3xl font-bold text-green-900 mt-2">
                  {reconciliationStats.matched}
                </p>
              </div>
              <CheckCircle className="w-8 h-8 text-green-600" />
            </div>
          </div>
          <div className="bg-red-50 border border-red-200 rounded-lg p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-red-700 font-medium">Mismatched</p>
                <p className="text-3xl font-bold text-red-900 mt-2">
                  {reconciliationStats.mismatched}
                </p>
              </div>
              <XCircle className="w-8 h-8 text-red-600" />
            </div>
          </div>
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-yellow-700 font-medium">Pending Review</p>
                <p className="text-3xl font-bold text-yellow-900 mt-2">
                  {reconciliationStats.pending_review}
                </p>
              </div>
              <AlertTriangle className="w-8 h-8 text-yellow-600" />
            </div>
          </div>
        </div>
      )}

      {/* Mismatches Queue */}
      {mismatches.length > 0 && (
        <div className="bg-white border border-gray-200 rounded-lg overflow-hidden">
          <div className="px-6 py-4 border-b border-gray-200 bg-gray-50">
            <h3 className="text-lg font-semibold text-gray-900">Reconciliation Mismatches</h3>
            <p className="text-sm text-gray-600 mt-1">Review and resolve discrepancies</p>
          </div>

          <div className="divide-y divide-gray-200">
            {mismatches.map((mismatch) => (
              <div key={mismatch.id} className="p-6 hover:bg-gray-50 transition-colors">
                <div className="flex items-start justify-between">
                  <div className="flex-1">
                    <div className="flex items-center gap-3 mb-3">
                      <span className="px-3 py-1 bg-red-100 text-red-700 rounded-full text-xs font-medium">
                        Mismatch
                      </span>
                      <span className="px-3 py-1 bg-blue-100 text-blue-700 rounded-full text-xs font-medium">
                        {mismatch.gateway}
                      </span>
                    </div>

                    <h4 className="font-semibold text-gray-900 mb-2">
                      Order #{mismatch.order_id}
                    </h4>

                    <div className="grid grid-cols-2 gap-4 mb-4">
                      <div className="bg-blue-50 border border-blue-200 rounded p-3">
                        <p className="text-xs text-blue-700 font-medium mb-1">PSP Record</p>
                        <p className="text-lg font-bold text-blue-900">
                          ₹{mismatch.psp_amount.toLocaleString('en-IN')}
                        </p>
                        <p className="text-xs text-blue-600 mt-1">Status: {mismatch.psp_status}</p>
                      </div>
                      <div className="bg-purple-50 border border-purple-200 rounded p-3">
                        <p className="text-xs text-purple-700 font-medium mb-1">Database Record</p>
                        <p className="text-lg font-bold text-purple-900">
                          ₹{mismatch.db_amount.toLocaleString('en-IN')}
                        </p>
                        <p className="text-xs text-purple-600 mt-1">Status: {mismatch.db_status}</p>
                      </div>
                    </div>

                    <div className="bg-red-50 border border-red-200 rounded p-3">
                      <p className="text-sm font-medium text-red-900">
                        Difference: ₹{Math.abs(mismatch.difference).toLocaleString('en-IN')}
                      </p>
                      <p className="text-xs text-red-700 mt-1">
                        Transaction Date: {new Date(mismatch.transaction_date).toLocaleDateString('en-IN')}
                      </p>
                    </div>
                  </div>

                  <div className="ml-6 space-y-2">
                    <button
                      onClick={() => resolveMismatch(mismatch.id, 'accept_psp')}
                      className="w-full px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm font-medium"
                    >
                      Accept PSP
                    </button>
                    <button
                      onClick={() => resolveMismatch(mismatch.id, 'accept_db')}
                      className="w-full px-4 py-2 bg-purple-600 text-white rounded-lg hover:bg-purple-700 transition-colors text-sm font-medium"
                    >
                      Accept DB
                    </button>
                    <button
                      onClick={() => resolveMismatch(mismatch.id, 'manual')}
                      className="w-full px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors text-sm font-medium"
                    >
                      Manual Review
                    </button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Empty State */}
      {mismatches.length === 0 && reconciliationStats.total_transactions === 0 && (
        <div className="text-center py-12 bg-gray-50 rounded-lg border-2 border-dashed border-gray-300">
          <CheckCircle className="w-12 h-12 text-gray-400 mx-auto mb-3" />
          <h3 className="font-semibold text-gray-900 mb-2">No Reconciliation Data</h3>
          <p className="text-gray-600">Upload a settlement file to start reconciliation</p>
        </div>
      )}
    </div>
  );
}
