import React, { useEffect, useState } from 'react';
import { Button } from '@/components/ui/button';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';

export function RefundManagement() {
  const [refunds, setRefunds] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedRefund, setSelectedRefund] = useState<any>(null);
  const [actionDialog, setActionDialog] = useState(false);
  const [rejectionReason, setRejectionReason] = useState('');

  useEffect(() => {
    fetchRefunds();
  }, []);

  const fetchRefunds = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/transactions/refunds/list', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      
      if (!response.ok) {
        console.error('Refunds API error:', response.status);
        setRefunds([]);
        return;
      }
      
      const data = await response.json();
      // Ensure data is an array
      setRefunds(Array.isArray(data) ? data : []);
    } catch (error) {
      console.error('Failed to fetch refunds:', error);
      setRefunds([]);
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async (refundId: string, approved: boolean) => {
    try {
      const token = localStorage.getItem('access_token');
      await fetch(`http://localhost:8090/api/transactions/refunds/${refundId}/approve?approved=${approved}${!approved && rejectionReason ? `&reason=${rejectionReason}` : ''}`, {
        method: 'PUT',
        headers: { 'Authorization': `Bearer ${token}` }
      });
      setActionDialog(false);
      setRejectionReason('');
      fetchRefunds();
    } catch (error) {
      console.error('Failed to process refund:', error);
    }
  };

  const getStatusBadge = (status: string) => {
    const colors: any = {
      pending: 'bg-yellow-100 text-yellow-800',
      approved: 'bg-green-100 text-green-800',
      rejected: 'bg-red-100 text-red-800',
      completed: 'bg-blue-100 text-blue-800'
    };
    return <span className={`px-2 py-1 rounded text-xs ${colors[status] || 'bg-gray-100 text-gray-800'}`}>{status}</span>;
  };

  if (loading) return <div className="text-center py-8">Loading refunds...</div>;

  return (
    <div className="space-y-6">
      <h1 className="text-2xl font-bold text-gray-900">Refund Management</h1>

      {refunds.length === 0 ? (
        <div className="bg-white rounded-lg shadow p-8 text-center">
          <p className="text-gray-600">No refund requests found</p>
        </div>
      ) : (
        <div className="bg-white rounded-lg shadow overflow-hidden">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Refund #</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Transaction #</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Amount</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Reason</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-gray-200">
              {refunds.map((refund) => (
                <tr key={refund.id}>
                  <td className="px-6 py-4 text-sm text-gray-900">{refund.refund_number}</td>
                  <td className="px-6 py-4 text-sm text-gray-900">{refund.transaction_number}</td>
                  <td className="px-6 py-4 text-sm text-gray-900">₹{refund.refund_amount?.toLocaleString()}</td>
                  <td className="px-6 py-4 text-sm text-gray-600">{refund.refund_reason}</td>
                  <td className="px-6 py-4">{getStatusBadge(refund.refund_status)}</td>
                  <td className="px-6 py-4 text-sm text-gray-500">{new Date(refund.requested_at).toLocaleDateString()}</td>
                  <td className="px-6 py-4">
                    {refund.refund_status === 'pending' && (
                      <div className="flex gap-2">
                        <Button size="sm" onClick={() => handleApprove(refund.id, true)}>Approve</Button>
                        <Button size="sm" variant="outline" onClick={() => { setSelectedRefund(refund); setActionDialog(true); }}>Reject</Button>
                      </div>
                    )}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

      <Dialog open={actionDialog} onOpenChange={setActionDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Reject Refund</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <textarea
              className="w-full border rounded p-2"
              placeholder="Rejection reason..."
              value={rejectionReason}
              onChange={(e) => setRejectionReason(e.target.value)}
              rows={3}
            />
            <div className="flex gap-2 justify-end">
              <Button variant="outline" onClick={() => setActionDialog(false)}>Cancel</Button>
              <Button onClick={() => selectedRefund && handleApprove(selectedRefund.id, false)}>Confirm Reject</Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}
