import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';
import { Button } from '../ui/button';
import { Textarea } from '../ui/textarea';
import { AlertTriangle, FileText, CheckCircle, XCircle, Clock, Eye, Download } from 'lucide-react';
import { riskComplianceService } from '../../api/riskCompliance';

interface RiskSubmission {
  id: string;
  property_address: string;
  user_name: string;
  status: string;
  risk_level: string | null;
  overall_score: number | null;
  submitted_at: string;
  documents_count: number;
  admin_review_status?: string;
}

interface SubmissionDetail {
  id: string;
  property_address: string;
  builder_name: string;
  project_name: string;
  property_type: string;
  budget_range: string;
  status: string;
  user_id: string;
  created_at: string;
  documents: Array<{
    id: string;
    document_type: string;
    file_name: string;
    file_url: string;
    status: string;
    uploaded_at: string;
  }>;
  rera_check: {
    rera_id: string;
    status: string;
    details: any;
    checked_at: string;
  } | null;
  ownership_check: {
    owner_name: string;
    status: string;
    details: any;
    checked_at: string;
  } | null;
  encumbrance_check: {
    status: string;
    details: any;
    checked_at: string;
  } | null;
  risk_score: {
    overall_score: number;
    risk_level: string;
    document_score: number;
    rera_score: number;
    ownership_score: number;
    encumbrance_score: number;
    breakdown: any;
    recommendations: string[];
    calculated_at: string;
  } | null;
  admin_review: {
    status: string;
    comments: string;
    admin_id: string;
    reviewed_at: string;
  } | null;
}

export function RiskComplianceAdmin() {
  const [submissions, setSubmissions] = useState<RiskSubmission[]>([]);
  const [selectedSubmission, setSelectedSubmission] = useState<SubmissionDetail | null>(null);
  const [loading, setLoading] = useState(true);
  const [reviewComments, setReviewComments] = useState('');
  const [submittingReview, setSubmittingReview] = useState(false);
  const [isEditingReview, setIsEditingReview] = useState(false);

  useEffect(() => {
    fetchSubmissions();
  }, []);

  const fetchSubmissions = async () => {
    try {
      setLoading(true);
      const data = await riskComplianceService.getAdminReviews();
      setSubmissions(data);
    } catch (error) {
      console.error('Error fetching submissions:', error);
      // Set empty array on error to prevent UI crash
      setSubmissions([]);
    } finally {
      setLoading(false);
    }
  };

  const fetchSubmissionDetail = async (submissionId: string) => {
    try {
      console.log('Fetching submission detail for ID:', submissionId);
      const data = await riskComplianceService.getAdminSubmission(submissionId);
      console.log('Submission detail data:', data);
      console.log('Admin review data:', data.admin_review);
      console.log('Submission status:', data.status);
      setSelectedSubmission(data);
      setReviewComments(data.admin_review?.comments || '');
      setIsEditingReview(false);
    } catch (error) {
      console.error('Error fetching submission detail:', error);
    }
  };

  const handleReview = async (status: 'approved' | 'rejected') => {
    if (!selectedSubmission) return;

    try {
      setSubmittingReview(true);
      await riskComplianceService.adminReviewSubmission(selectedSubmission.id, status, reviewComments);

      // Refresh data
      await fetchSubmissions();
      await fetchSubmissionDetail(selectedSubmission.id);
      setIsEditingReview(false);
      
      // Trigger notification refresh by dispatching a custom event
      window.dispatchEvent(new CustomEvent('refreshNotifications'));
      
      // Show success message
      console.log(`Risk assessment ${status} successfully. Notifications should be sent.`);
    } catch (error) {
      console.error('Error submitting review:', error);
    } finally {
      setSubmittingReview(false);
    }
  };

  const getRiskLevelColor = (level: string | null) => {
    switch (level) {
      case 'low': return 'bg-green-100 text-green-800';
      case 'medium': return 'bg-yellow-100 text-yellow-800';
      case 'high': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'approved': return 'bg-green-100 text-green-800';
      case 'rejected': return 'bg-red-100 text-red-800';
      case 'completed': return 'bg-green-100 text-green-800';
      case 'under_review': return 'bg-yellow-100 text-yellow-800';
      case 'draft': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getCheckStatusIcon = (status: string) => {
    switch (status) {
      case 'pass': return <CheckCircle className="w-4 h-4 text-green-600" />;
      case 'fail': return <XCircle className="w-4 h-4 text-red-600" />;
      case 'warning': return <AlertTriangle className="w-4 h-4 text-yellow-600" />;
      default: return <Clock className="w-4 h-4 text-gray-600" />;
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Risk & Compliance Review</h1>
          <p className="text-gray-600">Review and manage property risk assessments</p>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Submissions List */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertTriangle className="w-5 h-5" />
              Risk Assessment Submissions ({submissions.length})
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3 max-h-96 overflow-y-auto">
              {submissions.length === 0 ? (
                <div className="text-center text-gray-500 py-8">
                  No submissions found or connection error
                </div>
              ) : (
                submissions.map((submission) => (
                <div
                  key={submission.id}
                  className={`p-4 border rounded-lg cursor-pointer transition-colors hover:bg-gray-50 ${
                    selectedSubmission?.id === submission.id ? 'border-blue-500 bg-blue-50' : 'border-gray-200'
                  }`}
                  onClick={() => fetchSubmissionDetail(submission.id)}
                >
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <h3 className="font-medium text-gray-900 truncate">
                        {submission.property_address}
                      </h3>
                      <p className="text-sm text-gray-600">By: {submission.user_name}</p>
                      <div className="flex items-center gap-2 mt-2">
                        <Badge className={getStatusColor(submission.admin_review_status || submission.status)}>
                          {(submission.admin_review_status || submission.status).replace('_', ' ')}
                        </Badge>
                        {submission.admin_review_status === 'approved' && (
                          <Badge className="bg-green-600 text-white">
                            ✓ Approved
                          </Badge>
                        )}
                        {submission.admin_review_status === 'rejected' && (
                          <Badge className="bg-red-600 text-white">
                            ✗ Rejected
                          </Badge>
                        )}
                        {submission.risk_level && (
                          <Badge className={getRiskLevelColor(submission.risk_level)}>
                            {submission.risk_level} risk
                          </Badge>
                        )}
                      </div>
                    </div>
                    <div className="text-right">
                      {submission.overall_score && (
                        <div className="text-lg font-bold text-gray-900">
                          {submission.overall_score}/100
                        </div>
                      )}
                      <div className="text-xs text-gray-500">
                        {submission.documents_count} docs
                      </div>
                    </div>
                  </div>
                </div>
                ))
              )}
            </div>
          </CardContent>
        </Card>

        {/* Submission Detail */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Eye className="w-5 h-5" />
              Submission Details
            </CardTitle>
          </CardHeader>
          <CardContent>
            {selectedSubmission ? (
              <div className="space-y-6">
                {/* Property Info */}
                <div>
                  <h3 className="font-medium text-gray-900 mb-2">Property Information</h3>
                  <div className="grid grid-cols-2 gap-4 text-sm">
                    <div>
                      <span className="text-gray-600">Address:</span>
                      <p className="font-medium">{selectedSubmission.property_address}</p>
                    </div>
                    <div>
                      <span className="text-gray-600">Builder:</span>
                      <p className="font-medium">{selectedSubmission.builder_name}</p>
                    </div>
                    <div>
                      <span className="text-gray-600">Project:</span>
                      <p className="font-medium">{selectedSubmission.project_name}</p>
                    </div>
                    <div>
                      <span className="text-gray-600">Budget:</span>
                      <p className="font-medium">{selectedSubmission.budget_range}</p>
                    </div>
                  </div>
                </div>

                {/* Documents */}
                <div>
                  <h3 className="font-medium text-gray-900 mb-2">Documents ({selectedSubmission.documents.length})</h3>
                  <div className="space-y-2">
                    {selectedSubmission.documents.map((doc) => (
                      <div key={doc.id} className="flex items-center justify-between p-2 bg-gray-50 rounded">
                        <div className="flex items-center gap-2">
                          <FileText className="w-4 h-4 text-gray-600" />
                          <span className="text-sm">{doc.file_name}</span>
                          <Badge variant="outline" className="text-xs">
                            {doc.document_type.replace('_', ' ')}
                          </Badge>
                        </div>
                        <Button size="sm" variant="outline" onClick={() => window.open(doc.file_url, '_blank')}>
                          <Download className="w-3 h-3" />
                        </Button>
                      </div>
                    ))}
                  </div>
                </div>

                {/* Checks */}
                <div>
                  <h3 className="font-medium text-gray-900 mb-2">Verification Checks</h3>
                  <div className="space-y-2">
                    {selectedSubmission.rera_check && (
                      <div className="flex items-center justify-between p-2 bg-gray-50 rounded">
                        <span className="text-sm">RERA Check</span>
                        <div className="flex items-center gap-2">
                          {getCheckStatusIcon(selectedSubmission.rera_check.status)}
                          <span className="text-sm capitalize">{selectedSubmission.rera_check.status}</span>
                        </div>
                      </div>
                    )}
                    {selectedSubmission.ownership_check && (
                      <div className="flex items-center justify-between p-2 bg-gray-50 rounded">
                        <span className="text-sm">Ownership Check</span>
                        <div className="flex items-center gap-2">
                          {getCheckStatusIcon(selectedSubmission.ownership_check.status)}
                          <span className="text-sm capitalize">{selectedSubmission.ownership_check.status}</span>
                        </div>
                      </div>
                    )}
                    {selectedSubmission.encumbrance_check && (
                      <div className="flex items-center justify-between p-2 bg-gray-50 rounded">
                        <span className="text-sm">Encumbrance Check</span>
                        <div className="flex items-center gap-2">
                          {getCheckStatusIcon(selectedSubmission.encumbrance_check.status)}
                          <span className="text-sm capitalize">{selectedSubmission.encumbrance_check.status}</span>
                        </div>
                      </div>
                    )}
                  </div>
                </div>

                {/* Risk Score */}
                {selectedSubmission.risk_score && (
                  <div>
                    <h3 className="font-medium text-gray-900 mb-2">Risk Assessment</h3>
                    <div className="p-4 bg-gray-50 rounded-lg">
                      <div className="flex items-center justify-between mb-3">
                        <span className="text-lg font-bold">Overall Score: {selectedSubmission.risk_score.overall_score}/100</span>
                        <Badge className={getRiskLevelColor(selectedSubmission.risk_score.risk_level)}>
                          {selectedSubmission.risk_score.risk_level} risk
                        </Badge>
                      </div>
                      <div className="grid grid-cols-2 gap-2 text-sm">
                        <div>Documents: {selectedSubmission.risk_score.document_score}/100</div>
                        <div>RERA: {selectedSubmission.risk_score.rera_score}/100</div>
                        <div>Ownership: {selectedSubmission.risk_score.ownership_score}/100</div>
                        <div>Encumbrance: {selectedSubmission.risk_score.encumbrance_score}/100</div>
                      </div>
                    </div>
                  </div>
                )}

                {/* Admin Review */}
                <div>
                  <h3 className="font-medium text-gray-900 mb-2">Admin Review</h3>
                  <Textarea
                    placeholder="Add your review comments..."
                    value={reviewComments}
                    onChange={(e) => setReviewComments(e.target.value)}
                    className="mb-3"
                    disabled={selectedSubmission.admin_review && !isEditingReview}
                  />
                  
                  {/* Check if submission is already reviewed (approved or rejected) */}
                  {(() => {
                    console.log('Button logic check:', {
                      hasAdminReview: !!selectedSubmission.admin_review,
                      isEditingReview,
                      shouldShowEditButton: selectedSubmission.admin_review && !isEditingReview
                    });
                    return selectedSubmission.admin_review && !isEditingReview;
                  })() ? (
                    <div className="flex gap-2 w-full">
                      <Button
                        onClick={() => setIsEditingReview(true)}
                        variant="outline"
                        className="flex-1"
                      >
                        Edit Review
                      </Button>
                    </div>
                  ) : (
                    <div className="flex gap-2 w-full">
                      <Button
                        onClick={() => handleReview('approved')}
                        disabled={submittingReview}
                        style={{ backgroundColor: '#16a34a', color: 'white' }}
                        className="hover:bg-green-700 flex-1"
                      >
                        {submittingReview ? 'Processing...' : 'Approve'}
                      </Button>
                      <Button
                        onClick={() => handleReview('rejected')}
                        disabled={submittingReview}
                        className="bg-red-600 hover:bg-red-700 text-white flex-1"
                      >
                        {submittingReview ? 'Processing...' : 'Reject'}
                      </Button>
                      {isEditingReview && (
                        <Button
                          onClick={() => {
                            setIsEditingReview(false);
                            setReviewComments(selectedSubmission.admin_review?.comments || '');
                          }}
                          variant="outline"
                          className="flex-1"
                        >
                          Cancel
                        </Button>
                      )}
                    </div>
                  )}
                  {selectedSubmission.admin_review && (
                    <div className="mt-3 p-3 bg-blue-50 rounded">
                      <p className="text-sm">
                        <strong>Previous Review:</strong> {selectedSubmission.admin_review.status}
                      </p>
                      <p className="text-sm text-gray-600">{selectedSubmission.admin_review.comments}</p>
                    </div>
                  )}
                </div>
              </div>
            ) : (
              <div className="text-center text-gray-500 py-8">
                Select a submission to view details
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}