import React, { useState, useEffect } from 'react';
import {
  Shield, CheckCircle, AlertTriangle, Activity, Settings,
  TrendingUp, IndianRupee, Clock, Lock, Key, Zap, Database,
  RefreshCw, Download, Eye, EyeOff, Copy, Check
} from 'lucide-react';

interface GatewayStatus {
  name: string;
  status: 'active' | 'inactive' | 'maintenance';
  health: number;
  last_check: string;
  transactions_today: number;
  success_rate: number;
}

interface SecurityMetrics {
  fraud_alerts_today: number;
  suspicious_transactions: number;
  blocked_transactions: number;
  security_score: number;
  pci_compliance: boolean;
  last_security_audit: string;
}

interface GatewayPerformance {
  gateway: string;
  total_transactions: number;
  successful: number;
  failed: number;
  avg_response_time: number;
  uptime_percentage: number;
}

export function SecurePaymentGateway() {
  const [gateways, setGateways] = useState<GatewayStatus[]>([]);
  const [securityMetrics, setSecurityMetrics] = useState<SecurityMetrics | null>(null);
  const [performance, setPerformance] = useState<GatewayPerformance[]>([]);
  const [loading, setLoading] = useState(true);
  const [showApiKeys, setShowApiKeys] = useState(false);
  const [copiedKey, setCopiedKey] = useState<string | null>(null);

  useEffect(() => {
    fetchGatewayData();
  }, []);

  const fetchGatewayData = async () => {
    try {
      // Mock data - replace with actual API calls
      setGateways([
        {
          name: 'Razorpay',
          status: 'active',
          health: 98,
          last_check: new Date().toISOString(),
          transactions_today: 245,
          success_rate: 97.5
        },
        {
          name: 'Cashfree',
          status: 'active',
          health: 95,
          last_check: new Date().toISOString(),
          transactions_today: 189,
          success_rate: 96.2
        },
        {
          name: 'Paytm',
          status: 'active',
          health: 92,
          last_check: new Date().toISOString(),
          transactions_today: 156,
          success_rate: 94.8
        }
      ]);

      setSecurityMetrics({
        fraud_alerts_today: 0,
        suspicious_transactions: 3,
        blocked_transactions: 12,
        security_score: 98.5,
        pci_compliance: true,
        last_security_audit: '2024-11-15'
      });

      setPerformance([
        {
          gateway: 'Razorpay',
          total_transactions: 1245,
          successful: 1214,
          failed: 31,
          avg_response_time: 1.2,
          uptime_percentage: 99.8
        },
        {
          gateway: 'Cashfree',
          total_transactions: 987,
          successful: 950,
          failed: 37,
          avg_response_time: 1.5,
          uptime_percentage: 99.5
        },
        {
          gateway: 'Paytm',
          total_transactions: 756,
          successful: 717,
          failed: 39,
          avg_response_time: 1.8,
          uptime_percentage: 99.2
        }
      ]);
    } catch (error) {
      console.error('Failed to fetch gateway data:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-700 border-green-200';
      case 'inactive': return 'bg-red-100 text-red-700 border-red-200';
      case 'maintenance': return 'bg-yellow-100 text-yellow-700 border-yellow-200';
      default: return 'bg-gray-100 text-gray-700 border-gray-200';
    }
  };

  const getHealthColor = (health: number) => {
    if (health >= 95) return 'text-green-600';
    if (health >= 85) return 'text-yellow-600';
    return 'text-red-600';
  };

  const copyToClipboard = (text: string, keyName: string) => {
    navigator.clipboard.writeText(text);
    setCopiedKey(keyName);
    setTimeout(() => setCopiedKey(null), 2000);
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h2 className="text-2xl font-bold text-gray-900 flex items-center gap-2">
            <Shield className="w-7 h-7 text-blue-600" />
            Secure Payment Gateway Management
          </h2>
          <p className="text-gray-600 mt-1">Monitor and manage payment gateway integrations and security</p>
        </div>
        <button className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
          <RefreshCw className="w-4 h-4" />
          Refresh Status
        </button>
      </div>

      {/* Gateway Status Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        {gateways.map((gateway) => (
          <div key={gateway.name} className="bg-white p-6 rounded-lg shadow border border-gray-200">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900">{gateway.name}</h3>
              <span className={`px-3 py-1 rounded-full text-xs font-medium border ${getStatusColor(gateway.status)}`}>
                {gateway.status}
              </span>
            </div>

            <div className="space-y-3">
              <div className="flex items-center justify-between">
                <span className="text-sm text-gray-600">Health Score</span>
                <span className={`text-2xl font-bold ${getHealthColor(gateway.health)}`}>
                  {gateway.health}%
                </span>
              </div>

              <div className="flex items-center justify-between">
                <span className="text-sm text-gray-600">Today's Transactions</span>
                <span className="text-lg font-semibold text-gray-900">{gateway.transactions_today}</span>
              </div>

              <div className="flex items-center justify-between">
                <span className="text-sm text-gray-600">Success Rate</span>
                <span className="text-lg font-semibold text-green-600">{gateway.success_rate}%</span>
              </div>

              <div className="pt-3 border-t">
                <div className="flex items-center gap-2 text-xs text-gray-500">
                  <Activity className="w-3 h-3" />
                  Last checked: {new Date(gateway.last_check).toLocaleTimeString()}
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Security Dashboard */}
      {securityMetrics && (
        <div className="bg-white rounded-lg shadow border border-gray-200 p-6">
          <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
            <Lock className="w-5 h-5 text-red-600" />
            Security & Fraud Monitoring
          </h3>

          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
            <div className="p-4 bg-green-50 border border-green-200 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-green-700 font-medium">Security Score</p>
                  <p className="text-3xl font-bold text-green-900 mt-1">{securityMetrics.security_score}%</p>
                </div>
                <Shield className="w-8 h-8 text-green-600" />
              </div>
            </div>

            <div className="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-yellow-700 font-medium">Suspicious Today</p>
                  <p className="text-3xl font-bold text-yellow-900 mt-1">{securityMetrics.suspicious_transactions}</p>
                </div>
                <AlertTriangle className="w-8 h-8 text-yellow-600" />
              </div>
            </div>

            <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-red-700 font-medium">Blocked Today</p>
                  <p className="text-3xl font-bold text-red-900 mt-1">{securityMetrics.blocked_transactions}</p>
                </div>
                <Lock className="w-8 h-8 text-red-600" />
              </div>
            </div>

            <div className="p-4 bg-blue-50 border border-blue-200 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-blue-700 font-medium">PCI Compliance</p>
                  <p className="text-2xl font-bold text-blue-900 mt-1">
                    {securityMetrics.pci_compliance ? 'Certified' : 'Pending'}
                  </p>
                </div>
                <CheckCircle className="w-8 h-8 text-blue-600" />
              </div>
            </div>
          </div>

          <div className="p-4 bg-green-50 border border-green-200 rounded-lg">
            <div className="flex items-center gap-2">
              <CheckCircle className="w-5 h-5 text-green-600" />
              <div>
                <p className="text-sm font-medium text-green-800">All Systems Secure</p>
                <p className="text-xs text-green-700 mt-1">
                  No critical security alerts. Last audit: {new Date(securityMetrics.last_security_audit).toLocaleDateString()}
                </p>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Gateway Configuration */}
      <div className="bg-white rounded-lg shadow border border-gray-200 p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Settings className="w-5 h-5 text-gray-600" />
            Gateway Configuration
          </h3>
          <button
            onClick={() => setShowApiKeys(!showApiKeys)}
            className="flex items-center gap-2 px-3 py-1.5 text-sm border border-gray-300 rounded-lg hover:bg-gray-50"
          >
            {showApiKeys ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
            {showApiKeys ? 'Hide' : 'Show'} API Keys
          </button>
        </div>

        <div className="space-y-4">
          {['Razorpay', 'Cashfree', 'Paytm'].map((gateway) => (
            <div key={gateway} className="p-4 border border-gray-200 rounded-lg">
              <h4 className="font-medium text-gray-900 mb-3">{gateway} Configuration</h4>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">API Key</label>
                  <div className="flex items-center gap-2">
                    <input
                      type={showApiKeys ? 'text' : 'password'}
                      value={showApiKeys ? `${gateway.toLowerCase()}_key_abc123xyz789` : '••••••••••••••••'}
                      readOnly
                      className="flex-1 px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-sm"
                    />
                    {showApiKeys && (
                      <button
                        onClick={() => copyToClipboard(`${gateway.toLowerCase()}_key_abc123xyz789`, `${gateway}-key`)}
                        className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50"
                      >
                        {copiedKey === `${gateway}-key` ? (
                          <Check className="w-4 h-4 text-green-600" />
                        ) : (
                          <Copy className="w-4 h-4 text-gray-600" />
                        )}
                      </button>
                    )}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">API Secret</label>
                  <div className="flex items-center gap-2">
                    <input
                      type={showApiKeys ? 'text' : 'password'}
                      value={showApiKeys ? `${gateway.toLowerCase()}_secret_xyz789abc123` : '••••••••••••••••'}
                      readOnly
                      className="flex-1 px-3 py-2 border border-gray-300 rounded-lg bg-gray-50 text-sm"
                    />
                    {showApiKeys && (
                      <button
                        onClick={() => copyToClipboard(`${gateway.toLowerCase()}_secret_xyz789abc123`, `${gateway}-secret`)}
                        className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50"
                      >
                        {copiedKey === `${gateway}-secret` ? (
                          <Check className="w-4 h-4 text-green-600" />
                        ) : (
                          <Copy className="w-4 h-4 text-gray-600" />
                        )}
                      </button>
                    )}
                  </div>
                </div>
              </div>

              <div className="mt-3 pt-3 border-t">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-gray-600">Webhook URL:</span>
                  <code className="px-2 py-1 bg-gray-100 rounded text-xs">
                    https://api.nalplatform.com/webhooks/{gateway.toLowerCase()}
                  </code>
                </div>
              </div>
            </div>
          ))}
        </div>

        <div className="mt-4 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
          <div className="flex items-start gap-2">
            <AlertTriangle className="w-5 h-5 text-yellow-600 mt-0.5" />
            <div>
              <p className="text-sm font-medium text-yellow-800">Security Notice</p>
              <p className="text-xs text-yellow-700 mt-1">
                API credentials are encrypted and only visible to super-admin users. All configuration changes are logged for audit purposes.
              </p>
            </div>
          </div>
        </div>
      </div>

      {/* Performance Analytics */}
      <div className="bg-white rounded-lg shadow border border-gray-200 p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <TrendingUp className="w-5 h-5 text-blue-600" />
          Gateway Performance Analytics
        </h3>

        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50 border-b">
              <tr>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Gateway</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Total Txns</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Successful</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Failed</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Success Rate</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Avg Response</th>
                <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Uptime</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-gray-200">
              {performance.map((perf) => {
                const successRate = ((perf.successful / perf.total_transactions) * 100).toFixed(1);
                return (
                  <tr key={perf.gateway} className="hover:bg-gray-50">
                    <td className="px-4 py-3 text-sm font-medium text-gray-900">{perf.gateway}</td>
                    <td className="px-4 py-3 text-sm text-gray-600">{perf.total_transactions.toLocaleString()}</td>
                    <td className="px-4 py-3 text-sm text-green-600 font-medium">{perf.successful.toLocaleString()}</td>
                    <td className="px-4 py-3 text-sm text-red-600 font-medium">{perf.failed}</td>
                    <td className="px-4 py-3">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${parseFloat(successRate) >= 95 ? 'bg-green-100 text-green-700' :
                          parseFloat(successRate) >= 90 ? 'bg-yellow-100 text-yellow-700' :
                            'bg-red-100 text-red-700'
                        }`}>
                        {successRate}%
                      </span>
                    </td>
                    <td className="px-4 py-3 text-sm text-gray-600">{perf.avg_response_time}s</td>
                    <td className="px-4 py-3 text-sm font-medium text-green-600">{perf.uptime_percentage}%</td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>
      </div>

      {/* Settlement Tracking */}
      <div className="bg-white rounded-lg shadow border border-gray-200 p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <IndianRupee className="w-5 h-5 text-green-600" />
          Settlement Tracking
        </h3>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
          <div className="p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Pending Settlements</p>
                <p className="text-2xl font-bold text-yellow-600 mt-1">₹8.5L</p>
                <p className="text-xs text-gray-500 mt-1">23 transactions</p>
              </div>
              <Clock className="w-8 h-8 text-yellow-500" />
            </div>
          </div>

          <div className="p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Settled Today</p>
                <p className="text-2xl font-bold text-green-600 mt-1">₹12.3L</p>
                <p className="text-xs text-gray-500 mt-1">45 transactions</p>
              </div>
              <CheckCircle className="w-8 h-8 text-green-500" />
            </div>
          </div>

          <div className="p-4 border border-gray-200 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">This Month</p>
                <p className="text-2xl font-bold text-blue-600 mt-1">₹2.45Cr</p>
                <p className="text-xs text-gray-500 mt-1">1,245 transactions</p>
              </div>
              <TrendingUp className="w-8 h-8 text-blue-500" />
            </div>
          </div>
        </div>

        <div className="flex justify-end">
          <button className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700">
            <Download className="w-4 h-4" />
            Download Settlement Report
          </button>
        </div>
      </div>
    </div>
  );
}
