import React, { useState, useEffect } from "react";
import { AuditDashboard } from './AuditDashboard';
import { SecurityPolicies } from './SecurityPolicies';
import { auditService } from '../../services/auditService';
import { Shield, Clock, Globe, AlertTriangle, Eye, CheckCircle } from 'lucide-react';

export function SecurityAudit() {
  const [activeTab, setActiveTab] = useState('overview');
  const [incidents, setIncidents] = useState([]);
  const [incidentStats, setIncidentStats] = useState(null);
  const [securityData, setSecurityData] = useState({
    healthScore: 95,
    auditLogs: [],
    suspiciousActivities: [],
    securitySettings: {
      twoFactorAuth: true,
      sessionTimeout: 30,
      ipWhitelist: 'partial'
    }
  });
  const [loading, setLoading] = useState(true);
  const [processedActivities, setProcessedActivities] = useState(new Set());

  useEffect(() => {
    fetchSecurityData();
    if (activeTab === 'incidents') {
      fetchIncidents();
    }
  }, [activeTab]);

  const fetchSecurityData = async () => {
    try {
      setLoading(true);
      console.log('🔍 Fetching security data...');
      
      // Fetch security status
      const securityStatus = await auditService.getSecurityStatus();
      
      // Fetch audit logs
      const auditLogs = await auditService.getAuditLogs({ limit: 10 });
      
      // Fetch suspicious activities
      const suspiciousActivities = await auditService.getSuspiciousActivities();
      
      // Fetch security settings
      const securitySettings = await auditService.getSecuritySettings();
      
      setSecurityData({
        healthScore: securityStatus.security_score || 95,
        auditLogs: auditLogs || [],
        suspiciousActivities: suspiciousActivities.alerts || [],
        securitySettings: {
          twoFactorAuth: securitySettings.two_factor_auth?.enabled || true,
          sessionTimeout: securitySettings.session_timeout?.minutes || 30,
          ipWhitelist: securitySettings.ip_whitelist?.status || 'partial'
        }
      });
      
      console.log('✅ Security data updated successfully');
    } catch (error) {
      console.error('❌ Error fetching security data:', error);
      // Keep default values on error
    } finally {
      setLoading(false);
    }
  };

  const getSeverityColor = (severity) => {
    switch (severity) {
      case 'high': return 'red';
      case 'medium': return 'yellow';
      case 'low': return 'orange';
      default: return 'gray';
    }
  };

  const seedAuditData = async () => {
    try {
      setLoading(true);
      await auditService.seedAuditData();
      await fetchSecurityData();
      alert('Audit data seeded successfully!');
    } catch (error) {
      console.error('Error seeding audit data:', error);
      alert('Failed to seed audit data. Please check console for details.');
    } finally {
      setLoading(false);
    }
  };

  const exportAuditLogs = async () => {
    try {
      // Convert to CSV format
      const csvHeaders = ['Log ID', 'User/Admin', 'Action', 'Entity Affected', 'Timestamp', 'IP Address', 'Status'];
      const csvRows = securityData.auditLogs.map(log => [
        log.id,
        log.user_id || 'Anonymous',
        log.operation,
        log.resource_type,
        new Date(log.created_at).toLocaleString(),
        log.ip_address || 'Unknown',
        log.success ? 'Success' : 'Failed'
      ]);
      
      const csvContent = [csvHeaders, ...csvRows]
        .map(row => row.map(field => `"${field}"`).join(','))
        .join('\n');
      
      // Download CSV file
      const blob = new Blob([csvContent], { type: 'text/csv' });
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `audit-logs-${new Date().toISOString().split('T')[0]}.csv`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      window.URL.revokeObjectURL(url);
      
      alert('Audit logs exported successfully!');
    } catch (error) {
      console.error('Error exporting audit logs:', error);
      alert('Failed to export audit logs');
    }
  };

  const exportAuditLogsCSV = async () => {
    try {
      setLoading(true);
      await auditService.exportAuditLogsCSV();
      alert('CSV file downloaded successfully!');
    } catch (error) {
      console.error('Error exporting audit logs to CSV:', error);
      alert('Failed to export audit logs to CSV. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const fetchIncidents = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/admin/security/incidents?limit=100', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      if (!response.ok) {
        console.error('Incidents response:', response.status, await response.text());
        throw new Error('Failed to fetch incidents');
      }
      const data = await response.json();
      console.log('Incidents data:', data);
      setIncidents(Array.isArray(data) ? data : []);
      
      const statsResponse = await fetch('http://localhost:8090/api/admin/security/incidents/stats', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      if (statsResponse.ok) {
        const statsData = await statsResponse.json();
        console.log('Stats data:', statsData);
        setIncidentStats(statsData);
      }
    } catch (error) {
      console.error('Error fetching incidents:', error);
      setIncidents([]);
    }
  };

  const resolveIncident = async (incidentId) => {
    try {
      const token = localStorage.getItem('access_token');
      await fetch(`http://localhost:8090/api/admin/security/incidents/${incidentId}/resolve`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ resolution_notes: 'Resolved by admin' })
      });
      fetchIncidents();
    } catch (error) {
      console.error('Error resolving incident:', error);
    }
  };

  const handleActivityAction = (index, action) => {
    const activityId = `activity-${index}`;
    setProcessedActivities(prev => new Set([...prev, activityId]));
    
    // Show success message
    alert(`${action} completed successfully for suspicious activity.`);
    
    // Remove from list after 2 seconds
    setTimeout(() => {
      setSecurityData(prev => ({
        ...prev,
        suspiciousActivities: prev.suspiciousActivities.filter((_, i) => i !== index)
      }));
      setProcessedActivities(prev => {
        const newSet = new Set(prev);
        newSet.delete(activityId);
        return newSet;
      });
    }, 2000);
  };

  if (loading) {
    return (
      <div className="space-y-6">
        <h1 className="text-2xl font-bold text-gray-900">Security & Audit</h1>
        <div className="flex items-center justify-center h-64">
          <div className="text-gray-500">Loading security data...</div>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-8 p-6" style={{ scrollBehavior: 'auto' }}>
      <div className="flex items-center gap-4 mb-8">
        <div className="bg-gradient-to-r from-red-600 to-pink-600 p-4 rounded-2xl shadow-lg">
          <Shield className="w-10 h-10 text-white" />
        </div>
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Security &amp; Audit</h1>
          <p className="text-gray-600">Monitor system security and audit activities</p>
        </div>
      </div>
      
      {/* Tab Navigation */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="flex border-b border-gray-200">
          <button
            onClick={(e) => {
              e.preventDefault();
              setActiveTab('overview');
            }}
            className={`px-6 py-3 text-sm font-medium border-b-2 transition-colors ${
              activeTab === 'overview'
                ? 'border-blue-500 text-blue-600 bg-blue-50'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
            }`}
          >
            Security Overview
          </button>
          <button
            onClick={(e) => {
              e.preventDefault();
              setActiveTab('incidents');
            }}
            className={`px-6 py-3 text-sm font-medium border-b-2 transition-colors ${
              activeTab === 'incidents'
                ? 'border-blue-500 text-blue-600 bg-blue-50'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
            }`}
          >
            Security Incidents
          </button>
          <button
            onClick={(e) => {
              e.preventDefault();
              setActiveTab('audit-logs');
            }}
            className={`px-6 py-3 text-sm font-medium border-b-2 transition-colors ${
              activeTab === 'audit-logs'
                ? 'border-blue-500 text-blue-600 bg-blue-50'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
            }`}
          >
            Audit Logs
          </button>
          <button
            onClick={(e) => {
              e.preventDefault();
              setActiveTab('policies');
            }}
            className={`px-6 py-3 text-sm font-medium border-b-2 transition-colors ${
              activeTab === 'policies'
                ? 'border-blue-500 text-blue-600 bg-blue-50'
                : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
            }`}
          >
            Security Policies
          </button>
        </div>
      </div>
      
      {/* Tab Content */}
      {activeTab === 'overview' && (
        <div className="space-y-6 mt-6">
      {/* Security Health Score */}
      <div className="bg-gradient-to-br from-white to-blue-50 rounded-2xl shadow-xl border border-blue-100 overflow-hidden mb-6">

        <div className="p-8">
          <div className="text-center">
            <div className="w-40 h-40 mx-auto mb-6 relative">
              <div className="w-full h-full rounded-full bg-gradient-to-r from-green-400 to-emerald-500 p-2 shadow-2xl">
                <div className="w-full h-full rounded-full bg-white flex items-center justify-center">
                  <span className="text-4xl font-bold text-green-600">{securityData.healthScore}%</span>
                </div>
              </div>
              <div className="absolute -top-2 -right-2 w-8 h-8 bg-gradient-to-r from-yellow-400 to-orange-400 rounded-full flex items-center justify-center">
              </div>
            </div>
            <p className="text-xl font-semibold text-green-600">
              {securityData.healthScore >= 95 ? 'Excellent' : 
               securityData.healthScore >= 85 ? 'Good' : 
               securityData.healthScore >= 70 ? 'Fair' : 'Poor'} Security Status
            </p>
            <p className="text-gray-600 mt-2">Your system is highly secure and well-protected</p>
          </div>
        </div>
      </div>

      {/* Suspicious Activity Alerts */}
      <div className="bg-gradient-to-br from-white to-red-50 rounded-2xl shadow-xl border border-red-100 overflow-hidden mb-6">
        <div className="bg-gradient-to-r from-red-600 to-pink-600 p-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <h2 className="text-xl font-bold text-black">Suspicious Activity Alerts</h2>
            </div>
            <div className="bg-white/20 text-white px-3 py-1 rounded-full text-sm font-medium">
              {securityData.suspiciousActivities.length} Active
            </div>
          </div>
        </div>
        <div className="p-6">
          <div className="space-y-4">
            {securityData.suspiciousActivities.length === 0 ? (
              <div className="text-center py-8">
                <div className="flex flex-col items-center gap-2">
                  <p className="text-gray-500">No suspicious activities detected</p>
                  <p className="text-sm text-gray-400">System is monitoring for unusual patterns</p>
                </div>
              </div>
            ) : (
              securityData.suspiciousActivities.map((activity, index) => {
                const colorMap = {
                  high: { bg: 'from-red-50 to-pink-50', border: 'border-red-200', dot: 'bg-red-500', text: 'text-red-800', subtext: 'text-red-600', btn: 'bg-red-100 text-red-800 hover:bg-red-200' },
                  medium: { bg: 'from-yellow-50 to-amber-50', border: 'border-yellow-200', dot: 'bg-yellow-500', text: 'text-yellow-800', subtext: 'text-yellow-600', btn: 'bg-yellow-100 text-yellow-800 hover:bg-yellow-200' },
                  low: { bg: 'from-orange-50 to-red-50', border: 'border-orange-200', dot: 'bg-orange-500', text: 'text-orange-800', subtext: 'text-orange-600', btn: 'bg-orange-100 text-orange-800 hover:bg-orange-200' }
                };
                const style = colorMap[activity.severity];
                return (
                  <div key={index} className={`flex items-center gap-4 p-4 bg-gradient-to-r ${style.bg} rounded-xl border ${style.border} shadow-lg hover:shadow-xl transition-all duration-300`}>
                    <div className="flex items-center gap-3">
                      <div className={`w-3 h-3 ${style.dot} rounded-full animate-pulse`}></div>
                    </div>
                    <div className="flex-1">
                      <p className={`text-sm font-bold ${style.text}`}>{activity.type}</p>
                      <p className={`text-xs ${style.subtext} mt-1`}>{activity.description}</p>
                    </div>
                    <button 
                      onClick={() => handleActivityAction(index, activity.action)}
                      disabled={processedActivities.has(`activity-${index}`)}
                      className={`${style.btn} px-4 py-2 rounded-lg text-sm font-medium transition-all duration-200 shadow-md hover:shadow-lg disabled:opacity-50 disabled:cursor-not-allowed`}
                    >
                      {processedActivities.has(`activity-${index}`) ? 'Processing...' : activity.action}
                    </button>
                  </div>
                );
              })
            )}
          </div>
        </div>
      </div>

      {/* Security Settings */}
      <div className="bg-white rounded-2xl shadow-lg border border-gray-200">
        <div className="p-6 border-b border-gray-200">
          <h2 className="text-xl font-bold text-gray-900">Security Settings</h2>
        </div>
        <div className="p-6">
          <div className="space-y-4">
            <div className="flex items-center justify-between p-4 bg-blue-50 border border-blue-200 rounded-xl">
              <div className="flex items-center gap-3">
                <div className="w-12 h-12 bg-blue-500 rounded-xl flex items-center justify-center">
                  <Clock className="w-6 h-6 text-white" />
                </div>
                <div>
                  <h3 className="font-bold text-gray-900">Session Timeout</h3>
                  <p className="text-sm text-gray-600">Auto-logout after inactivity</p>
                </div>
              </div>
              <div className="flex items-center gap-3">
                <span className="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-xs font-bold">30 minutes</span>
                <button className="bg-blue-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-600 transition-all">
                  Configure
                </button>
              </div>
            </div>
            <div className="flex items-center justify-between p-4 bg-yellow-50 border border-yellow-200 rounded-xl">
              <div className="flex items-center gap-3">
                <div className="w-12 h-12 bg-yellow-500 rounded-xl flex items-center justify-center">
                  <Globe className="w-6 h-6 text-white" />
                </div>
                <div>
                  <h3 className="font-bold text-gray-900">IP Whitelist</h3>
                  <p className="text-sm text-gray-600">Restrict admin access to specific IPs</p>
                </div>
              </div>
              <div className="flex items-center gap-3">
                <span className="bg-yellow-100 text-yellow-800 px-3 py-1 rounded-full text-xs font-bold">Partial</span>
                <button className="bg-blue-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-600 transition-all">
                  Configure
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>
        </div>
      )}
      
      {activeTab === 'incidents' && (
        <div className="space-y-6 mt-6">
          {/* Incident Stats */}
          {incidentStats && (
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div className="bg-white p-6 rounded-lg shadow border border-gray-200">
                <div className="text-sm text-gray-600">Total Incidents</div>
                <div className="text-2xl font-bold text-gray-900">{incidentStats.total_incidents}</div>
              </div>
              <div className="bg-red-50 p-6 rounded-lg shadow border border-red-200">
                <div className="text-sm text-red-600">Active</div>
                <div className="text-2xl font-bold text-red-900">{incidentStats.active_incidents}</div>
              </div>
              <div className="bg-green-50 p-6 rounded-lg shadow border border-green-200">
                <div className="text-sm text-green-600">Resolved</div>
                <div className="text-2xl font-bold text-green-900">{incidentStats.resolved_incidents}</div>
              </div>
              <div className="bg-orange-50 p-6 rounded-lg shadow border border-orange-200">
                <div className="text-sm text-orange-600">Last 24h</div>
                <div className="text-2xl font-bold text-orange-900">{incidentStats.last_24h}</div>
              </div>
            </div>
          )}

          {/* Incidents List */}
          <div className="bg-white rounded-lg shadow border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-xl font-bold text-gray-900">Security Incidents</h2>
            </div>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Severity</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Description</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">IP Address</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                  </tr>
                </thead>
                <tbody className="divide-y divide-gray-200">
                  {incidents.length === 0 ? (
                    <tr>
                      <td colSpan="7" className="px-6 py-8 text-center text-gray-500">
                        No security incidents found
                      </td>
                    </tr>
                  ) : (
                    incidents.map((incident) => (
                      <tr key={incident.id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="flex items-center gap-2">
                            <AlertTriangle className="w-4 h-4 text-orange-500" />
                            <span className="text-sm font-medium text-gray-900">
                              {incident.incident_type.replace('_', ' ').toUpperCase()}
                            </span>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={`px-2 py-1 text-xs font-semibold rounded-full ${
                            incident.severity === 'critical' ? 'bg-red-100 text-red-800' :
                            incident.severity === 'high' ? 'bg-orange-100 text-orange-800' :
                            incident.severity === 'medium' ? 'bg-yellow-100 text-yellow-800' :
                            'bg-blue-100 text-blue-800'
                          }`}>
                            {incident.severity.toUpperCase()}
                          </span>
                        </td>
                        <td className="px-6 py-4">
                          <div className="text-sm text-gray-900">{incident.title}</div>
                          <div className="text-xs text-gray-500">{incident.description}</div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                          {incident.ip_address || 'N/A'}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={`px-2 py-1 text-xs font-semibold rounded-full ${
                            incident.status === 'resolved' ? 'bg-green-100 text-green-800' :
                            'bg-red-100 text-red-800'
                          }`}>
                            {incident.status.toUpperCase()}
                          </span>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                          {new Date(incident.created_at).toLocaleString()}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm">
                          {incident.status === 'detected' && (
                            <button
                              onClick={() => resolveIncident(incident.id)}
                              className="text-green-600 hover:text-green-900 font-medium flex items-center gap-1"
                            >
                              <CheckCircle className="w-4 h-4" />
                              Resolve
                            </button>
                          )}
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>
          </div>
        </div>
      )}
      
      {activeTab === 'audit-logs' && (
        <AuditDashboard />
      )}
      
      {activeTab === 'policies' && (
        <SecurityPolicies />
      )}
    </div>
  );
}