import React, { useState, useEffect } from "react";
import { auditService } from '../../services/auditService';

export function SecurityAuditSimple() {
  const [auditLogs, setAuditLogs] = useState([]);
  const [loading, setLoading] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalRecords, setTotalRecords] = useState(0);
  const recordsPerPage = 25;

  const fetchAuditLogs = async (page = 1) => {
    try {
      setLoading(true);
      const offset = (page - 1) * recordsPerPage;
      const logs = await auditService.getAuditLogs({ limit: recordsPerPage, offset });
      setAuditLogs(Array.isArray(logs) ? logs : []);
      
      // Get total count for pagination
      if (logs.length > 0) {
        setTotalRecords(logs.length === recordsPerPage ? page * recordsPerPage + 1 : (page - 1) * recordsPerPage + logs.length);
      }
    } catch (error) {
      console.error('Error fetching audit logs:', error);
      setAuditLogs([]);
    } finally {
      setLoading(false);
    }
  };

  const handleNextPage = () => {
    if (auditLogs.length === recordsPerPage) {
      const nextPage = currentPage + 1;
      setCurrentPage(nextPage);
      fetchAuditLogs(nextPage);
    }
  };

  const handlePrevPage = () => {
    if (currentPage > 1) {
      const prevPage = currentPage - 1;
      setCurrentPage(prevPage);
      fetchAuditLogs(prevPage);
    }
  };

  const exportCSV = async () => {
    try {
      const allLogs = await auditService.getAuditLogs({ limit: 1000 });
      const csvHeaders = ['ID', 'User ID', 'Operation', 'Resource Type', 'IP Address', 'Status', 'Timestamp'];
      const csvRows = allLogs.map(log => [
        log.id,
        log.user_id || 'Anonymous',
        log.operation,
        log.resource_type,
        log.ip_address || '',
        log.success ? 'Success' : 'Failed',
        new Date(log.created_at).toISOString()
      ]);
      
      const csvContent = [csvHeaders, ...csvRows]
        .map(row => row.map(field => `"${field}"`).join(','))
        .join('\n');
      
      const blob = new Blob([csvContent], { type: 'text/csv' });
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `audit-logs-${new Date().toISOString().split('T')[0]}.csv`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      window.URL.revokeObjectURL(url);
    } catch (error) {
      console.error('Error exporting CSV:', error);
      alert('Failed to export CSV');
    }
  };

  const exportExcel = async () => {
    try {
      setLoading(true);
      await auditService.exportAuditLogsExcel({ limit: 10000 });
      alert('Excel file downloaded successfully!');
    } catch (error) {
      console.error('Error exporting Excel:', error);
      alert('Failed to export Excel file');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchAuditLogs();
  }, []);

  return (
    <div className="p-6">
      <h1 className="text-3xl font-bold mb-6 text-black">Security & Audit</h1>
      
      {/* Audit Logs Section */}
      <div className="bg-white rounded-lg shadow-lg border border-gray-300">
        {/* Header with buttons - CLEAN */}
        <div className="bg-white p-6 border-b border-gray-200">
          <div className="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-4">
            <div>
              <h2 className="text-2xl font-bold text-gray-800 mb-2">📋 Audit Logs</h2>
              <p className="text-gray-600 text-lg">{auditLogs.length} Records</p>
            </div>
            <div className="flex flex-wrap gap-3">
              <button 
                onClick={fetchAuditLogs}
                disabled={loading}
                className="bg-gray-200 hover:bg-gray-300 px-4 py-2 rounded-lg text-black font-medium disabled:opacity-50 min-w-[100px] border border-gray-400"
              >
                {loading ? 'Loading...' : 'Refresh'}
              </button>
              <button 
                onClick={exportCSV}
                disabled={loading}
                className="bg-gray-200 hover:bg-gray-300 px-4 py-2 rounded-lg text-black font-medium disabled:opacity-50 min-w-[120px] border border-gray-400"
              >
                {loading ? 'Loading...' : 'Export CSV'}
              </button>
            </div>
          </div>
        </div>

        {/* Table */}
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-100 border-b-2 border-gray-200">
              <tr>
                <th className="px-6 py-4 text-left text-sm font-bold text-gray-700 uppercase">Log ID</th>
                <th className="px-6 py-4 text-left text-sm font-bold text-gray-700 uppercase">User/Admin</th>
                <th className="px-6 py-4 text-left text-sm font-bold text-gray-700 uppercase">Action</th>
                <th className="px-6 py-4 text-left text-sm font-bold text-gray-700 uppercase">Entity Affected</th>
                <th className="px-6 py-4 text-left text-sm font-bold text-gray-700 uppercase">Timestamp</th>
                <th className="px-6 py-4 text-left text-sm font-bold text-gray-700 uppercase">IP Address</th>
                <th className="px-6 py-4 text-left text-sm font-bold text-gray-700 uppercase">Status</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-gray-200">
              {auditLogs.length === 0 ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center text-gray-500">
                    <div className="flex flex-col items-center gap-4">
                      <div className="text-4xl">📋</div>
                      <p className="text-lg font-medium">No audit logs available</p>
                      <p className="text-sm text-gray-400">Audit logs will appear here as users interact with the system</p>
                    </div>
                  </td>
                </tr>
              ) : (
                auditLogs.map((log) => (
                  <tr key={log.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4 text-sm font-medium">#{log.id}</td>
                    <td className="px-6 py-4 text-sm">
                      {log.user_id ? `User ${log.user_id.toString().slice(0, 8)}...` : 'Anonymous'}
                    </td>
                    <td className="px-6 py-4 text-sm">
                      <span className="bg-yellow-100 text-yellow-800 px-3 py-1 rounded-full text-xs font-medium">
                        {log.operation}
                      </span>
                    </td>
                    <td className="px-6 py-4 text-sm">{log.resource_type}</td>
                    <td className="px-6 py-4 text-sm">
                      {new Date(log.created_at).toLocaleString()}
                    </td>
                    <td className="px-6 py-4 text-sm font-mono">{log.ip_address || 'Unknown'}</td>
                    <td className="px-6 py-4 text-sm">
                      <span className={`px-3 py-1 rounded-full text-xs font-medium ${
                        log.success 
                          ? 'bg-green-100 text-green-800' 
                          : 'bg-red-100 text-red-800'
                      }`}>
                        {log.success ? 'Success' : 'Failed'}
                      </span>
                    </td>
                  </tr>
                ))
              )}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        <div className="bg-gray-50 px-6 py-4 border-t border-gray-200">
          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-700">
              Showing page {currentPage} ({auditLogs.length} records)
            </div>
            <div className="flex items-center gap-2">
              <button
                onClick={handlePrevPage}
                disabled={currentPage === 1 || loading}
                className="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                &lt;
              </button>
              
              {/* Page numbers */}
              <div className="flex gap-1">
                {[...Array(Math.min(5, Math.max(1, Math.ceil(totalRecords / recordsPerPage))))].map((_, index) => {
                  const pageNum = Math.max(1, currentPage - 2) + index;
                  if (pageNum <= Math.ceil(totalRecords / recordsPerPage)) {
                    return (
                      <button
                        key={pageNum}
                        onClick={() => {
                          setCurrentPage(pageNum);
                          fetchAuditLogs(pageNum);
                        }}
                        disabled={loading}
                        className={`w-8 h-8 text-sm font-medium rounded-full border ${
                          pageNum === currentPage
                            ? 'bg-blue-600 text-white border-blue-600'
                            : 'bg-white text-gray-700 border-gray-300 hover:bg-gray-50'
                        } disabled:opacity-50 disabled:cursor-not-allowed`}
                      >
                        {pageNum}
                      </button>
                    );
                  }
                  return null;
                })}
              </div>
              
              <button
                onClick={handleNextPage}
                disabled={auditLogs.length < recordsPerPage || loading}
                className="px-3 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                &gt;
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}