import React, { useState, useEffect } from 'react';
import { Shield, Plus, Edit2, Trash2, Power, Check, X, Lock, Clock, Key, AlertTriangle } from 'lucide-react';

export function SecurityPolicies() {
  const [policies, setPolicies] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showModal, setShowModal] = useState(false);
  const [editingPolicy, setEditingPolicy] = useState(null);
  const [formData, setFormData] = useState({
    policy_name: '',
    policy_type: '',
    policy_value: '',
    description: '',
    severity: 'medium'
  });

  useEffect(() => {
    fetchPolicies();
  }, []);

  const fetchPolicies = async () => {
    try {
      const token = localStorage.getItem('access_token');
      console.log('Fetching policies with token:', token ? 'Token exists' : 'No token');
      
      const response = await fetch('http://localhost:8090/api/admin/security/policies', {
        headers: { 'Authorization': `Bearer ${token}` }
      });
      
      console.log('Response status:', response.status);
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('API Error:', errorText);
        throw new Error(`API returned ${response.status}`);
      }
      
      const data = await response.json();
      console.log('Policies data:', data);
      
      setPolicies(data.policies || []);
    } catch (error) {
      console.error('Error fetching policies:', error);
      // Show error to user
      alert('Failed to load security policies. Please check console for details.');
    } finally {
      setLoading(false);
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      const token = localStorage.getItem('access_token');
      const url = editingPolicy 
        ? `http://localhost:8090/api/admin/security/policies/${editingPolicy.id}`
        : 'http://localhost:8090/api/admin/security/policies';
      
      await fetch(url, {
        method: editingPolicy ? 'PUT' : 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(formData)
      });
      
      setShowModal(false);
      setEditingPolicy(null);
      setFormData({ policy_name: '', policy_type: '', policy_value: '', description: '', severity: 'medium' });
      fetchPolicies();
    } catch (error) {
      console.error('Error saving policy:', error);
    }
  };

  const togglePolicy = async (policyId, isActive) => {
    try {
      const token = localStorage.getItem('access_token');
      await fetch(`http://localhost:8090/api/admin/security/policies/${policyId}/toggle?is_active=${!isActive}`, {
        method: 'PATCH',
        headers: { 'Authorization': `Bearer ${token}` }
      });
      fetchPolicies();
    } catch (error) {
      console.error('Error toggling policy:', error);
    }
  };

  const deletePolicy = async (policyId) => {
    if (!confirm('Are you sure you want to delete this policy?')) return;
    try {
      const token = localStorage.getItem('access_token');
      await fetch(`http://localhost:8090/api/admin/security/policies/${policyId}`, {
        method: 'DELETE',
        headers: { 'Authorization': `Bearer ${token}` }
      });
      fetchPolicies();
    } catch (error) {
      console.error('Error deleting policy:', error);
    }
  };

  const openEditModal = (policy) => {
    setEditingPolicy(policy);
    setFormData({
      policy_name: policy.policy_name,
      policy_type: policy.policy_type,
      policy_value: policy.policy_value,
      description: policy.description,
      severity: policy.severity
    });
    setShowModal(true);
  };

  const getSeverityColor = (severity) => {
    switch (severity) {
      case 'critical': return 'bg-red-50 text-red-700 border-red-300';
      case 'high': return 'bg-orange-50 text-orange-700 border-orange-300';
      case 'medium': return 'bg-yellow-50 text-yellow-700 border-yellow-300';
      case 'low': return 'bg-blue-50 text-blue-700 border-blue-300';
      default: return 'bg-gray-50 text-gray-700 border-gray-300';
    }
  };

  const getPolicyIcon = (policyType) => {
    if (policyType.includes('password')) return <Key className="w-5 h-5" />;
    if (policyType.includes('login') || policyType.includes('lockout')) return <Lock className="w-5 h-5" />;
    if (policyType.includes('session') || policyType.includes('timeout')) return <Clock className="w-5 h-5" />;
    return <Shield className="w-5 h-5" />;
  };

  if (loading) {
    return <div className="flex items-center justify-center h-64"><div className="text-gray-500">Loading policies...</div></div>;
  }

  return (
    <div className="space-y-6 p-6">
      {/* Header Section */}
      <div className="bg-white rounded-2xl shadow-xl p-8 border border-gray-200">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <div className="bg-blue-100 p-4 rounded-xl">
              <Shield className="w-8 h-8 text-blue-600" />
            </div>
            <div>
              <h2 className="text-3xl font-bold mb-2 text-black">Security Configuration & Policies</h2>
              <p className="text-lg text-black">Manage system security policies and configurations</p>
            </div>
          </div>
          <button
            onClick={() => {
              setEditingPolicy(null);
              setFormData({ policy_name: '', policy_type: '', policy_value: '', description: '', severity: 'medium' });
              setShowModal(true);
            }}
            className="flex items-center gap-2 bg-white text-blue-600 px-6 py-3 rounded-xl hover:bg-blue-50 transition-all shadow-lg hover:shadow-xl font-semibold"
          >
            <Plus className="w-5 h-5" />
            Add Policy
          </button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-red-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-600 text-sm font-medium">Critical</p>
              <p className="text-3xl font-bold text-gray-900">{policies.filter(p => p.severity === 'critical').length}</p>
            </div>
            <AlertTriangle className="w-10 h-10 text-red-500" />
          </div>
        </div>
        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-orange-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-600 text-sm font-medium">High</p>
              <p className="text-3xl font-bold text-gray-900">{policies.filter(p => p.severity === 'high').length}</p>
            </div>
            <Shield className="w-10 h-10 text-orange-500" />
          </div>
        </div>
        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-yellow-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-600 text-sm font-medium">Medium</p>
              <p className="text-3xl font-bold text-gray-900">{policies.filter(p => p.severity === 'medium').length}</p>
            </div>
            <Lock className="w-10 h-10 text-yellow-500" />
          </div>
        </div>
        <div className="bg-white rounded-xl shadow-md p-6 border-l-4 border-green-500">
          <div className="flex items-center justify-between">
            <div>
              <p className="text-gray-600 text-sm font-medium">Active</p>
              <p className="text-3xl font-bold text-gray-900">{policies.filter(p => p.is_active).length}</p>
            </div>
            <Check className="w-10 h-10 text-green-500" />
          </div>
        </div>
      </div>

      {policies.length === 0 ? (
        <div className="bg-white rounded-2xl shadow-lg border-2 border-dashed border-gray-300 p-16 text-center">
          <div className="bg-gray-100 w-24 h-24 rounded-full flex items-center justify-center mx-auto mb-6">
            <Shield className="w-12 h-12 text-gray-400" />
          </div>
          <h3 className="text-2xl font-bold text-gray-900 mb-3">No Security Policies Found</h3>
          <p className="text-gray-600 mb-8 max-w-md mx-auto">Get started by creating your first security policy to protect your system and enforce security standards.</p>
          <button
            onClick={() => {
              setEditingPolicy(null);
              setFormData({ policy_name: '', policy_type: '', policy_value: '', description: '', severity: 'medium' });
              setShowModal(true);
            }}
            className="inline-flex items-center gap-2 bg-gradient-to-r from-blue-600 to-indigo-600 text-white px-8 py-4 rounded-xl hover:from-blue-700 hover:to-indigo-700 transition-all shadow-lg hover:shadow-xl font-semibold text-lg"
          >
            <Plus className="w-6 h-6" />
            Create First Policy
          </button>
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {policies.map((policy) => (
          <div key={policy.id} className="bg-gradient-to-br from-white to-gray-50 rounded-2xl shadow-xl border-2 border-gray-200 overflow-hidden hover:shadow-2xl hover:border-blue-300 transition-all duration-300">
            <div className="p-6 border-b-2 border-gray-200">
              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center gap-3">
                  <div className={`p-3 rounded-xl ${
                    policy.severity === 'critical' ? 'bg-red-100' :
                    policy.severity === 'high' ? 'bg-orange-100' :
                    policy.severity === 'medium' ? 'bg-yellow-100' :
                    'bg-blue-100'
                  }`}>
                    <div className={`${
                      policy.severity === 'critical' ? 'text-red-600' :
                      policy.severity === 'high' ? 'text-orange-600' :
                      policy.severity === 'medium' ? 'text-yellow-600' :
                      'text-blue-600'
                    }`}>
                      {getPolicyIcon(policy.policy_type)}
                    </div>
                  </div>
                  <div>
                    <h3 className="font-bold text-gray-900 text-lg">{policy.policy_name}</h3>
                    <span className={`inline-block mt-1 px-2.5 py-1 text-xs font-bold rounded-full ${
                      policy.severity === 'critical' ? 'bg-red-100 text-red-700' :
                      policy.severity === 'high' ? 'bg-orange-100 text-orange-700' :
                      policy.severity === 'medium' ? 'bg-yellow-100 text-yellow-700' :
                      'bg-blue-100 text-blue-700'
                    }`}>
                      {policy.severity.toUpperCase()}
                    </span>
                  </div>
                </div>
              </div>
            </div>
            <div className="p-6 space-y-4">
              <div className="bg-white rounded-xl p-4 border border-gray-200 shadow-sm">
                <div className="text-xs text-gray-500 uppercase font-semibold mb-1">Policy Type</div>
                <div className="text-sm font-semibold text-gray-900">{policy.policy_type.replace(/_/g, ' ')}</div>
              </div>
              <div className="bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl p-4 border-2 border-blue-200 shadow-sm">
                <div className="text-xs text-blue-600 uppercase font-bold mb-2">Value</div>
                <div className="text-4xl font-bold text-blue-700">{policy.policy_value}</div>
              </div>
              <div className="bg-white rounded-xl p-4 border border-gray-200 shadow-sm">
                <div className="text-xs text-gray-500 uppercase font-semibold mb-2">Description</div>
                <div className="text-sm text-gray-700 leading-relaxed">{policy.description}</div>
              </div>
              <div className="flex items-center justify-between pt-4 border-t-2 border-gray-200">
                <div className="flex items-center gap-2">
                  <button
                    onClick={() => togglePolicy(policy.id, policy.is_active)}
                    className={`p-2.5 rounded-xl transition-all shadow-md hover:shadow-lg ${
                      policy.is_active 
                        ? 'bg-green-500 text-white hover:bg-green-600' 
                        : 'bg-gray-300 text-gray-600 hover:bg-gray-400'
                    }`}
                    title={policy.is_active ? 'Disable' : 'Enable'}
                  >
                    <Power className="w-4 h-4" />
                  </button>
                  <button
                    onClick={() => openEditModal(policy)}
                    className="p-2.5 rounded-xl bg-blue-500 text-white hover:bg-blue-600 transition-all shadow-md hover:shadow-lg"
                    title="Edit"
                  >
                    <Edit2 className="w-4 h-4" />
                  </button>
                  <button
                    onClick={() => deletePolicy(policy.id)}
                    className="p-2.5 rounded-xl bg-red-500 text-white hover:bg-red-600 transition-all shadow-md hover:shadow-lg"
                    title="Delete"
                  >
                    <Trash2 className="w-4 h-4" />
                  </button>
                </div>
                <div className="flex items-center gap-2 bg-white px-3 py-1.5 rounded-full border-2 border-gray-200 shadow-sm">
                  <div className={`w-2.5 h-2.5 rounded-full ${policy.is_active ? 'bg-green-500 animate-pulse' : 'bg-gray-400'}`}></div>
                  <span className={`text-xs font-bold ${policy.is_active ? 'text-green-600' : 'text-gray-500'}`}>
                    {policy.is_active ? 'Active' : 'Inactive'}
                  </span>
                </div>
              </div>
            </div>
          </div>
        ))}
        </div>
      )}

      {/* Modal */}
      {showModal && (
        <div className="fixed inset-0 bg-black/60 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-2xl shadow-2xl w-full max-w-2xl">
            <div className="bg-gradient-to-r from-blue-600 to-indigo-600 p-6 rounded-t-2xl">
              <h3 className="text-2xl font-bold text-white flex items-center gap-3">
                <Shield className="w-7 h-7" />
                {editingPolicy ? 'Edit Policy' : 'Add New Policy'}
              </h3>
            </div>
            <form onSubmit={handleSubmit} className="p-6 space-y-5">
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Policy Name</label>
                <input
                  type="text"
                  value={formData.policy_name}
                  onChange={(e) => setFormData({ ...formData, policy_name: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all text-gray-900"
                  placeholder="e.g., Password Expiry"
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Policy Type</label>
                <input
                  type="text"
                  value={formData.policy_type}
                  onChange={(e) => setFormData({ ...formData, policy_type: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all text-gray-900"
                  placeholder="e.g., password_expiry, login_attempts"
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Policy Value</label>
                <input
                  type="text"
                  value={formData.policy_value}
                  onChange={(e) => setFormData({ ...formData, policy_value: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all text-gray-900"
                  placeholder="e.g., 90, 8, 5"
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Description</label>
                <textarea
                  value={formData.description}
                  onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all text-gray-900"
                  rows="3"
                  placeholder="Describe the policy..."
                  required
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">Severity</label>
                <select
                  value={formData.severity}
                  onChange={(e) => setFormData({ ...formData, severity: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-300 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all text-gray-900 bg-white"
                >
                  <option value="low">Low</option>
                  <option value="medium">Medium</option>
                  <option value="high">High</option>
                  <option value="critical">Critical</option>
                </select>
              </div>
              <div className="flex gap-3 pt-4">
                <button
                  type="submit"
                  className="flex-1 flex items-center justify-center gap-2 bg-gradient-to-r from-blue-600 to-indigo-600 text-white px-6 py-3 rounded-xl hover:from-blue-700 hover:to-indigo-700 transition-all font-semibold shadow-lg hover:shadow-xl"
                >
                  <Check className="w-5 h-5" />
                  {editingPolicy ? 'Update' : 'Create'}
                </button>
                <button
                  type="button"
                  onClick={() => {
                    setShowModal(false);
                    setEditingPolicy(null);
                  }}
                  className="flex-1 flex items-center justify-center gap-2 bg-gray-200 text-gray-700 px-6 py-3 rounded-xl hover:bg-gray-300 transition-all font-semibold"
                >
                  <X className="w-5 h-5" />
                  Cancel
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
}
