import React, { useState, useEffect, useCallback, memo } from "react";
import logoImage from "../../assets/logo.jpg";
import { useNavigate, useLocation } from "react-router-dom";
import { collaborationApi } from "../../api/collaborations";
import { getTenantAwareHref } from "../../utils/tenantNavigation";
import {
  Home,
  Building,
  Users,
  Calendar,
  BarChart3,
  Settings,
  TrendingUp,
  LogOut,
  ChevronLeft,
  ChevronRight,
  FileText,
  Heart,
  Bookmark,
  History,
  MessageSquare,
  GitCompare,
  User,
  LifeBuoy,
  IndianRupee,
  UserPlus,
  Activity,
  PieChart,
  LineChart,
  MapPin,
  Globe,
  Shield,
  Megaphone,
  CheckSquare,
  Flag,
  TrendingDown,
  DollarSign,
  FileCheck,
  Monitor,
  Database,
  Package,
  AlertCircle,
  Wallet,
  Tag,
  Percent,
  GraduationCap,
  Key,
  Webhook,
  ShoppingBag,
  Link2,
  ShieldCheck,
} from "lucide-react";

interface SidebarProps {
  collapsed?: boolean;
  onToggle?: (collapsed: boolean) => void;
  isSidebarCollapsed?: boolean;
  setIsSidebarCollapsed?: (collapsed: boolean) => void;
}

const Sidebar: React.FC<SidebarProps> = ({
  collapsed = false,
  onToggle,
  isSidebarCollapsed = false,
  setIsSidebarCollapsed,
}) => {
  const [isCollapsed, setIsCollapsed] = useState(
    collapsed || isSidebarCollapsed
  );
  const [invitationCount, setInvitationCount] = useState(0);
  const [themeColor, setThemeColor] = useState('#0056D2');
  const navigate = useNavigate();
  const location = useLocation();

  React.useEffect(() => {
    setIsCollapsed(collapsed || isSidebarCollapsed);
  }, [collapsed, isSidebarCollapsed]);

  // Load theme color from sessionStorage
  useEffect(() => {
    const updateThemeColor = () => {
      const savedColor = sessionStorage.getItem('tenant_theme_color');
      console.log('Sidebar: Loading theme color from sessionStorage:', savedColor);
      if (savedColor) {
        setThemeColor(savedColor);
        console.log('Sidebar: Theme color set to:', savedColor);
      } else {
        setThemeColor('#0056D2');
      }
    };
    
    updateThemeColor();
    
    // Listen for visibility changes to reload theme
    const handleVisibilityChange = () => {
      if (!document.hidden) {
        updateThemeColor();
      }
    };
    
    document.addEventListener('visibilitychange', handleVisibilityChange);
    
    return () => {
      document.removeEventListener('visibilitychange', handleVisibilityChange);
    };
  }, [location.search]);

  console.log('Sidebar rendering with themeColor:', themeColor);

  // Only fetch invitation count once
  useEffect(() => {
    const fetchInvitationCount = async () => {
      try {
        const userEmail =
          localStorage.getItem("agentEmail") ||
          localStorage.getItem("sellerEmail");
        if (!userEmail) {
          setInvitationCount(0);
          return;
        }

        const response = await collaborationApi.getMyInvitations();
        const allInvitations = response.collaborations || [];

        // Count only pending received invitations

        const pendingReceived = allInvitations.filter(
          (invitation: any) =>
            invitation.secondary_agent_email &&
            invitation.secondary_agent_email.toLowerCase() ===
            userEmail.toLowerCase() &&
            invitation.status === "pending"
        );

        setInvitationCount(pendingReceived.length);
      } catch (error) {
        setInvitationCount(0);
      }
    };

    fetchInvitationCount();
  }, []);

  const handleToggle = useCallback(() => {
    const newState = !isCollapsed;
    setIsCollapsed(newState);
    if (onToggle) onToggle(newState);
    if (setIsSidebarCollapsed) setIsSidebarCollapsed(newState);
  }, [isCollapsed, onToggle, setIsSidebarCollapsed]);

  const handleNavigation = useCallback((itemId: string) => {
    const path = `/admin/dashboard${itemId === "dashboard" ? "" : `/${itemId}`}`;
    const fullPath = getTenantAwareHref(path);
    if (location.pathname !== path) {
      navigate(fullPath);
    }
  }, [navigate, location.pathname]);
  const menuItems = [
    { id: "dashboard", label: "Dashboard Overview", icon: Home },
    { id: "properties", label: "Property Management", icon: Building },
    { id: "users", label: "User Management", icon: Users },
    { id: "visits", label: "Visit Schedules", icon: Calendar },
    { id: "listing-packages", label: "Listing Packages", icon: Package },
    { id: "coupons", label: "Coupons Management", icon: Tag },
    { id: "discount-rules", label: "Discount Rules", icon: Percent },
    { id: "scholarships", label: "Scholarships", icon: GraduationCap },
    { id: "wallet", label: "Wallet & Credits Management", icon: Wallet },
    { id: "entitlements", label: "Entitlement Management", icon: Key },
    {
      id: "hyperlocal",
      label: "Hyperlocal Economic Indicators",
      icon: Activity,
    },
    { id: "market-analysis", label: "Market Analysis", icon: TrendingUp },
    { id: "inquiries", label: "Inquiry Management", icon: MessageSquare },
    { id: "escrows", label: "Escrow Review", icon: FileText },
    { id: "risk-compliance", label: "Risk and Compliance", icon: LifeBuoy },
    { id: "compliance", label: "Compliance Management", icon: CheckSquare },
    { id: "content-quality", label: "Content Quality Governance", icon: FileCheck },
    { id: "feature-flags", label: "Feature Flags & Experiments", icon: Flag },
    { id: "usage-analytics", label: "Usage Analytics & Funnels", icon: TrendingDown },
    { id: "price-prediction", label: "Price Prediction", icon: LineChart },
    { id: "billing", label: "Billing & Subscriptions", icon: DollarSign },
    { id: "financials", label: "Financial and Payments", icon: IndianRupee },
    { id: "business-revenue", label: "Business & Revenue Dashboard", icon: DollarSign },
    { id: "payment-analytics", label: "Payment Analytics", icon: PieChart },
    { id: "refunds", label: "Refund Management", icon: IndianRupee },
    { id: "campaign-management", label: "Campaign Management", icon: Megaphone },
    { id: "configuration", label: "Configuration Management", icon: Settings },
    { id: "system-health", label: "System Health & Monitoring", icon: Monitor },
    { id: "data-retention", label: "Data Retention & Privacy", icon: Database },
    { id: "grievances", label: "Grievance Management", icon: AlertCircle },
    { id: "support", label: "Support Tickets", icon: LifeBuoy },
    { id: "tax-settings", label: "Tax Settings", icon: Settings },
    { id: "usage-limits", label: "Usage Limits & Monitoring", icon: BarChart3 },
    { id: "security", label: "Security and Audit", icon: Shield },
    { id: "trends", label: "Property Trends", icon: TrendingUp },
    { id: "documents", label: "Document Verification", icon: FileText },
    { id: "nri-assistance", label: "NRI Assistance", icon: Globe },
    { id: "data-management", label: "Data Management", icon: Database },
    { id: "platform-settings", label: "Platform Settings", icon: Settings },
    { id: "developer/api-keys", label: "API Keys", icon: Key },
    { id: "developer/webhooks", label: "Webhooks", icon: Webhook },
    { id: "developer/marketplace", label: "Marketplace", icon: ShoppingBag },
    { id: "developer/connectors", label: "Connectors", icon: Link2 },
    { id: "developer/vault", label: "Secrets Vault", icon: ShieldCheck },
  ];

  const [logoError, setLogoError] = useState(false);
  const [logoIdx, setLogoIdx] = useState(0);
  const logoCandidates = ["/nal.jpg", "/Public/nal.jpg", "/nal-logo.png"];
  const currentLogo = logoCandidates[logoIdx];

  return (
    <div
      className={`bg-white shadow-sm border-r fixed left-0 top-0 h-full min-h-screen z-50 transition-all duration-300 flex flex-col ${isCollapsed ? "w-20" : "w-64"
        }`}
    >
      {/* Debug color indicator */}
      <div 
        style={{ backgroundColor: themeColor, height: '4px' }} 
        className="w-full"
        title={`Current theme: ${themeColor}`}
      />
      
      <div className="p-4">
        <div
          className="flex items-center gap-3 cursor-pointer"
          onClick={() => {
            // Set flag to allow admin to visit homepage
            sessionStorage.setItem("admin_visited_homepage", "true");
            navigate(getTenantAwareHref("/"));
          }}
        >
          {!isCollapsed ? (
            <>
              <img
                src={logoImage}
                alt="NAL Logo"
                className="w-12 h-12 rounded-lg object-cover"
              />
              <div>
                <h1 className="text-lg font-bold text-gray-900">
                  New Age Land
                </h1>
                <p className="text-xs text-gray-600">Admin Panel</p>
              </div>
            </>
          ) : (
            <img
              src={logoImage}
              alt="NAL Logo"
              className="w-12 h-12 rounded-lg object-cover mx-auto"
            />
          )}
        </div>
      </div>

      <nav
        className="px-2 flex-1 overflow-y-auto"
        style={{ scrollbarWidth: "none", msOverflowStyle: "none" }}
      >
        <div className="space-y-1">
          {menuItems.map((item, index) => {
            const Icon = item.icon;
            const isActive =
              location.pathname ===
              `/admin/dashboard/${item.id === "dashboard" ? "" : item.id}` ||
              (item.id === "dashboard" &&
                location.pathname === "/admin/dashboard");
            const isComparisonItem = item.id === "comparison";

            return (
              <div
                key={item.id}
                className={isComparisonItem ? "flex items-center" : ""}
              >
                <button
                  key={`${item.id}-${themeColor}`}
                  onClick={() => handleNavigation(item.id)}
                  className={`${isComparisonItem ? "flex-1" : "w-full"
                    } flex items-center px-3 py-4 text-sm font-medium rounded-lg transition-colors group relative ${isActive
                      ? "bg-[#0056D2] text-white shadow-sm"
                      : "text-gray-700 hover:bg-gray-100 hover:text-[#0056D2]"
                    }`}
                >
                  <div 
                    className="absolute inset-0 rounded-lg"
                    style={isActive ? { backgroundColor: themeColor } : {}}
                  />
                  <Icon
                    className={`w-5 h-5 ${isCollapsed ? "mx-auto" : "mr-3"} relative z-10`}
                  />
                  {!isCollapsed && (
                    <span className="whitespace-nowrap relative z-10">{item.label}</span>
                  )}
                  {item.id === "users" && invitationCount > 0 && (
                    <span className="ml-auto bg-red-500 text-white text-xs px-2 py-1 rounded-full min-w-[20px] text-center relative z-10">
                      {invitationCount}
                    </span>
                  )}
                  {isActive && !isCollapsed && (
                    <div className="absolute right-0 top-0 bottom-0 w-1 bg-[#00BFA6] rounded-l z-10"></div>
                  )}
                </button>
                {isComparisonItem && (
                  <button
                    onClick={handleToggle}
                    style={{ backgroundColor: themeColor }}
                    className="ml-2 w-8 h-8 rounded-full flex items-center justify-center transition-colors shadow-sm hover:opacity-80"
                  >
                    {isCollapsed ? (
                      <ChevronRight className="w-4 h-4 text-white" />
                    ) : (
                      <ChevronLeft className="w-4 h-4 text-white" />
                    )}
                  </button>
                )}
              </div>
            );
          })}
        </div>
      </nav>
    </div>
  );
};

export default Sidebar;
