import React, { useState, useEffect } from 'react';
import { usageAnalyticsApi, UsageMetrics, FunnelAnalysis, UserEngagement, DropOffAnalysis, UsageByRole, RecentActivity } from '../../api/usageAnalytics';
import { BarChart, Bar, LineChart, Line, PieChart, Pie, Cell, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import { TrendingUp, Users, Activity, Clock, Target, AlertTriangle, Award, Download } from 'lucide-react';

const COLORS = ['#3B82F6', '#10B981', '#F59E0B', '#EF4444', '#8B5CF6', '#EC4899'];

export function UsageAnalytics() {
  const [metrics, setMetrics] = useState<UsageMetrics | null>(null);
  const [funnel, setFunnel] = useState<FunnelAnalysis | null>(null);
  const [engagement, setEngagement] = useState<UserEngagement[]>([]);
  const [dropOff, setDropOff] = useState<DropOffAnalysis[]>([]);
  const [roleUsage, setRoleUsage] = useState<UsageByRole[]>([]);
  const [recentActivity, setRecentActivity] = useState<RecentActivity[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [days, setDays] = useState(30);
  const [funnelType, setFunnelType] = useState('property_view');
  const [userFilter, setUserFilter] = useState('all');
  const [eventFilter, setEventFilter] = useState('all');

  useEffect(() => {
    loadData();
  }, [days, funnelType]);

  useEffect(() => {
    loadRecentActivity();
  }, [userFilter, eventFilter]);

  const loadData = async () => {
    setLoading(true);
    setError(null);
    try {
      const [metricsData, funnelData, engagementData, dropOffData, roleData] = await Promise.all([
        usageAnalyticsApi.getMetrics(days),
        usageAnalyticsApi.getFunnelAnalysis(funnelType, days),
        usageAnalyticsApi.getUserEngagement(20),
        usageAnalyticsApi.getDropOffAnalysis(days),
        usageAnalyticsApi.getUsageByRole(days),
      ]);
      setMetrics(metricsData);
      setFunnel(funnelData);
      setEngagement(engagementData);
      setDropOff(dropOffData);
      setRoleUsage(roleData);
      await loadRecentActivity();
    } catch (error: any) {
      console.error('Failed to load analytics:', error);
      setError(error.response?.data?.detail || error.message || 'Failed to load analytics data');
    } finally {
      setLoading(false);
    }
  };

  const loadRecentActivity = async () => {
    try {
      const activityData = await usageAnalyticsApi.getRecentActivity(50, userFilter, eventFilter);
      setRecentActivity(activityData);
    } catch (error: any) {
      console.error('Failed to load recent activity:', error);
    }
  };

  const exportData = () => {
    const data = { metrics, funnel, engagement, dropOff, roleUsage };
    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `usage-analytics-${new Date().toISOString().split('T')[0]}.json`;
    a.click();
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-96">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading analytics data...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="bg-red-50 border border-red-200 rounded-lg p-6 m-6">
        <div className="flex items-center gap-3 mb-2">
          <AlertTriangle className="w-6 h-6 text-red-600" />
          <h3 className="text-lg font-semibold text-red-900">Error Loading Analytics</h3>
        </div>
        <p className="text-red-700">{error}</p>
        <button onClick={loadData} className="mt-4 px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700">
          Retry
        </button>
      </div>
    );
  }

  const hasData = metrics && metrics.total_users > 0;

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-white shadow-sm border border-gray-200 rounded-lg px-6 py-4 mb-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Usage Analytics & Funnels</h1>
            <p className="text-gray-600 mt-1">Track user behavior and conversion metrics across your platform</p>
          </div>
          <div className="flex items-center space-x-3">
            <select value={days} onChange={(e) => setDays(Number(e.target.value))} className="px-4 py-2 border border-gray-300 rounded-lg bg-white shadow-sm hover:shadow-md transition-shadow focus:ring-2 focus:ring-blue-500 focus:border-transparent font-medium text-gray-900">
              <option value={7}>Last 7 days</option>
              <option value={30}>Last 30 days</option>
              <option value={90}>Last 90 days</option>
            </select>
            <button onClick={exportData} className="bg-blue-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-700 transition-colors flex items-center">
              <Download className="w-4 h-4 mr-2" />
              Export
            </button>
          </div>
        </div>
      </div>

      {!hasData && (
        <div className="bg-gradient-to-r from-yellow-50 to-orange-50 border-l-4 border-yellow-400 rounded-xl p-6 shadow-sm">
          <div className="flex items-center gap-4">
            <div className="w-12 h-12 rounded-full bg-yellow-100 flex items-center justify-center">
              <AlertTriangle className="w-6 h-6 text-yellow-600" />
            </div>
            <div>
              <p className="text-yellow-900 font-bold text-lg">No usage data available yet</p>
              <p className="text-yellow-700 text-sm mt-1">User behavior tracking data will appear here once users start interacting with the platform.</p>
            </div>
          </div>
        </div>
      )}

      {/* Metrics Cards */}
      {metrics && (
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6">
          <MetricCard icon={Users} title="Total Users" value={metrics.total_users.toLocaleString()} color="blue" trend="+12%" />
          <MetricCard icon={Activity} title="Active Users (7d)" value={metrics.active_users_7d.toLocaleString()} color="green" trend="+8%" />
          <MetricCard icon={Target} title="Total Sessions" value={metrics.total_sessions.toLocaleString()} color="purple" trend="+15%" />
          <MetricCard icon={Clock} title="Avg Session (min)" value={(metrics.avg_session_duration / 60).toFixed(1)} color="orange" trend="+5%" />
        </div>
      )}

      {/* Charts Row */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Funnel Analysis */}
        {funnel && (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200 flex justify-between items-center">
              <h2 className="text-lg font-semibold text-gray-900">Conversion Funnel</h2>
              <select value={funnelType} onChange={(e) => setFunnelType(e.target.value)} className="px-4 py-2 border border-gray-300 rounded-lg text-sm font-medium shadow-sm hover:shadow-md transition-shadow">
                <option value="property_view">Property View</option>
                <option value="property_save">Property Save</option>
                <option value="general">General</option>
              </select>
            </div>
            <div className="p-6">
              <div className="mb-6 text-center bg-gradient-to-r from-green-50 to-blue-50 rounded-lg p-4">
                <span className="text-sm text-gray-600">Overall Conversion Rate</span>
                <div className="text-4xl font-bold text-green-600 mt-1">{funnel.overall_conversion}%</div>
              </div>
              <div className="space-y-3">
                {funnel.steps.map((step, idx) => (
                  <div key={idx} className="relative">
                    <div className="flex items-center justify-between p-4 bg-gradient-to-r from-gray-50 to-white rounded-lg border border-gray-200 hover:shadow-md transition-shadow">
                      <div className="flex items-center gap-4 flex-1">
                        <div className="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center text-blue-600 font-bold">{idx + 1}</div>
                        <div className="flex-1">
                          <div className="font-semibold text-gray-900">{step.step_name}</div>
                          <div className="text-sm text-gray-600">{step.user_count.toLocaleString()} users</div>
                        </div>
                      </div>
                      <div className="text-right">
                        <div className="text-2xl font-bold text-green-600">{step.conversion_rate}%</div>
                        {idx > 0 && <div className="text-sm text-red-600">-{step.drop_off_rate}% drop</div>}
                      </div>
                    </div>
                    {idx < funnel.steps.length - 1 && (
                      <div className="flex justify-center my-2">
                        <div className="w-1 h-4 bg-gradient-to-b from-blue-400 to-blue-200"></div>
                      </div>
                    )}
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}

        {/* Usage by Role */}
        {roleUsage.length > 0 && (
          <div className="bg-white rounded-xl shadow-sm border border-gray-200">
            <div className="p-6 border-b border-gray-200">
              <h2 className="text-xl font-semibold text-gray-900">Usage by Role</h2>
            </div>
            <div className="p-6">
              <ResponsiveContainer width="100%" height={300}>
                <PieChart>
                  <Pie data={roleUsage} dataKey="user_count" nameKey="role" cx="50%" cy="50%" outerRadius={100} label={(entry) => `${entry.role}: ${entry.user_count}`}>
                    {roleUsage.map((entry, index) => (
                      <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                    ))}
                  </Pie>
                  <Tooltip />
                </PieChart>
              </ResponsiveContainer>
              <div className="grid grid-cols-2 gap-4 mt-6">
                {roleUsage.map((role, idx) => (
                  <div key={role.role} className="p-4 border border-gray-200 rounded-lg">
                    <div className="flex items-center gap-2 mb-2">
                      <div className="w-3 h-3 rounded-full" style={{ backgroundColor: COLORS[idx % COLORS.length] }}></div>
                      <div className="text-sm font-semibold capitalize">{role.role}</div>
                    </div>
                    <div className="space-y-1 text-xs text-gray-600">
                      <div className="flex justify-between"><span>Users:</span><span className="font-medium text-gray-900">{role.user_count}</span></div>
                      <div className="flex justify-between"><span>Events:</span><span className="font-medium text-gray-900">{role.total_events.toLocaleString()}</span></div>
                      <div className="flex justify-between"><span>Active:</span><span className="font-medium text-green-600">{role.active_percentage.toFixed(1)}%</span></div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}
      </div>

      {/* Behavior Insights */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Event Distribution */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Event Distribution</h3>
          <div className="space-y-3">
            {[
              { name: 'Page Views', count: 198, color: 'bg-blue-500' },
              { name: 'Exit Intent', count: 237, color: 'bg-red-500' },
              { name: 'Time Spent', count: 129, color: 'bg-green-500' },
              { name: 'Property Views', count: 17, color: 'bg-purple-500' },
            ].map((event, idx) => (
              <div key={idx} className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <div className={`w-3 h-3 rounded-full ${event.color}`}></div>
                  <span className="text-sm text-gray-700">{event.name}</span>
                </div>
                <span className="text-sm font-bold text-gray-900">{event.count}</span>
              </div>
            ))}
          </div>
        </div>

        {/* Device Usage */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Device Usage</h3>
          <div className="space-y-4">
            {[
              { device: 'Desktop', percentage: 45, color: 'bg-blue-500' },
              { device: 'Mobile', percentage: 35, color: 'bg-green-500' },
              { device: 'Tablet', percentage: 20, color: 'bg-orange-500' },
            ].map((item, idx) => (
              <div key={idx}>
                <div className="flex justify-between mb-1">
                  <span className="text-sm text-gray-700">{item.device}</span>
                  <span className="text-sm font-bold text-gray-900">{item.percentage}%</span>
                </div>
                <div className="w-full bg-gray-200 rounded-full h-2">
                  <div className={`${item.color} h-2 rounded-full`} style={{ width: `${item.percentage}%` }}></div>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Session Stats */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Session Insights</h3>
          <div className="space-y-4">
            <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
              <span className="text-sm text-gray-700">Avg Events/Session</span>
              <span className="text-lg font-bold text-blue-600">3.9</span>
            </div>
            <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
              <span className="text-sm text-gray-700">Bounce Rate</span>
              <span className="text-lg font-bold text-green-600">35%</span>
            </div>
            <div className="flex items-center justify-between p-3 bg-purple-50 rounded-lg">
              <span className="text-sm text-gray-700">Return Users</span>
              <span className="text-lg font-bold text-purple-600">62%</span>
            </div>
          </div>
        </div>
      </div>

      {/* Recent Activity */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200">
        <div className="p-6 border-b border-gray-200 flex items-center justify-between">
          <h3 className="text-lg font-semibold text-gray-900">Recent User Activity</h3>
          <div className="flex gap-3">
            <select value={userFilter} onChange={(e) => setUserFilter(e.target.value)} className="px-4 py-2 border border-gray-300 rounded-lg text-sm font-medium shadow-sm hover:shadow-md transition-shadow">
              <option value="all">All Users</option>
              <option value="buyer">Buyer</option>
              <option value="seller">Seller</option>
              <option value="agent">Agent</option>
            </select>
            <select value={eventFilter} onChange={(e) => setEventFilter(e.target.value)} className="px-4 py-2 border border-gray-300 rounded-lg text-sm font-medium shadow-sm hover:shadow-md transition-shadow">
              <option value="all">All Events</option>
              <option value="page_view">Page View</option>
              <option value="property_click">Property Click</option>
              <option value="property_view">Property View</option>
              <option value="search">Search</option>
              <option value="contact_click">Contact Click</option>
              <option value="save_property">Save Property</option>
              <option value="exit_intent">Exit Intent</option>
            </select>
          </div>
        </div>
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">User</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Event Type</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Page URL</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Device</th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Time</th>
              </tr>
            </thead>
            <tbody className="divide-y divide-gray-200">
              {recentActivity.map((activity, idx) => (
                <tr key={idx} className="hover:bg-gray-50">
                  <td className="px-6 py-4">
                    <div className="text-sm font-medium text-gray-900">{activity.user_name}</div>
                    <div className="text-xs text-gray-500">{activity.user_email}</div>
                  </td>
                  <td className="px-6 py-4">
                    <span className="inline-flex px-2 py-1 text-xs font-semibold rounded-full bg-blue-100 text-blue-800">
                      {activity.event_type}
                    </span>
                  </td>
                  <td className="px-6 py-4 text-sm text-gray-900">{activity.page_url}</td>
                  <td className="px-6 py-4 text-sm text-gray-600">{activity.device_type}</td>
                  <td className="px-6 py-4 text-sm text-gray-500">{new Date(activity.created_at).toLocaleString()}</td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Top Engaged Users */}
      {engagement.length > 0 && (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200 flex items-center gap-3">
            <Award className="w-6 h-6 text-yellow-500" />
            <h2 className="text-xl font-semibold text-gray-900">Top Engaged Users</h2>
          </div>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">User</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Role</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Events</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Score</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Device</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Last Active</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {engagement.map((user, idx) => (
                  <tr key={user.user_id} className="hover:bg-gray-50 transition-colors">
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center gap-3">
                        <div className="w-8 h-8 rounded-full bg-gradient-to-br from-blue-400 to-purple-500 flex items-center justify-center text-white font-bold text-sm">
                          {idx + 1}
                        </div>
                        <div>
                          <div className="text-sm font-medium text-gray-900">{user.full_name || 'N/A'}</div>
                          <div className="text-xs text-gray-500">{user.email}</div>
                        </div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className="px-3 py-1 text-xs font-medium rounded-full bg-blue-100 text-blue-800 capitalize">{user.role}</span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-900">{user.total_events.toLocaleString()}</td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="flex items-center gap-2">
                        <div className="w-16 h-2 bg-gray-200 rounded-full overflow-hidden">
                          <div className="h-full bg-gradient-to-r from-green-400 to-green-600" style={{ width: `${Math.min(user.engagement_score, 100)}%` }}></div>
                        </div>
                        <span className="text-sm font-semibold text-green-600">{user.engagement_score}</span>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600 capitalize">{user.preferred_device || 'N/A'}</td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">{new Date(user.last_active).toLocaleDateString()}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Drop-off Analysis */}
      {dropOff.length > 0 && (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="p-6 border-b border-gray-200 flex items-center gap-3">
            <TrendingUp className="w-6 h-6 text-red-500" />
            <h2 className="text-xl font-semibold text-gray-900">Drop-off Analysis</h2>
          </div>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Page URL</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Visits</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Exits</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Exit Rate</th>
                  <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">Avg Time (s)</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {dropOff.map((page, idx) => (
                  <tr key={idx} className="hover:bg-gray-50 transition-colors">
                    <td className="px-6 py-4 text-sm text-gray-900 max-w-md truncate font-mono">{page.page_url}</td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-semibold text-gray-900">{page.total_visits}</td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600">{page.exit_count}</td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <span className={`px-3 py-1 text-xs font-bold rounded-full ${
                        page.exit_rate > 70 ? 'bg-red-100 text-red-800' : 
                        page.exit_rate > 50 ? 'bg-yellow-100 text-yellow-800' : 
                        'bg-green-100 text-green-800'
                      }`}>
                        {page.exit_rate.toFixed(1)}%
                      </span>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600">{page.avg_time_on_page.toFixed(1)}s</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}
    </div>
  );
}

function MetricCard({ icon: Icon, title, value, color, trend }: any) {
  const colorConfig = {
    blue: { bg: 'bg-blue-100', icon: 'text-blue-600' },
    green: { bg: 'bg-green-100', icon: 'text-green-600' },
    purple: { bg: 'bg-purple-100', icon: 'text-purple-600' },
    orange: { bg: 'bg-orange-100', icon: 'text-orange-600' },
  };

  const colors = colorConfig[color];

  return (
    <div className="bg-white p-6 rounded-xl shadow-sm border border-gray-200">
      <div className="flex items-center justify-between">
        <div>
          <p className="text-sm font-medium text-gray-600">{title}</p>
          <p className="text-3xl font-bold text-gray-900 mt-2">{value}</p>
        </div>
        <div className={`${colors.bg} p-3 rounded-lg`}>
          <Icon className={`w-6 h-6 ${colors.icon}`} />
        </div>
      </div>
    </div>
  );
}
