import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../ui/card";
import { Button } from "../ui/button";
import { Badge } from "../ui/badge";
import { Input } from "../ui/input";
import { 
  Search, Filter, Download, UserPlus, Users, UserCheck, 
  Eye, Edit, Ban, ChevronLeft, ChevronRight, Shield, X, Activity, TrendingUp, Bookmark, Calendar
} from "lucide-react";
import { getAdminUsersList, AdminUserResponse, AdminPaginatedUsersResponse, updateUserRole } from "../../api/admin";
import { toast } from "sonner";
import { behaviorAnalysisApi } from "../../api/behaviorAnalysis";

interface UserManagementProps {
  isSidebarCollapsed?: boolean;
}

export function UserManagement({ isSidebarCollapsed = false }: UserManagementProps) {
  const [usersData, setUsersData] = useState<AdminPaginatedUsersResponse | null>(null);
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [verifiedFilter, setVerifiedFilter] = useState<boolean | undefined>(undefined);
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize, setPageSize] = useState(20);
  const [showRoleForm, setShowRoleForm] = useState(false);
  const [selectedUserId, setSelectedUserId] = useState('');
  const [selectedRole, setSelectedRole] = useState<'admin' | 'user'>('user');
  const [updating, setUpdating] = useState(false);
  const [viewingUser, setViewingUser] = useState<AdminUserResponse | null>(null);
  const [userInsights, setUserInsights] = useState<any>(null);
  const [userEvents, setUserEvents] = useState<any[]>([]);
  const [loadingBehavior, setLoadingBehavior] = useState(false);

  const fetchUsers = async () => {
    try {
      setLoading(true);
      const params = {
        page: currentPage,
        limit: pageSize,
        ...(searchQuery && { search: searchQuery }),
        ...(statusFilter && { status_filter: statusFilter }),
        ...(verifiedFilter !== undefined && { verified_filter: verifiedFilter })
      };
      
      const data = await getAdminUsersList(params);
      setUsersData(data);
    } catch (error) {
      console.error('Error fetching users:', error);
      toast.error('Failed to fetch users');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchUsers();
  }, [currentPage, pageSize]);

  const handleSearch = () => {
    setCurrentPage(1);
    fetchUsers();
  };

  const handleFilterChange = () => {
    setCurrentPage(1);
    fetchUsers();
  };

  const clearFilters = () => {
    setSearchQuery('');
    setStatusFilter('');
    setVerifiedFilter(undefined);
    setCurrentPage(1);
    fetchUsers();
  };

  const handleViewUser = async (user: AdminUserResponse) => {
    setViewingUser(user);
    setLoadingBehavior(true);
    try {
      const [insights, events] = await Promise.all([
        behaviorAnalysisApi.getSpecificUserInsights(user.user_id),
        behaviorAnalysisApi.getSpecificUserEvents(user.user_id, 50)
      ]);
      setUserInsights(insights);
      setUserEvents(events);
    } catch (error) {
      console.error('Error loading behavior data:', error);
    } finally {
      setLoadingBehavior(false);
    }
  };

  const handleRoleUpdate = async () => {
    if (!selectedUserId) {
      toast.error('Please select a user');
      return;
    }

    setUpdating(true);
    try {
      await updateUserRole(selectedUserId, selectedRole);
      toast.success(`User role updated to ${selectedRole}`);
      setShowRoleForm(false);
      setSelectedUserId('');
      setSelectedRole('user');
      fetchUsers(); // Refresh the list
    } catch (error) {
      toast.error('Failed to update user role');
    } finally {
      setUpdating(false);
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  const getStatusBadge = (status: string) => {
    const colors = {
      active: 'bg-green-100 text-green-800',
      inactive: 'bg-gray-100 text-gray-800', 
      suspended: 'bg-red-100 text-red-800',
      pending_verification: 'bg-yellow-100 text-yellow-800'
    };
    return (
      <Badge className={`${colors[status as keyof typeof colors] || colors.active} text-xs`}>
        {status.replace('_', ' ')}
      </Badge>
    );
  };

  if (loading && !usersData) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-[#2B256D]"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900">User Management</h1>
          <p className="text-sm text-gray-600 mt-1">Manage and monitor platform users</p>
        </div>
        <div className="flex items-center gap-2">
          <Button variant="outline" size="sm" className="text-xs">
            <Download className="w-3 h-3 mr-1" />
            Export
          </Button>
          <Button size="sm" className="bg-blue-500 hover:bg-[#1f1a5a] text-xs" onClick={() => setShowRoleForm(true)}>
             Add Admin
          </Button>
        </div>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Total Users</p>
                <p className="text-2xl font-semibold text-gray-900 mt-1">{usersData?.total_count || 0}</p>
              </div>
              <div className="p-2 bg-[#2B256D]/10 rounded-lg">
                <Users className="w-5 h-5 text-[#2B256D]" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Verified</p>
                <p className="text-2xl font-semibold text-gray-900 mt-1">
                  {usersData?.users.filter(u => u.is_verified).length || 0}
                </p>
              </div>
              <div className="p-2 bg-green-100 rounded-lg">
                <UserCheck className="w-5 h-5 text-green-600" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Active</p>
                <p className="text-2xl font-semibold text-gray-900 mt-1">
                  {usersData?.users.filter(u => u.status === 'active').length || 0}
                </p>
              </div>
              <div className="p-2 bg-blue-100 rounded-lg">
                <Shield className="w-5 h-5 text-blue-600" />
              </div>
            </div>
          </CardContent>
        </Card>
        
        <Card className="border border-gray-200 shadow-sm">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <p className="text-xs font-medium text-gray-600 uppercase tracking-wide">Avg. Profile</p>
                <p className="text-2xl font-semibold text-gray-900 mt-1">
                  {usersData?.users.length ? 
                    Math.round(usersData.users.reduce((acc, u) => acc + u.profile_completion_percentage, 0) / usersData.users.length) 
                    : 0}%
                </p>
              </div>
              <div className="p-2 bg-purple-100 rounded-lg">
                <Users className="w-5 h-5 text-purple-600" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

   
{/* Filters */}
{/* <Card className="border border-gray-200 shadow-sm">
  <CardContent className="p-4">
    <div className="flex flex-col md:flex-row gap-3 items-center">
      <div className="flex-1 w-full">
        <div className="relative">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
          <Input 
            placeholder="Search users..." 
            value={searchQuery} 
            onChange={(e) => {
              setSearchQuery(e.target.value);
              if (e.target.value === '') {
                handleSearch();
              }
            }} 
            className="pl-10 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
            style={{ height: '32px' }}
            onKeyPress={(e) => e.key === 'Enter' && handleSearch()} 
          />
        </div>
      </div>
      <select 
        value={statusFilter} 
        onChange={(e) => { setStatusFilter(e.target.value); handleFilterChange(); }} 
        className="px-3 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D] min-w-[100px]"
        style={{ height: '20px', padding: '0 12px' }}
      >
        <option value="">All Status</option>
        <option value="active">Active</option>
        <option value="inactive">Inactive</option>
        <option value="suspended">Suspended</option>
      </select>
      <select 
        value={verifiedFilter === undefined ? '' : verifiedFilter.toString()} 
        onChange={(e) => { setVerifiedFilter(e.target.value === '' ? undefined : e.target.value === 'true'); handleFilterChange(); }} 
        className="px-3 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D] min-w-[120px]"
        style={{ height: '32px', padding: '0 12px' }}
      >
        <option value="">All Verification</option>
        <option value="true">Verified</option>
        <option value="false">Unverified</option>
      </select>
      <Button 
        variant="outline" 
        size="sm" 
        className="text-xs px-3 min-w-[60px]"
        style={{ height: '32px' }}
        onClick={clearFilters}
      >
        Clear
      </Button>
    </div>
  </CardContent>
</Card> */}

 

      {/* Table */}
      <Card className="border border-gray-200 shadow-sm">
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50 border-b">
                <tr>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">User</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Contact</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Status</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Profile</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Joined</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {loading ? (
                  Array.from({ length: 5 }).map((_, i) => (
                    <tr key={i} className="animate-pulse">
                      <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-32"></div></td>
                      <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-40"></div></td>
                      <td className="px-4 py-3"><div className="h-6 bg-gray-200 rounded w-20"></div></td>
                      <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-16"></div></td>
                      <td className="px-4 py-3"><div className="h-4 bg-gray-200 rounded w-24"></div></td>
                      <td className="px-4 py-3"><div className="h-8 bg-gray-200 rounded w-24"></div></td>
                    </tr>
                  ))
                ) : usersData?.users.map((user) => (
                  <tr key={user.user_id} className="hover:bg-gray-50">
                    <td className="px-4 py-3">
                      <div className="flex items-center space-x-3">
                        <div className="w-7 h-7 bg-[#2B256D] rounded-full flex items-center justify-center text-white text-xs font-medium">
                          {(user.first_name || 'U').charAt(0)}{(user.last_name || 'N').charAt(0)}
                        </div>
                        <div>
                          <div className="text-sm font-medium text-gray-900">
                            {user.first_name || 'Unknown'} {user.last_name || 'User'}
                          </div>
                          <div className="text-xs text-gray-500">ID: {user.user_id.slice(0, 8)}</div>
                        </div>
                      </div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="text-sm text-gray-900">{user.email}</div>
                      <div className="text-xs text-gray-500">{user.phone_number || 'No phone'}</div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="space-y-1">
                        {getStatusBadge(user.status)}
                        {user.is_verified && (
                          <Badge className="bg-blue-100 text-blue-800 text-xs">
                            <Shield className="w-3 h-3 mr-1" />
                            Verified
                          </Badge>
                        )}
                      </div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="flex items-center space-x-2">
                        <div className="w-10 bg-gray-200 rounded-full h-1.5">
                          <div 
                            className="bg-[#2B256D] h-1.5 rounded-full" 
                            style={{ width: `${user.profile_completion_percentage}%` }}
                          ></div>
                        </div>
                        <span className="text-xs text-gray-600">
                          {user.profile_completion_percentage}%
                        </span>
                      </div>
                    </td>
                    <td className="px-4 py-3 text-xs text-gray-500">
                      {formatDate(user.created_at)}
                    </td>
                    <td className="px-4 py-3">
                      <div className="flex items-center space-x-1">
                        <Button variant="ghost" size="sm" className="text-blue-600 hover:bg-blue-50 p-1" onClick={() => handleViewUser(user)}>
                          <Eye className="w-3 h-3" />
                        </Button>
                        <Button variant="ghost" size="sm" className="text-gray-600 hover:bg-gray-50 p-1">
                          <Edit className="w-3 h-3" />
                        </Button>
                        <Button variant="ghost" size="sm" className="text-red-600 hover:bg-red-50 p-1">
                          <Ban className="w-3 h-3" />
                        </Button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
          
          {/* Pagination */}
          {usersData && usersData.total_pages > 1 && (
            <div className="bg-gray-50 px-4 py-3 border-t">
              <div className="flex items-center justify-between">
                <div className="text-xs text-gray-700">
                  Showing {((currentPage - 1) * pageSize) + 1} to {Math.min(currentPage * pageSize, usersData.total_count)} of {usersData.total_count} users
                </div>
                <div className="flex items-center space-x-1">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                    disabled={currentPage === 1}
                    className="text-xs p-2"
                  >
                    <ChevronLeft className="w-3 h-3" />
                  </Button>
                  
                  {Array.from({ length: Math.min(5, usersData.total_pages) }, (_, i) => {
                    const page = i + 1;
                    return (
                      <Button
                        key={page}
                        variant={currentPage === page ? "default" : "outline"}
                        size="sm"
                        onClick={() => setCurrentPage(page)}
                        className={currentPage === page ? "bg-[#2B256D] hover:bg-[#1f1a5a] text-xs p-2" : "text-xs p-2"}
                      >
                        {page}
                      </Button>
                    );
                  })}
                  
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setCurrentPage(prev => Math.min(usersData.total_pages, prev + 1))}
                    disabled={currentPage === usersData.total_pages}
                    className="text-xs p-2"
                  >
                    <ChevronRight className="w-3 h-3" />
                  </Button>
                </div>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* User Behavior Modal */}
      {viewingUser && (
        <div className="fixed top-0 bottom-0 backdrop-blur-md bg-white/70 flex items-center justify-center z-[9999] p-4 animate-in fade-in duration-200" style={{ left: isSidebarCollapsed ? '80px' : '256px', right: 0 }}>
          <div className="bg-white rounded-xl w-full max-w-4xl max-h-[90vh] overflow-hidden shadow-2xl border border-gray-200 animate-in zoom-in-95 duration-200">
            <div className="flex items-center justify-between p-6 border-b bg-gradient-to-r from-blue-50 to-purple-50">
              <div className="flex items-center gap-3">
                <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white font-semibold shadow-lg">
                  {viewingUser.first_name?.charAt(0)}{viewingUser.last_name?.charAt(0)}
                </div>
                <div>
                  <h3 className="text-xl font-bold text-gray-900">User Behavior Analytics</h3>
                  <p className="text-sm font-medium text-gray-700 mt-1">{viewingUser.first_name} {viewingUser.last_name} • {viewingUser.email}</p>
                </div>
              </div>
              <Button variant="ghost" size="sm" className="hover:bg-white/50" onClick={() => { setViewingUser(null); setUserInsights(null); setUserEvents([]); }}>
                <X className="w-5 h-5" />
              </Button>
            </div>
            
            <div className="p-6 overflow-y-auto bg-gradient-to-b from-gray-50 to-white" style={{ maxHeight: 'calc(90vh - 100px)' }}>
              {loadingBehavior ? (
                <div className="flex items-center justify-center py-12">
                  <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                </div>
              ) : userInsights ? (
                <div className="space-y-6">
                  {/* Time Spent Visualization */}
                  {userEvents.filter((e: any) => e.event_type === 'time_spent').length > 0 && (
                    <div className="bg-gradient-to-br from-blue-50 to-purple-50 rounded-lg p-5 border border-blue-200">
                      <div className="flex items-center justify-between mb-4">
                        <h4 className="font-semibold text-base text-gray-800">Time Spent Analysis</h4>
                        <span className="text-xs text-gray-600 bg-white px-3 py-1 rounded-full">
                          {userEvents.filter((e: any) => e.event_type === 'time_spent').length} sessions
                        </span>
                      </div>
                      <div className="bg-white rounded-lg p-4 relative">
                        <div className="absolute left-0 top-0 bottom-0 w-12 flex flex-col justify-between text-xs text-gray-500 py-4">
                          {(() => {
                            const timeSpentEvents = userEvents.filter((e: any) => e.event_type === 'time_spent').slice(0, 10);
                            const maxDuration = Math.max(...timeSpentEvents.map((e: any) => e.event_data?.duration_ms || 0));
                            const maxMinutes = Math.ceil(maxDuration / 60000);
                            return [maxMinutes, Math.floor(maxMinutes * 0.75), Math.floor(maxMinutes * 0.5), Math.floor(maxMinutes * 0.25), 0].map((val, i) => (
                              <span key={i}>{val}m</span>
                            ));
                          })()}
                        </div>
                        <div className="ml-12 relative h-56">
                          <svg className="w-full h-full" viewBox="0 0 800 200" preserveAspectRatio="none">
                            {/* Grid lines */}
                            {[0, 50, 100, 150, 200].map((y) => (
                              <line key={y} x1="0" y1={y} x2="800" y2={y} stroke="#f3f4f6" strokeWidth="1" strokeDasharray="5,5" />
                            ))}
                            {/* Area fill */}
                            <polygon
                              fill="url(#areaGradient)"
                              opacity="0.3"
                              points={`0,200 ${userEvents
                                .filter((e: any) => e.event_type === 'time_spent')
                                .slice(0, 10)
                                .map((event: any, idx: number, arr: any[]) => {
                                  const duration = event.event_data?.duration_ms || 0;
                                  const maxDuration = Math.max(...arr.map((e: any) => e.event_data?.duration_ms || 0));
                                  const x = (idx / Math.max(arr.length - 1, 1)) * 800;
                                  const y = 200 - (duration / maxDuration) * 180;
                                  return `${x},${y}`;
                                })
                                .join(' ')} 800,200`}
                            />
                            {/* Line graph */}
                            <polyline
                              fill="none"
                              stroke="url(#lineGradient)"
                              strokeWidth="3"
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              points={userEvents
                                .filter((e: any) => e.event_type === 'time_spent')
                                .slice(0, 10)
                                .map((event: any, idx: number, arr: any[]) => {
                                  const duration = event.event_data?.duration_ms || 0;
                                  const maxDuration = Math.max(...arr.map((e: any) => e.event_data?.duration_ms || 0));
                                  const x = (idx / Math.max(arr.length - 1, 1)) * 800;
                                  const y = 200 - (duration / maxDuration) * 180;
                                  return `${x},${y}`;
                                })
                                .join(' ')}
                            />
                            {/* Data points with hover effect */}
                            {userEvents
                              .filter((e: any) => e.event_type === 'time_spent')
                              .slice(0, 10)
                              .map((event: any, idx: number, arr: any[]) => {
                                const duration = event.event_data?.duration_ms || 0;
                                const maxDuration = Math.max(...arr.map((e: any) => e.event_data?.duration_ms || 0));
                                const x = (idx / Math.max(arr.length - 1, 1)) * 800;
                                const y = 200 - (duration / maxDuration) * 180;
                                const minutes = Math.floor(duration / 60000);
                                const seconds = Math.floor((duration % 60000) / 1000);
                                return (
                                  <g key={idx}>
                                    <circle cx={x} cy={y} r="6" fill="white" stroke="#8b5cf6" strokeWidth="2" />
                                    <circle cx={x} cy={y} r="3" fill="#8b5cf6" />
                                    <title>{`${minutes}m ${seconds}s`}</title>
                                  </g>
                                );
                              })}
                            <defs>
                              <linearGradient id="lineGradient" x1="0%" y1="0%" x2="100%" y2="0%">
                                <stop offset="0%" stopColor="#3b82f6" />
                                <stop offset="100%" stopColor="#8b5cf6" />
                              </linearGradient>
                              <linearGradient id="areaGradient" x1="0%" y1="0%" x2="0%" y2="100%">
                                <stop offset="0%" stopColor="#8b5cf6" />
                                <stop offset="100%" stopColor="#3b82f6" />
                              </linearGradient>
                            </defs>
                          </svg>
                        </div>
                        <div className="ml-12 mt-2 flex justify-between text-xs text-gray-500">
                          {userEvents.filter((e: any) => e.event_type === 'time_spent').slice(0, 10).map((event: any, idx: number) => (
                            <span key={idx}>•</span>
                          ))}
                        </div>
                      </div>
                      <div className="mt-4 grid grid-cols-2 gap-2">
                        {userEvents.filter((e: any) => e.event_type === 'time_spent').slice(0, 6).map((event: any, idx: number) => {
                          const duration = event.event_data?.duration_ms || 0;
                          const minutes = Math.floor(duration / 60000);
                          const seconds = Math.floor((duration % 60000) / 1000);
                          const url = event.event_data?.url || '';
                          const pageName = url.split('/').filter(Boolean).pop() || 'Home';
                          return (
                            <div key={idx} className="bg-white px-3 py-2 rounded-lg border border-gray-200 flex items-center justify-between">
                              <span className="text-xs text-gray-700 font-medium truncate flex-1">{pageName}</span>
                              <span className="text-xs font-bold text-purple-600 ml-2">{minutes}:{seconds.toString().padStart(2, '0')}</span>
                            </div>
                          );
                        })}
                      </div>
                    </div>
                  )}

                  <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg p-4 border border-blue-200">
                      <div className="flex items-center gap-2 mb-2">
                        <Activity className="w-5 h-5 text-blue-600" />
                        <span className="text-2xl font-bold text-blue-900">{userInsights.total_events}</span>
                      </div>
                      <p className="text-xs text-blue-700 font-medium">Total Events</p>
                    </div>
                    <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-lg p-4 border border-green-200">
                      <div className="flex items-center gap-2 mb-2">
                        <TrendingUp className="w-5 h-5 text-green-600" />
                        <span className="text-2xl font-bold text-green-900">{userInsights.engagement_score.toFixed(1)}</span>
                      </div>
                      <p className="text-xs text-green-700 font-medium">Engagement Score</p>
                    </div>
                    <div className="bg-gradient-to-br from-purple-50 to-purple-100 rounded-lg p-4 border border-purple-200">
                      <div className="flex items-center gap-2 mb-2">
                        <Users className="w-5 h-5 text-purple-600" />
                        <span className="text-lg font-bold text-purple-900">{userInsights.preferred_device}</span>
                      </div>
                      <p className="text-xs text-purple-700 font-medium">Preferred Device</p>
                    </div>
                    <div className="bg-gradient-to-br from-orange-50 to-orange-100 rounded-lg p-4 border border-orange-200">
                      <div className="flex items-center gap-2 mb-2">
                        <Calendar className="w-5 h-5 text-orange-600" />
                        <span className="text-sm font-bold text-orange-900">{new Date(userInsights.last_activity).toLocaleDateString()}</span>
                      </div>
                      <p className="text-xs text-orange-700 font-medium">Last Activity</p>
                    </div>
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="bg-white rounded-lg p-4 border border-gray-200">
                      <h4 className="font-semibold text-sm text-gray-700 mb-3 flex items-center gap-2">
                        <Search className="w-4 h-4" />
                        Top Searched Locations
                      </h4>
                      <div className="space-y-2">
                        {userInsights.top_searched_locations.length > 0 ? (
                          userInsights.top_searched_locations.map((loc: string, idx: number) => (
                            <div key={idx} className="text-sm text-gray-600 bg-blue-50 px-3 py-1.5 rounded">{loc}</div>
                          ))
                        ) : (
                          <p className="text-xs text-gray-400">No data</p>
                        )}
                      </div>
                    </div>
                    <div className="bg-white rounded-lg p-4 border border-gray-200">
                      <h4 className="font-semibold text-sm text-gray-700 mb-3 flex items-center gap-2">
                        <Bookmark className="w-4 h-4" />
                        Preferred Property Types
                      </h4>
                      <div className="space-y-2">
                        {userInsights.preferred_property_types.length > 0 ? (
                          userInsights.preferred_property_types.map((type: string, idx: number) => (
                            <div key={idx} className="text-sm text-gray-600 bg-green-50 px-3 py-1.5 rounded capitalize">{type}</div>
                          ))
                        ) : (
                          <p className="text-xs text-gray-400">No data</p>
                        )}
                      </div>
                    </div>
                  </div>

                  <div className="bg-white rounded-lg p-4 border border-gray-200">
                    <h4 className="font-semibold text-sm text-gray-700 mb-3">Recent Activity</h4>
                    <div className="space-y-2 max-h-60 overflow-y-auto">
                      {userEvents.filter((e: any) => !['time_spent', 'exit_intent', 'page_view', 'scroll'].includes(e.event_type)).map((event: any, idx: number) => {
                        const eventData = event.event_data || {};
                        const propertyTitle = eventData.title || eventData.property_title || (event.property_id ? `Property #${event.property_id}` : '');
                        
                        return (
                          <div key={idx} className="flex items-start gap-3 text-sm p-3 hover:bg-gray-50 rounded border border-gray-100">
                            <div className="flex-shrink-0 mt-0.5">
                              {event.event_type === 'property_view' && <Eye className="w-4 h-4 text-blue-500" />}
                              {event.event_type === 'search' && <Search className="w-4 h-4 text-orange-500" />}
                              {event.event_type === 'save_property' && <Bookmark className="w-4 h-4 text-green-500" />}
                              {event.event_type === 'wishlist_add' && (
                                <svg className="w-4 h-4 text-red-500" fill="currentColor" viewBox="0 0 20 20">
                                  <path fillRule="evenodd" d="M3.172 5.172a4 4 0 015.656 0L10 6.343l1.172-1.171a4 4 0 115.656 5.656L10 17.657l-6.828-6.829a4 4 0 010-5.656z" clipRule="evenodd" />
                                </svg>
                              )}
                              {event.event_type === 'schedule_visit' && <Calendar className="w-4 h-4 text-pink-500" />}
                            </div>
                            <div className="flex-1 min-w-0">
                              <p className="text-gray-900 font-medium capitalize">{event.event_type.replace(/_/g, ' ')}</p>
                              {propertyTitle && <p className="text-xs text-gray-600 mt-0.5 truncate">{propertyTitle}</p>}
                              {event.event_type === 'search' && eventData.query && (
                                <p className="text-xs text-gray-600 mt-0.5">Searched: "{eventData.query}"</p>
                              )}
                            </div>
                            <span className="text-xs text-gray-500 whitespace-nowrap">{new Date(event.created_at).toLocaleString()}</span>
                          </div>
                        );
                      })}
                      {userEvents.filter((e: any) => !['time_spent', 'exit_intent', 'page_view', 'scroll'].includes(e.event_type)).length === 0 && <p className="text-xs text-gray-400 text-center py-4">No recent events</p>}
                    </div>
                  </div>
                </div>
              ) : (
                <div className="text-center py-12">
                  <Activity className="w-12 h-12 text-gray-300 mx-auto mb-2" />
                  <p className="text-gray-500">No behavior data available</p>
                </div>
              )}
            </div>
          </div>
        </div>
      )}

      {/* Role Assignment Modal */}
      {showRoleForm && (
        <div className="fixed inset-0 backdrop-blur-sm bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 w-full max-w-md mx-4 shadow-xl">
            <h3 className="text-lg font-medium mb-4">Assign User Role</h3>
            
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Select User</label>
                <select
                  value={selectedUserId}
                  onChange={(e) => setSelectedUserId(e.target.value)}
                  className="w-full px-3 py-2 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
                >
                  <option value="">Choose a user...</option>
                  {usersData?.users
                    .filter(user => user.status === 'active')
                    .map((user) => {
                      const currentUser = JSON.parse(localStorage.getItem('user') || '{}');
                      const isCurrentUser = user.user_id === currentUser.user_id;
                      const displayName = user.first_name && user.last_name 
                        ? ` (${user.first_name} ${user.last_name})` 
                        : '';
                      return (
                        <option key={user.user_id} value={user.user_id}>
                          {user.email || 'No email'}{displayName}{isCurrentUser ? ' (You)' : ''}
                        </option>
                      );
                    })}
                </select>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Role</label>
                <select
                  value={selectedRole}
                  onChange={(e) => setSelectedRole(e.target.value as 'admin' | 'user')}
                  className="w-full px-3 py-2 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-[#2B256D] focus:border-[#2B256D]"
                >
                  <option value="user">User</option>
                  <option value="admin">Admin</option>
                </select>
              </div>
            </div>
            
            <div className="flex justify-end space-x-3 mt-6">
              <Button 
                variant="outline" 
                size="sm"
                onClick={() => {
                  setShowRoleForm(false);
                  setSelectedUserId('');
                  setSelectedRole('user');
                }}
                disabled={updating}
                className="text-xs"
              >
                Cancel
              </Button>
              <Button 
                onClick={handleRoleUpdate}
                disabled={updating || !selectedUserId}
                size="sm"
                className="bg-[#2B256D] hover:bg-[#1f1a5a] text-xs"
              >
                {updating ? 'Updating...' : 'Update Role'}
              </Button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}