import React, { useState, useEffect } from 'react';
import { Calendar, Clock, User, MapPin, Phone, Mail, Eye, Check, X } from 'lucide-react';
import { getAdminVisitSchedules, approveVisitSchedule, rejectVisitSchedule, VisitSchedule } from '../../api/adminVisitSchedules';
import { toast } from 'sonner';

interface VisitSchedulesProps {
  initialTab?: string;
}

export function VisitSchedules({ initialTab }: VisitSchedulesProps) {
  const [activeTab, setActiveTab] = useState<'approved' | 'pending' | 'rejected'>((initialTab as 'approved' | 'pending' | 'rejected') || 'approved');
  const [selectedVisit, setSelectedVisit] = useState<VisitSchedule | null>(null);
  const [visits, setVisits] = useState<VisitSchedule[]>([]);
  const [loading, setLoading] = useState(true);
  const [totalCount, setTotalCount] = useState(0);
  const [pendingCount, setPendingCount] = useState(0);
  const [currentPage, setCurrentPage] = useState(1);
  const [processingAction, setProcessingAction] = useState<{visitId: number, action: 'approve' | 'reject'} | null>(null);
  const [successMessage, setSuccessMessage] = useState<{ message: string; type: 'approve' | 'reject' } | null>(null);

  // Prevent background scroll when modal is open
  useEffect(() => {
    if (selectedVisit) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = 'unset';
    }
    return () => {
      document.body.style.overflow = 'unset';
    };
  }, [selectedVisit]);

  const fetchVisits = async (status: string, page: number = 1) => {
    try {
      setLoading(true);
      const data = await getAdminVisitSchedules(page, 10, status);
      setVisits(data.visits);
      setTotalCount(data.total_count);
      setCurrentPage(page);
    } catch (error) {
      console.error('Error fetching visits:', error);
      toast.error('Fetch Failed', 'Failed to fetch visit schedules');
    } finally {
      setLoading(false);
    }
  };

  const fetchPendingCount = async () => {
    try {
      const data = await getAdminVisitSchedules(1, 1, 'pending');
      setPendingCount(data.total_count);
    } catch (error) {
      console.error('Error fetching pending count:', error);
    }
  };

  useEffect(() => {
    if (initialTab && ['pending', 'approved', 'rejected'].includes(initialTab)) {
      setActiveTab(initialTab as 'pending' | 'approved' | 'rejected');
    }
  }, [initialTab]);

  useEffect(() => {
    fetchVisits(activeTab);
    fetchPendingCount(); // Always fetch pending count
  }, [activeTab]);

  const handleApprove = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'approve'});
      await approveVisitSchedule(visitId);
      
      setSuccessMessage({ message: 'Visit Approved', type: 'approve' });
      setTimeout(() => setSuccessMessage(null), 3000);
      
      setTimeout(() => {
        setVisits(prev => prev.filter(v => v.id !== visitId));
        setTotalCount(prev => prev - 1);
        setProcessingAction(null);
      }, 500);
    } catch (error) {
      setProcessingAction(null);
      toast.error('Approval Failed', 'Failed to approve visit schedule');
      fetchVisits(activeTab);
    }
  };

  const handleReject = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'reject'});
      await rejectVisitSchedule(visitId);
      
      setSuccessMessage({ message: 'Visit Rejected', type: 'reject' });
      setTimeout(() => setSuccessMessage(null), 3000);
      
      setTimeout(() => {
        setVisits(prev => prev.filter(v => v.id !== visitId));
        setTotalCount(prev => prev - 1);
        setProcessingAction(null);
      }, 500);
    } catch (error) {
      setProcessingAction(null);
      toast.error('Rejection Failed', 'Failed to reject visit schedule');
      fetchVisits(activeTab);
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      case 'admin_approved': return 'bg-blue-100 text-blue-800';
      case 'seller_approved':
      case 'approved': return 'bg-green-100 text-green-800';
      case 'admin_rejected':
      case 'seller_rejected':
      case 'rejected': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <div className="space-y-6 font-['Inter',system-ui,sans-serif] antialiased">
      {/* Success Message */}
      {successMessage && (
        <div className={`fixed top-4 right-4 z-[9999] flex items-center gap-3 px-6 py-4 rounded-lg shadow-lg transform transition-all duration-500 ${
          successMessage.type === 'approve' ? 'bg-green-500 text-white' : 'bg-red5600 text-white'
        }`}>
          <div className="flex items-center gap-2">
            {successMessage.type === 'approve' ? (
              <Check className="w-5 h-5" />
            ) : (
              <X className="w-5 h-5 text-green" />
            )}
            <span className="font-medium">{successMessage.message}</span>
          </div>
          <button
            onClick={() => setSuccessMessage(null)}
            className="ml-2 p-1 hover:bg-white/20 rounded-full transition-colors"
          >
            <X className="w-4 h-4" />
          </button>
        </div>
      )}
      
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Visit Schedules</h1>
      </div>

      {/* Enhanced Tabs */}
      <div className="bg-white rounded-xl border border-gray-200 p-1 shadow-sm">
        <div className="grid w-full grid-cols-3 gap-1">
          {[
            { key: 'approved', label: 'Approved Visits' },
            { key: 'pending', label: 'Pending Approval' },
            { key: 'rejected', label: 'Rejected Visits' }
          ].map((tab) => {
            const isActive = activeTab === tab.key;
            const showBadge = tab.key === 'pending' && pendingCount > 0;
            const badgeCount = tab.key === 'pending' ? pendingCount : 0;
            
            return (
              <button
                key={tab.key}
                onClick={() => setActiveTab(tab.key as any)}
                className={`flex items-center justify-center gap-3 px-6 py-4 rounded-lg text-sm font-semibold transition-all duration-200 relative ${
                  isActive 
                    ? 'bg-blue-500 text-white shadow-lg' 
                    : 'text-gray-600 bg-transparent hover:bg-gray-50'
                }`}
              >
                <span>{tab.label}</span>
                {showBadge && (
                  <div className="ml-2 px-2 py-1 text-xs rounded-full font-bold bg-orange-500 text-white min-w-[20px] flex items-center justify-center">
                    {badgeCount > 99 ? '99+' : badgeCount}
                  </div>
                )}
              </button>
            );
          })}
        </div>
      </div>

      {/* Content */}
      {loading ? (
        <div className="flex justify-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      ) : visits.length === 0 ? (
        <div className="text-center py-12">
          <Calendar className="mx-auto h-12 w-12 text-gray-400" />
          <h3 className="mt-2 text-sm font-medium text-gray-900">No visit schedules</h3>
          <p className="mt-1 text-sm text-gray-500">
            No {activeTab} visit schedules found.
          </p>
        </div>
      ) : (
        <div className="bg-white rounded-xl shadow-sm border border-gray-200">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50 border-b">
                <tr>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Name</th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Contact</th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Visit Schedule</th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Status</th>
                  <th className="px-6 py-4 text-left text-xs font-medium text-gray-500 uppercase tracking-wide">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {visits.map((visit) => (
                  <tr key={visit.id} className="hover:bg-gray-50 transition-colors">
                    <td className="px-6 py-4">
                      <div className="text-sm font-semibold text-gray-900">
                        {visit.full_name.split(' ')[0]}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900">
                        {visit.mobile}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="text-sm text-gray-900">
                        {formatDate(visit.preferred_date)}
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <span className={`text-xs font-semibold ${
                        visit.status === 'admin_approved' || visit.status === 'seller_approved' || visit.status === 'approved' ? 'text-green-700' : 
                        visit.status === 'admin_rejected' || visit.status === 'seller_rejected' || visit.status === 'rejected' ? 'text-red-700' : 
                        'text-yellow-700'
                      }`}>
                        {visit.status.charAt(0).toUpperCase() + visit.status.slice(1)}
                      </span>
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex items-center space-x-2">
                        <button
                          onClick={() => setSelectedVisit(visit)}
                          className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-green-500 rounded-md hover:bg-green-600 transition-all duration-200"
                        >
                          View
                        </button>
                        {activeTab === 'pending' && (
                          <>
                            <button
                              onClick={() => handleApprove(visit.id)}
                              disabled={processingAction?.visitId === visit.id && processingAction?.action === 'approve'}
                              className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-blue-500 rounded-md hover:bg-blue-600 disabled:bg-blue-400 transition-all duration-200"
                            >
                              {processingAction?.visitId === visit.id && processingAction?.action === 'approve' ? 'Approving...' : 'Approve'}
                            </button>
                            <button
                              onClick={() => handleReject(visit.id)}
                              disabled={processingAction?.visitId === visit.id && processingAction?.action === 'reject'}
                              className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-red-600 rounded-md hover:bg-red-700 disabled:bg-red-400 transition-all duration-200"
                            >
                              {processingAction?.visitId === visit.id && processingAction?.action === 'reject' ? 'Rejecting...' : 'Reject'}
                            </button>
                          </>
                        )}
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {/* Visit Details Modal */}
      {selectedVisit && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center p-4 z-50">
          <div 
            className="bg-white rounded-2xl w-full max-w-3xl flex flex-col border border-gray-200 overflow-hidden font-inter"
            style={{
              boxShadow: '0 25px 50px -12px rgba(0, 0, 0, 0.25), 0 0 0 1px rgba(0, 0, 0, 0.05), 0 10px 15px -3px rgba(0, 0, 0, 0.1)',
              maxHeight: '90vh'
            }}
          >
            {/* Header */}
            <div className="px-6 py-4 border-b border-gray-100">
              <div className="flex items-center justify-between">
                <div>
                  <h2 className="text-xl font-bold text-gray-900 tracking-tight">Visit Request Details</h2>
                  <p className="text-gray-600 text-sm mt-1 font-medium">Request ID: #{selectedVisit.id}</p>
                </div>
                <button
                  onClick={() => setSelectedVisit(null)}
                  className="p-2 hover:text-gray-700 transition-colors"
                >
                  <X className="w-5 h-5 text-gray-500" />
                </button>
              </div>
            </div>
            
            {/* Content */}
            <div 
              className="flex-1" 
              style={{ 
                minHeight: 0,
                overflowY: 'auto',
                scrollbarWidth: 'thin',
                scrollbarColor: '#cbd5e1 #f1f5f9'
              }}
            >
              <div className="p-6 space-y-6">
                <div className="bg-blue-50 p-4 rounded-lg border border-blue-100">
                  <h3 className="text-lg font-semibold text-gray-900 tracking-tight">{selectedVisit.property_title}</h3>
                  <p className="text-sm text-gray-600 mt-1 font-medium">{selectedVisit.property_city}</p>
                </div>

                <div className="grid grid-cols-2 gap-6">
                  <div>
                    <h4 className="text-base font-semibold text-gray-900 mb-3 tracking-tight">Contact Information</h4>
                    <div className="space-y-3">
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Full Name</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.full_name}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Phone Number</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.mobile}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Email Address</label>
                        <p className="text-sm font-semibold text-gray-900 break-words">{selectedVisit.email}</p>
                      </div>
                    </div>
                  </div>

                  <div>
                    <h4 className="text-base font-semibold text-gray-900 mb-3 tracking-tight">Visit Details</h4>
                    <div className="space-y-3">
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Preferred Date</label>
                        <p className="text-sm font-semibold text-gray-900">{formatDate(selectedVisit.preferred_date)}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Preferred Time</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.preferred_time || 'Flexible timing'}</p>
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-500 mb-1">Visit Mode</label>
                        <p className="text-sm font-semibold text-gray-900">{selectedVisit.visit_mode}</p>
                      </div>
                      {selectedVisit.video_call_enabled && (
                        <div>
                          <label className="block text-sm font-medium text-gray-500 mb-1">Video Call</label>
                          <div className="flex items-center gap-2">
                            <span className="text-sm font-semibold text-green-600">Enabled</span>
                            {selectedVisit.video_call_room_id && selectedVisit.status === 'seller_approved' && (
                              <a
                                href={`https://meet.jit.si/${selectedVisit.video_call_room_id}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="inline-flex items-center px-2 py-1 text-xs font-medium text-white bg-green-500 rounded hover:bg-green-600"
                              >
                                Join Call
                              </a>
                            )}
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                <div>
                  <h4 className="text-base font-semibold text-gray-900 mb-3 tracking-tight">Additional Information</h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-500 mb-1">Buyer Type</label>
                      <p className="text-sm font-semibold text-gray-900">{selectedVisit.buyer_type}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-500 mb-1">Number of Visitors</label>
                      <p className="text-sm font-semibold text-gray-900">{selectedVisit.number_of_visitors} person(s)</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-500 mb-1">Visit Purpose</label>
                      <p className="text-sm font-semibold text-gray-900">{selectedVisit.visit_reason}</p>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-500 mb-1">Preferred Contact</label>
                      <p className="text-sm font-semibold text-gray-900">{selectedVisit.preferred_contact}</p>
                    </div>
                  </div>
                </div>

                {selectedVisit.comments && (
                  <div>
                    <h4 className="text-base font-semibold text-gray-900 mb-2 tracking-tight">Special Requests</h4>
                    <p className="text-sm text-gray-700 leading-relaxed">{selectedVisit.comments}</p>
                  </div>
                )}

                <div className="flex justify-between items-center pt-4 border-t border-gray-200">
                  <span className="text-sm font-medium text-gray-500">Submitted on {formatDate(selectedVisit.created_at)}</span>
                  <span className={`px-3 py-1.5 text-xs font-semibold rounded-lg ${
                    selectedVisit.status === 'admin_approved' || selectedVisit.status === 'seller_approved' || selectedVisit.status === 'approved' ? 'bg-green-100 text-green-800' :
                    selectedVisit.status === 'admin_rejected' || selectedVisit.status === 'seller_rejected' || selectedVisit.status === 'rejected' ? 'bg-red-100 text-red-800' :
                    'bg-yellow-100 text-yellow-800'
                  }`}>
                    {selectedVisit.status.toUpperCase()}
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}
      

    </div>
  );
}