import React, { useState, useEffect } from "react";

declare global {
  interface Window {
    google: any;
  }
}
import { Plus, Construction } from "lucide-react";
import { Button } from "../../ui/button";
import { Input } from "../../ui/input";
import { Label } from "../../ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../../ui/select";
import { Textarea } from "../../ui/textarea";
import { DataTable } from "./components/DataTable";
import { SearchFilters } from "./components/SearchFilters";
import { Pagination } from "./components/Pagination";
import { FormModal } from "./components/FormModal";

import { PlacesAutocomplete } from "./components/PlacesAutocomplete";
import { getInfrastructureProjects, createInfrastructureProject, updateInfrastructureProject, deleteInfrastructureProject, InfrastructureProject } from "../../../api/hyperlocalEconomic";
import { toast } from "sonner";

export function InfrastructureProjects() {
  const [projects, setProjects] = useState<InfrastructureProject[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState("");
  const [typeFilter, setTypeFilter] = useState("all");
  const [statusFilter, setStatusFilter] = useState("all");
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalItems, setTotalItems] = useState(0);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingProject, setEditingProject] = useState<InfrastructureProject | null>(null);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isGoogleMapsReady, setIsGoogleMapsReady] = useState(false);

  useEffect(() => {
    const checkGoogleMaps = () => {
      if (window.google?.maps) {
        setIsGoogleMapsReady(true);
      } else {
        setTimeout(checkGoogleMaps, 100);
      }
    };
    checkGoogleMaps();
  }, []);

  const [formData, setFormData] = useState({
    name: "",
    type: "road",
    description: "",
    status: "planned" as 'planned' | 'under_construction' | 'completed' | 'cancelled',
    expected_year: new Date().getFullYear(),
    latitude: 28.6139,
    longitude: 77.2090,
    radius_meters: 1000,
    locationSearch: ""
  });

  const itemsPerPage = 10;

  const columns = [
    { key: "name", label: "Project Name", sortable: true },
    { key: "type", label: "Type", sortable: true },
    { key: "status", label: "Status", sortable: true },
    { key: "expected_year", label: "Expected Year", sortable: true }
  ];

  const typeOptions = [
    { value: "road", label: "Road" },
    { value: "bridge", label: "Bridge" },
    { value: "airport", label: "Airport" },
    { value: "hospital", label: "Hospital" },
    { value: "school", label: "School" },
    { value: "mall", label: "Mall" },
    { value: "other", label: "Other" }
  ];

  const statusOptions = [
    { value: "planned", label: "Planned" },
    { value: "under_construction", label: "Under Construction" },
    { value: "completed", label: "Completed" },
    { value: "cancelled", label: "Cancelled" }
  ];

  const fetchProjects = async () => {
    try {
      setLoading(true);
      const response = await getInfrastructureProjects(currentPage, itemsPerPage, searchQuery, typeFilter, statusFilter);
      setProjects(response.data);
      setTotalPages(Math.ceil(response.total / itemsPerPage));
      setTotalItems(response.total);
    } catch (error) {
      toast.error("Failed to fetch infrastructure projects");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchProjects();
  }, [currentPage, searchQuery, typeFilter, statusFilter]);

  const handleAdd = () => {
    setEditingProject(null);
    setFormData({
      name: "",
      type: "road",
      description: "",
      status: "planned" as 'planned' | 'under_construction' | 'completed' | 'cancelled',
      expected_year: new Date().getFullYear(),
      latitude: 28.6139,
      longitude: 77.2090,
      radius_meters: 1000,
      locationSearch: ""
    });
    setIsModalOpen(true);
  };

  const handleEdit = (project: InfrastructureProject) => {
    setEditingProject(project);
    setFormData({
      name: project.name,
      type: project.type,
      description: project.description || "",
      status: project.status,
      expected_year: project.expected_year || new Date().getFullYear(),
      latitude: project.latitude,
      longitude: project.longitude,
      radius_meters: project.radius_meters,
      locationSearch: ""
    });
    setIsModalOpen(true);
  };

  const handleDelete = async (project: InfrastructureProject) => {
    if (!confirm("Are you sure you want to delete this infrastructure project?")) return;
    
    try {
      await deleteInfrastructureProject(project.id);
      toast.success("Infrastructure project deleted successfully");
      fetchProjects();
    } catch (error) {
      toast.error("Failed to delete infrastructure project");
    }
  };

  const handleSubmit = async () => {
    try {
      setIsSubmitting(true);
      
      const projectData = {
        name: formData.name,
        type: formData.type,
        description: formData.description,
        status: formData.status,
        expected_year: formData.expected_year,
        latitude: formData.latitude,
        longitude: formData.longitude,
        radius_meters: formData.radius_meters
      };

      if (editingProject) {
        await updateInfrastructureProject(editingProject.id, projectData);
        toast.success("Infrastructure project updated successfully");
      } else {
        await createInfrastructureProject(projectData);
        toast.success("Infrastructure project created successfully");
      }

      setIsModalOpen(false);
      fetchProjects();
    } catch (error) {
      toast.error("Failed to save infrastructure project");
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleClearFilters = () => {
    setSearchQuery("");
    setTypeFilter("all");
    setStatusFilter("all");
    setCurrentPage(1);
  };

  const filters = [
    {
      key: "type",
      label: "Type",
      options: typeOptions,
      value: typeFilter,
      onChange: setTypeFilter
    },
    {
      key: "status",
      label: "Status",
      options: statusOptions,
      value: statusFilter,
      onChange: setStatusFilter
    }
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <Construction className="h-8 w-8 text-black" />
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Infrastructure Projects</h1>
            <p className="text-gray-600">Manage infrastructure development projects</p>
          </div>
        </div>
        <Button onClick={handleAdd} className="!bg-blue-600 !hover:bg-blue-700 !text-white border-0" style={{backgroundColor: '#2563eb', color: 'white'}}>
          <Plus className="h-4 w-4 mr-2" />
          Add Infrastructure Project
        </Button>
      </div>

      <SearchFilters
        searchQuery={searchQuery}
        onSearchChange={setSearchQuery}
        filters={filters}
        onClearFilters={handleClearFilters}
        placeholder="Search infrastructure projects..."
      />

      <DataTable
        columns={columns}
        data={projects}
        onEdit={handleEdit}
        onDelete={handleDelete}
        loading={loading}
      />

      {!loading && totalPages > 1 && (
        <Pagination
          currentPage={currentPage}
          totalPages={totalPages}
          onPageChange={setCurrentPage}
          totalItems={totalItems}
          itemsPerPage={itemsPerPage}
        />
      )}

      <FormModal
        isOpen={isModalOpen}
        onClose={() => setIsModalOpen(false)}
        title={editingProject ? "Edit Infrastructure Project" : "Add Infrastructure Project"}
        onSubmit={handleSubmit}
        isSubmitting={isSubmitting}
        size="lg"
      >
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="md:col-span-2">
            <Label htmlFor="locationSearch">Search Location</Label>
            <PlacesAutocomplete
              value={formData.locationSearch}
              onChange={(value) => setFormData({ ...formData, locationSearch: value })}
              placeholder="Search for project location..."
            />
          </div>

          <div className="md:col-span-2">
            <Label htmlFor="name">Project Name *</Label>
            <Input
              id="name"
              value={formData.name}
              onChange={(e) => setFormData({ ...formData, name: e.target.value })}
              placeholder="Enter project name"
              required
            />
          </div>

          <div>
            <Label htmlFor="type">Project Type *</Label>
            <Select value={formData.type} onValueChange={(value: any) => setFormData({ ...formData, type: value })}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {typeOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label htmlFor="status">Status *</Label>
            <Select value={formData.status} onValueChange={(value: any) => setFormData({ ...formData, status: value })}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {statusOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className="md:col-span-2">
            <Label htmlFor="description">Description</Label>
            <Textarea
              id="description"
              value={formData.description}
              onChange={(e) => setFormData({ ...formData, description: e.target.value })}
              placeholder="Enter project description"
              rows={3}
            />
          </div>

          <div>
            <Label htmlFor="expected_year">Expected Year *</Label>
            <Input
              id="expected_year"
              type="number"
              value={formData.expected_year}
              onChange={(e) => setFormData({ ...formData, expected_year: parseInt(e.target.value) || new Date().getFullYear() })}
              placeholder="Enter expected year"
              required
            />
          </div>

          <div>
            <Label htmlFor="radius_meters">Impact Radius (meters) *</Label>
            <Input
              id="radius_meters"
              type="number"
              value={formData.radius_meters}
              onChange={(e) => setFormData({ ...formData, radius_meters: parseInt(e.target.value) || 1000 })}
              placeholder="Enter impact radius"
              required
            />
          </div>

          <div>
            <Label htmlFor="latitude">Latitude *</Label>
            <Input
              id="latitude"
              type="number"
              step="any"
              value={formData.latitude}
              onChange={(e) => setFormData({ ...formData, latitude: parseFloat(e.target.value) || 0 })}
              placeholder="Enter latitude"
              required
            />
          </div>

          <div>
            <Label htmlFor="longitude">Longitude *</Label>
            <Input
              id="longitude"
              type="number"
              step="any"
              value={formData.longitude}
              onChange={(e) => setFormData({ ...formData, longitude: parseFloat(e.target.value) || 0 })}
              placeholder="Enter longitude"
              required
            />
          </div>
        </div>
      </FormModal>
    </div>
  );
}