import React, { useState, useEffect } from "react";

declare global {
  interface Window {
    google: any;
  }
}
import { Plus, Building2 } from "lucide-react";
import { Button } from "../../ui/button";
import { Input } from "../../ui/input";
import { Label } from "../../ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../../ui/select";
import { Textarea } from "../../ui/textarea";
import { DataTable } from "./components/DataTable";
import { SearchFilters } from "./components/SearchFilters";
import { Pagination } from "./components/Pagination";
import { FormModal } from "./components/FormModal";

import { PlacesAutocomplete } from "./components/PlacesAutocomplete";
import {
  getMetroProjects,
  createMetroProject,
  updateMetroProject,
  deleteMetroProject,
  MetroProject,
} from "../../../api/hyperlocalEconomic";
import { toast } from "sonner";

export function MetroProjects() {
  const [projects, setProjects] = useState<MetroProject[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchQuery, setSearchQuery] = useState("");
  const [statusFilter, setStatusFilter] = useState("all");
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalItems, setTotalItems] = useState(0);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingProject, setEditingProject] = useState<MetroProject | null>(
    null
  );
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [formErrors, setFormErrors] = useState<Record<string, string>>({});

  const [formData, setFormData] = useState({
    name: "",
    description: "",
    status: "planned" as
      | "planned"
      | "under_construction"
      | "completed"
      | "cancelled",
    expectedYear: "",
    coordinates: { lat: 28.6139, lng: 77.209 },
    radiusMeters: "",
    stationSearch: "",
  });

  const itemsPerPage = 10;

  const columns = [
    { key: "name", label: "Project Name", sortable: true },
    { key: "type", label: "Type", sortable: true },
    { key: "status", label: "Status", sortable: true },
    { key: "expected_year", label: "Expected Year", sortable: true },
  ];

  const statusOptions = [
    { value: "planned", label: "Planned" },
    { value: "under_construction", label: "Under Construction" },
    { value: "completed", label: "Completed" },
    { value: "delayed", label: "Delayed" },
  ];

  const fetchProjects = async () => {
    try {
      setLoading(true);
      const response = await getMetroProjects(
        currentPage,
        itemsPerPage,
        searchQuery,
        statusFilter
      );
      setProjects(response.data);
      setTotalPages(Math.ceil(response.total / itemsPerPage));
      setTotalItems(response.total);
    } catch (error) {
      toast.error("Failed to fetch metro projects");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchProjects();
  }, [currentPage, searchQuery, statusFilter]);

  const handleAdd = () => {
    setEditingProject(null);
    setFormData({
      name: "",
      description: "",
      status: "planned",
      expectedYear: "",
      coordinates: { lat: 28.6139, lng: 77.209 },
      radiusMeters: "",
      stationSearch: "",
    });
    setIsModalOpen(true);
  };

  const handleEdit = (project: MetroProject) => {
    setEditingProject(project);
    setFormData({
      name: project.name,
      description: project.description || "",
      status: project.status,
      expectedYear: project.expected_year?.toString() || "",
      coordinates: {
        lat: project.latitude || 28.6139,
        lng: project.longitude || 77.209,
      },
      radiusMeters: project.radius_meters.toString(),
      stationSearch: "",
    });
    setIsModalOpen(true);
  };

  const handleDelete = async (project: MetroProject) => {
    if (!confirm("Are you sure you want to delete this metro project?")) return;

    try {
      await deleteMetroProject(project.id);
      toast.success("Metro project deleted successfully");
      fetchProjects();
    } catch (error) {
      toast.error("Failed to delete metro project");
    }
  };

  const validateForm = () => {
    const errors: Record<string, string> = {};

    if (!formData.name.trim()) errors.name = "Project name is required";
    if (!formData.expectedYear)
      errors.expectedYear = "Expected year is required";
    if (!formData.radiusMeters || parseInt(formData.radiusMeters) <= 0)
      errors.radiusMeters = "Valid radius is required";
    if (!formData.coordinates)
      errors.coordinates = "Project location is required";

    setFormErrors(errors);
    return Object.keys(errors).length === 0;
  };

  const handleSubmit = async () => {
    if (!validateForm()) {
      toast.error("Please fix the form errors before submitting");
      return;
    }

    try {
      setIsSubmitting(true);

      const projectData = {
        name: formData.name,
        type: "metro",
        description: formData.description,
        status: formData.status,
        expected_year: parseInt(formData.expectedYear) || null,
        latitude: formData.coordinates.lat,
        longitude: formData.coordinates.lng,
        radius_meters: parseInt(formData.radiusMeters),
      };

      if (editingProject) {
        await updateMetroProject(editingProject.id, projectData);
        toast.success("Metro project updated successfully");
      } else {
        await createMetroProject(projectData);
        toast.success("Metro project created successfully");
      }

      setIsModalOpen(false);
      setFormErrors({});
      fetchProjects();
    } catch (error) {
      toast.error("Failed to save metro project");
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleClearFilters = () => {
    setSearchQuery("");
    setStatusFilter("all");
    setCurrentPage(1);
  };

  const filters = [
    {
      key: "status",
      label: "Status",
      options: statusOptions,
      value: statusFilter,
      onChange: setStatusFilter,
    },
  ];

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <Building2 className="h-8 w-8 text-blue-600" />
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Metro Projects</h1>
            <p className="text-gray-600">
              Manage metro infrastructure projects and their impact
            </p>
          </div>
        </div>
        <Button
          onClick={handleAdd}
          className="!bg-blue-600 !hover:bg-blue-700 !text-white border-0"
          style={{ backgroundColor: "#2563eb", color: "white" }}
        >
          <Plus className="h-4 w-4 mr-2" />
          Add Metro Project
        </Button>
      </div>

      <SearchFilters
        searchQuery={searchQuery}
        onSearchChange={setSearchQuery}
        filters={filters}
        onClearFilters={handleClearFilters}
        placeholder="Search metro projects..."
      />

      <DataTable
        columns={columns}
        data={projects}
        onEdit={handleEdit}
        onDelete={handleDelete}
        loading={loading}
      />

      {!loading && totalPages > 1 && (
        <Pagination
          currentPage={currentPage}
          totalPages={totalPages}
          onPageChange={setCurrentPage}
          totalItems={totalItems}
          itemsPerPage={itemsPerPage}
        />
      )}

      <FormModal
        isOpen={isModalOpen}
        onClose={() => setIsModalOpen(false)}
        title={editingProject ? "Edit Metro Project" : "Add Metro Project"}
        onSubmit={handleSubmit}
        isSubmitting={isSubmitting}
        size="lg"
      >
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="md:col-span-2">
            <Label htmlFor="name">Project Name *</Label>
            <Input
              id="name"
              value={formData.name}
              onChange={(e) =>
                setFormData({ ...formData, name: e.target.value })
              }
              placeholder="Enter project name"
              className={formErrors.name ? "border-red-500" : ""}
              required
            />
            {formErrors.name && (
              <p className="text-sm text-red-500 mt-1">{formErrors.name}</p>
            )}
          </div>

          <div>
            <Label htmlFor="status">Status *</Label>
            <Select
              value={formData.status}
              onValueChange={(value: any) =>
                setFormData({ ...formData, status: value })
              }
            >
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {statusOptions.map((option) => (
                  <SelectItem key={option.value} value={option.value}>
                    {option.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label htmlFor="expectedYear">Expected Year *</Label>
            <Input
              id="expectedYear"
              type="number"
              value={formData.expectedYear}
              onChange={(e) =>
                setFormData({ ...formData, expectedYear: e.target.value })
              }
              placeholder="Enter expected year"
              className={formErrors.expectedYear ? "border-red-500" : ""}
              required
            />
            {formErrors.expectedYear && (
              <p className="text-sm text-red-500 mt-1">
                {formErrors.expectedYear}
              </p>
            )}
          </div>

          <div>
            <Label htmlFor="radiusMeters">Radius (meters) *</Label>
            <Input
              id="radiusMeters"
              type="number"
              value={formData.radiusMeters}
              onChange={(e) =>
                setFormData({ ...formData, radiusMeters: e.target.value })
              }
              placeholder="Enter radius"
              className={formErrors.radiusMeters ? "border-red-500" : ""}
              required
            />
            {formErrors.radiusMeters && (
              <p className="text-sm text-red-500 mt-1">
                {formErrors.radiusMeters}
              </p>
            )}
          </div>

          <div className="md:col-span-2">
            <Label htmlFor="description">Description</Label>
            <Textarea
              id="description"
              value={formData.description}
              onChange={(e) =>
                setFormData({ ...formData, description: e.target.value })
              }
              placeholder="Enter project description"
              rows={3}
            />
          </div>

          <div>
            <Label htmlFor="latitude">Latitude *</Label>
            <Input
              id="latitude"
              type="number"
              step="any"
              value={formData.coordinates.lat}
              onChange={(e) =>
                setFormData({
                  ...formData,
                  coordinates: {
                    ...formData.coordinates,
                    lat: parseFloat(e.target.value) || 0,
                  },
                })
              }
              placeholder="Enter latitude"
              className={formErrors.coordinates ? "border-red-500" : ""}
              required
            />
            {formErrors.coordinates && (
              <p className="text-sm text-red-500 mt-1">
                {formErrors.coordinates}
              </p>
            )}
          </div>

          <div>
            <Label htmlFor="longitude">Longitude *</Label>
            <Input
              id="longitude"
              type="number"
              step="any"
              value={formData.coordinates.lng}
              onChange={(e) =>
                setFormData({
                  ...formData,
                  coordinates: {
                    ...formData.coordinates,
                    lng: parseFloat(e.target.value) || 0,
                  },
                })
              }
              placeholder="Enter longitude"
              className={formErrors.coordinates ? "border-red-500" : ""}
              required
            />
          </div>
        </div>
      </FormModal>
    </div>
  );
}
